package com.zoyi.channel.plugin.android.activity.userchat_list;

import android.content.Intent;
import android.graphics.Color;
import android.os.Bundle;
import android.support.annotation.AnimRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.view.View;
import com.zoyi.channel.plugin.android.ChannelIO;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.BaseActivity;
import com.zoyi.channel.plugin.android.activity.chat.ChatActivity2;
import com.zoyi.channel.plugin.android.activity.settings.SettingsActivity;
import com.zoyi.channel.plugin.android.activity.userchat_list.contract.UserChatListContract;
import com.zoyi.channel.plugin.android.activity.userchat_list.listener.OnUserChatClickListener;
import com.zoyi.channel.plugin.android.activity.userchat_list.model.ChatListItem;
import com.zoyi.channel.plugin.android.activity.userchat_list.model.PushBotChatItem;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.enumerate.StoreType;
import com.zoyi.channel.plugin.android.enumerate.Transition;
import com.zoyi.channel.plugin.android.enumerate.UpdateType;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.PushBotItem;
import com.zoyi.channel.plugin.android.model.entity.Entity;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.Plugin;
import com.zoyi.channel.plugin.android.push.ChannelPushManager;
import com.zoyi.channel.plugin.android.selector.ChannelSelector;
import com.zoyi.channel.plugin.android.selector.ColorSelector;
import com.zoyi.channel.plugin.android.selector.PluginSelector;
import com.zoyi.channel.plugin.android.selector.PushBotSelector;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.channel.plugin.android.view.external.fab.FloatingActionButton;
import com.zoyi.channel.plugin.android.view.handler.AlertDialogListener;
import com.zoyi.channel.plugin.android.view.handler.InfiniteScrollListener;
import com.zoyi.channel.plugin.android.view.layout.BigBar.MenuPosition;
import com.zoyi.channel.plugin.android.view.layout.BottomFloatingActionLayout;
import com.zoyi.channel.plugin.android.view.layout.MenuButton.MenuState;

import static com.zoyi.channel.plugin.android.view.layout.MenuButton.ActionType;

/**
 * Created by mika on 2016. 12. 5..
 */
public class UserChatListActivity extends BaseActivity implements View.OnClickListener, UserChatListContract.View, OnUserChatClickListener {

  private View rootView;
  private View contentLayout;
  private View emptyLayout;
  private View reconnectLayout;

  private RecyclerView chatList;
  private BottomFloatingActionLayout bottomFloatingActionLayout;
  private FloatingActionButton fab;
  private LinearLayoutManager layoutManager;

  private UserChatListAdapter adapter;
  private UserChatListContract.Presenter presenter;

  @Nullable
  private String redirectChatId;
  @Nullable
  private String redirectPushBotId;
  protected boolean animateOnInit = true;

  @Override
  protected void onCreate(@Nullable Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);
    init(R.layout.ch_plugin_activity_user_chat_list);

    // get redirecting chat id, it will be turn on after user chat list is loaded

    if (getIntent() != null) {
      redirectChatId = getIntent().getStringExtra(Const.EXTRA_CHAT_ID);
      redirectPushBotId = getIntent().getStringExtra(Const.EXTRA_PUSH_BOT_ID);
      animateOnInit = getIntent().getBooleanExtra(Const.EXTRA_REDIRECT_ANIMATED, true);
    }

    // view initialize

    rootView = findViewById(R.id.root_view);
    contentLayout = findViewById(R.id.layout_user_chat_list_content);
    emptyLayout = findViewById(R.id.layout_empty);
    reconnectLayout = findViewById(R.id.layout_reconnect);
    fab = findViewById(R.id.fab);
    chatList = findViewById(R.id.recycler_view);
    bottomFloatingActionLayout = findViewById(R.id.ch_layout_chat_list_bottom_frame);

    // floating action view

    fab.setOnClickListener(this);

    // big bar setting

    if (bigBar != null) {
      bigBar.withActivity(this)
          .addMenu(ActionType.SETTINGS, MenuPosition.LEFT)
          .addMenu(ActionType.REFRESH, MenuPosition.RIGHT, MenuState.HIDDEN)
          .addMenu(ActionType.EXIT, MenuPosition.RIGHT)
          .build();
    }

    // initialize components

    adapter = new UserChatListAdapter();
    adapter.setOnUserChatClickListener(this);

    presenter = new UserChatListPresenter(this);
    presenter.setView(this);
    presenter.setAdapterView(adapter);
    presenter.setAdapterModel(adapter);

    layoutManager = new LinearLayoutManager(this);
    chatList.setLayoutManager(layoutManager);
    chatList.setAdapter(adapter);
    chatList.setItemAnimator(null);
    chatList.getRecycledViewPool().setMaxRecycledViews(0, 40);
    chatList.addOnScrollListener(new InfiniteScrollListener(layoutManager, InfiniteScrollListener.TOP, chatList) {
      @Override
      public void scrolledInList() {
        bottomFloatingActionLayout.hide();
      }

      @Override
      public void scrollAttachedToBottom() {
        bottomFloatingActionLayout.show();
      }

      @Override
      public void refresh() {
      }
    });

    bottomFloatingActionLayout.setRecyclerView(chatList);

    // Set view style
    setStyle();

    // Socket error handling
    if (SocketManager.isError()) {
      setErrorView(true);
    }

    presenter.refresh();
  }

  private void setStyle() {
    Channel channel = ChannelSelector.getCurrentChannel();
    Plugin plugin = PluginSelector.getPlugin();

    if (channel == null || plugin == null) {
      finish();
    } else {
      if (bigBar != null) {
        bigBar.setTitle(channel.getName());
      }

      setFabTheme();
      fab.attachToRecyclerView(chatList);
    }
  }

  @Override
  protected void onActivityResult(int requestCode, int resultCode, Intent data) {
    super.onActivityResult(requestCode, resultCode, data);

    switch (requestCode) {
      case Const.REQUEST_PLUGIIN_SETTINGS:
        if (resultCode == Const.RESULT_CHAT_OPTION_CHANGED) {
          presenter.refresh();
        }
        presenter.redrawList();
        break;

      case Const.REQUEST_CHAT:
        switch (resultCode) {
          case Const.RESULT_CHAT_OPEN_NEW_CHAT:
            startChat(null, false, true, true);
            break;

          case Const.RESULT_CHAT_OPTION_CHANGED:
            presenter.refresh();
            break;

          case Const.RESULT_CHAT_GO_TO_MAIN:
            finish();
            break;
        }
        break;
    }
  }

  @Override
  protected void onStart() {
    ChannelPushManager.sendTokenToChannelIO(ChannelIO.getAppContext());
    onUserChatsUpdate(adapter.getItemCount(), false);
    super.onStart();
  }

  private void setFabTheme() {
    fab.setColorNormal(ColorSelector.getBackgroundColor());
    fab.setColorPressed(Utils.addBlackMask(ColorSelector.getBackgroundColor()));
    fab.setColorRipple(Color.WHITE);
    fab.setImageResource(
        ColorSelector.getTextColor() == Color.BLACK
            ? R.drawable.ch_plugin_ic_add_black_24dp
            : R.drawable.ch_plugin_ic_add_white_24dp);
  }

  @Override
  public void onClick(View view) {
    int id = view.getId();

    if (id == R.id.fab) {
      startChat(null, false, true, false);
    }
  }

  @Override
  public void optionClicked(ActionType actionType) {
    switch (actionType) {
      case REFRESH:
        if (bigBar != null) {
          bigBar.setMenuState(ActionType.REFRESH, MenuState.HIDDEN);
        }
        presenter.refresh();
        break;

      case SETTINGS:
        IntentUtils.setNextActivity(this, SettingsActivity.class)
            .setTransition(Transition.SLIDE_FROM_BOTTOM)
            .startActivityForResult(Const.REQUEST_PLUGIIN_SETTINGS);
        break;
    }
    super.optionClicked(actionType);
  }

  // UserChat list fetch 가 끝나고 바로 채팅방으로 이동해야할 경우 조건에 맞는 채팅방으로 이동
  @Override
  public void userChatsFetched(boolean init) {
    onUserChatsUpdate(adapter.getItemCount(), init);

    if (redirectChatId != null) { // 바로 이동해야할 유저챗 채팅방이 있을 때
      ChatListItem chatListItem = presenter.getUserChatItem(redirectChatId);

      if (chatListItem != null) {
        startRedirectChat(chatListItem);
      }
    } else if (redirectPushBotId != null) { // 바로 이동해야할 푸시봇 채팅방이 있을 때
      PushBotItem pushBotItem = PushBotSelector.get(redirectPushBotId);

      if (pushBotItem != null) {
        startRedirectChat(new PushBotChatItem(pushBotItem));
      }
    } else {
      if (adapter.getItemCount() <= 1 && init) { // 유저챗 개수가 0 ~ 1 개 일 경우 바로 채팅방으로 이동
        if (adapter.getItemCount() == 0) {
          startRedirectChat(null);
        } else {
          startRedirectChat(adapter.getFirstChatListItem());
        }
      }
    }

    redirectChatId = null;
    redirectPushBotId = null;
  }

  private void startRedirectChat(@Nullable ChatListItem chatListItem) {
    startChat(chatListItem, true, false, false);
  }

  private void startChat(@Nullable ChatListItem chatListItem, boolean redirect, boolean animate, boolean force) {
    Transition transition;

    if (!animate) {
      // set none when animate is false
      transition = Transition.NONE;
    } else if (redirect) {
      // when launch with redirect chat id, always show from bottom
      transition = Transition.SLIDE_FROM_BOTTOM;
    } else {
      // else transition is normal
      transition = Transition.SLIDE_FROM_RIGHT;
    }

    if (force || isActivityFocused) {
      presenter.fetchPreloadData(chatListItem, transition);
    }
  }

  @Override
  public void setErrorView(boolean visible) {
    if (bigBar != null) {
      bigBar.setMenuState(ActionType.REFRESH, visible ? MenuState.VISIBLE : MenuState.HIDDEN);
    }
    Views.setVisibility(reconnectLayout, visible);
  }

  private void showDeleteUserChatDialog(@NonNull final ChatListItem chatListItem) {
    UIUtils.showAlertDialog(
        this,
        ResUtils.getString(this, "ch.chat.delete_description"),
        true,
        new AlertDialogListener() {
          @Override
          public void onPositive() {
            switch (chatListItem.getItemType()) {
              case USER_CHAT:
                presenter.removeUserChat(chatListItem.getId());
                break;

              case PUSH_BOT_CHAT:
                presenter.removePushBotChat(chatListItem.getId());
                break;
            }
          }
        });
  }

  @Override
  public void completeRemoveChat() {
    hideProgress();
    onUserChatsUpdate(adapter.getItemCount(), false);
  }

  @Override
  public void successFetchPreloadData(@Nullable ChatListItem chatListItem, Transition transition) {
    String chatId = null;
    String pushBotId = null;

    if (chatListItem != null) {
      switch (chatListItem.getItemType()) {
        case USER_CHAT:
          chatId = chatListItem.getId();
          break;

        case PUSH_BOT_CHAT:
          pushBotId = chatListItem.getId();
          break;
      }
    }
    IntentUtils.setNextActivity(this, ChatActivity2.class)
        .putExtra(Const.EXTRA_CHAT_ID, chatId)
        .putExtra(Const.EXTRA_PUSH_BOT_ID, pushBotId)
        .setTransition(transition)
        .startActivityForResult(Const.REQUEST_CHAT);
  }

  @Override
  public void receiveCommand(Command command, @Nullable Object object) {
    presenter.receiveCommand(command, object);
  }

  @Override
  public void receiveData(Entity entity, boolean upsert) {
    presenter.receiveData(entity, upsert);

    onUserChatsUpdate(adapter.getItemCount(), false);
  }

  @Override
  public void receiveStoreEvent(StoreType storeType, UpdateType updateType, @Nullable Entity entity) {
    if (StoreType.PLUGIN == storeType) {
      setFabTheme();
    }
    presenter.receiveStoreEvent(storeType, updateType, entity);
  }

  private void onUserChatsUpdate(int userChatCount, boolean hideContentLayout) {
    if (userChatCount < 2 && hideContentLayout) {
      contentLayout.setVisibility(View.GONE);
    } else {
      contentLayout.setVisibility(View.VISIBLE);
      emptyLayout.setVisibility(UIUtils.getVisible(userChatCount <= 0, true));
      chatList.setVisibility(UIUtils.getVisible(userChatCount > 0, true));
    }
  }

  @Override
  @AnimRes
  protected int getExistAnimOfFinish() {
    return R.anim.ch_plugin_slide_out_bottom;
  }

  @Override
  public void onUserChatClick(@NonNull ChatListItem chatListItem) {
    startChat(chatListItem, false, true, false);
  }

  @Override
  public void onUserChatLongClick(@NonNull ChatListItem chatListItem) {
    showDeleteUserChatDialog(chatListItem);
  }
}
