package com.zoyi.channel.plugin.android.util.lang;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

public class StringUtils {

  public static final int INDEX_NOT_FOUND = -1;
  public static final String EMPTY = "";


  /**
   * <p>Strips any of a set of characters from the end of a String.</p>
   *
   * <p>A {@code null} input String returns {@code null}.
   * An empty string ("") input returns the empty string.</p>
   *
   * <p>If the stripChars String is {@code null}, whitespace is
   * stripped as defined by {@link Character#isWhitespace(char)}.</p>
   *
   * <pre>
   * StringUtils.stripEnd(null, *)          = null
   * StringUtils.stripEnd("", *)            = ""
   * StringUtils.stripEnd("abc", "")        = "abc"
   * StringUtils.stripEnd("abc", null)      = "abc"
   * StringUtils.stripEnd("  abc", null)    = "  abc"
   * StringUtils.stripEnd("abc  ", null)    = "abc"
   * StringUtils.stripEnd(" abc ", null)    = " abc"
   * StringUtils.stripEnd("  abcyx", "xyz") = "  abc"
   * StringUtils.stripEnd("120.00", ".0")   = "12"
   * </pre>
   *
   * @param str  the String to remove characters from, may be null
   * @param stripChars  the set of characters to remove, null treated as whitespace
   * @return the stripped String, {@code null} if null String input
   */
  public static String stripEnd(final String str, final String stripChars) {
    int end;
    if (str == null || (end = str.length()) == 0) {
      return str;
    }

    if (stripChars == null) {
      while (end != 0 && Character.isWhitespace(str.charAt(end - 1))) {
        end--;
      }
    } else if (stripChars.isEmpty()) {
      return str;
    } else {
      while (end != 0 && stripChars.indexOf(str.charAt(end - 1)) != INDEX_NOT_FOUND) {
        end--;
      }
    }
    return str.substring(0, end);
  }

  /**
   * <p>Checks if CharSequence contains a search CharSequence irrespective of case,
   * handling {@code null}. Case-insensitivity is defined as by
   * {@link String#equalsIgnoreCase(String)}.
   *
   * <p>A {@code null} CharSequence will return {@code false}.</p>
   *
   * <pre>
   * StringUtils.containsIgnoreCase(null, *) = false
   * StringUtils.containsIgnoreCase(*, null) = false
   * StringUtils.containsIgnoreCase("", "") = true
   * StringUtils.containsIgnoreCase("abc", "") = true
   * StringUtils.containsIgnoreCase("abc", "a") = true
   * StringUtils.containsIgnoreCase("abc", "z") = false
   * StringUtils.containsIgnoreCase("abc", "A") = true
   * StringUtils.containsIgnoreCase("abc", "Z") = false
   * </pre>
   *
   * @param str  the CharSequence to check, may be null
   * @param searchStr  the CharSequence to find, may be null
   * @return true if the CharSequence contains the search CharSequence irrespective of
   * case or false if not or {@code null} string input
   * @since 3.0 Changed signature from containsIgnoreCase(String, String) to containsIgnoreCase(CharSequence, CharSequence)
   */
  public static boolean containsIgnoreCase(final CharSequence str, final CharSequence searchStr) {
    if (str == null || searchStr == null) {
      return false;
    }
    final int len = searchStr.length();
    final int max = str.length() - len;
    for (int i = 0; i <= max; i++) {
      if (CharSequenceUtils.regionMatches(str, true, i, searchStr, 0, len)) {
        return true;
      }
    }
    return false;
  }

  /**
   * <p>Joins the elements of the provided {@code Iterator} into
   * a single String containing the provided elements.</p>
   *
   * <p>No delimiter is added before or after the list.
   * A {@code null} separator is the same as an empty String ("").</p>
   *
   * <p>See the examples here: {@link #join(Object[], String)}. </p>
   *
   * @param iterator  the {@code Iterator} of values to join together, may be null
   * @param separator  the separator character to use, null treated as ""
   * @return the joined String, {@code null} if null iterator input
   */
  public static String join(final Iterator<?> iterator, final String separator) {
    // handle null, zero and one elements before building a buffer
    if (iterator == null) {
      return null;
    }
    if (!iterator.hasNext()) {
      return EMPTY;
    }
    final Object first = iterator.next();
    if (!iterator.hasNext()) {
      String result = EMPTY;

      if (first != null) {
        result = first.toString();
      }
      return result;
    }

    // two or more elements
    final StringBuilder buf = new StringBuilder(256); // Java default is 16, probably too small
    if (first != null) {
      buf.append(first);
    }

    while (iterator.hasNext()) {
      if (separator != null) {
        buf.append(separator);
      }
      final Object obj = iterator.next();
      if (obj != null) {
        buf.append(obj);
      }
    }
    return buf.toString();
  }

  /**
   * <p>Joins the elements of the provided {@code Iterable} into
   * a single String containing the provided elements.</p>
   *
   * <p>No delimiter is added before or after the list.
   * A {@code null} separator is the same as an empty String ("").</p>
   *
   * <p>See the examples here: {@link #join(Object[], String)}. </p>
   *
   * @param iterable  the {@code Iterable} providing the values to join together, may be null
   * @param separator  the separator character to use, null treated as ""
   * @return the joined String, {@code null} if null iterator input
   * @since 2.3
   */
  public static String join(final Iterable<?> iterable, final String separator) {
    if (iterable == null) {
      return null;
    }
    return join(iterable.iterator(), separator);
  }

  /**
   * Performs the logic for the {@code split} and
   * {@code splitPreserveAllTokens} methods that do not return a
   * maximum array length.
   *
   * @param str  the String to parse, may be {@code null}
   * @param separatorChar the separate character
   * @param preserveAllTokens if {@code true}, adjacent separators are
   * treated as empty token separators; if {@code false}, adjacent
   * separators are treated as one separator.
   * @return an array of parsed Strings, {@code null} if null String input
   */
  private static String[] splitWorker(final String str, final char separatorChar, final boolean preserveAllTokens) {
    // Performance tuned for 2.0 (JDK1.4)

    if (str == null) {
      return null;
    }
    final int len = str.length();
    if (len == 0) {
//      return ArrayUtils.EMPTY_STRING_ARRAY;
      return new String[0];
    }
    final List<String> list = new ArrayList<>();
    int i = 0, start = 0;
    boolean match = false;
    boolean lastMatch = false;
    while (i < len) {
      if (str.charAt(i) == separatorChar) {
        if (match || preserveAllTokens) {
          list.add(str.substring(start, i));
          match = false;
          lastMatch = true;
        }
        start = ++i;
        continue;
      }
      lastMatch = false;
      match = true;
      i++;
    }
    if (match || preserveAllTokens && lastMatch) {
      list.add(str.substring(start, i));
    }
    return list.toArray(new String[list.size()]);
  }

  /**
   * <p>Splits the provided text into an array, separator specified.
   * This is an alternative to using StringTokenizer.</p>
   *
   * <p>The separator is not included in the returned String array.
   * Adjacent separators are treated as one separator.
   * For more control over the split use the StrTokenizer class.</p>
   *
   * <p>A {@code null} input String returns {@code null}.</p>
   *
   * <pre>
   * StringUtils.split(null, *)         = null
   * StringUtils.split("", *)           = []
   * StringUtils.split("a.b.c", '.')    = ["a", "b", "c"]
   * StringUtils.split("a..b.c", '.')   = ["a", "b", "c"]
   * StringUtils.split("a:b:c", '.')    = ["a:b:c"]
   * StringUtils.split("a b c", ' ')    = ["a", "b", "c"]
   * </pre>
   *
   * @param str  the String to parse, may be null
   * @param separatorChar  the character used as the delimiter
   * @return an array of parsed Strings, {@code null} if null String input
   * @since 2.0
   */
  public static String[] split(final String str, final char separatorChar) {
    return splitWorker(str, separatorChar, false);
  }

}
