package com.zoyi.channel.plugin.android;

import android.app.Application;
import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.event.BadgeBus;
import com.zoyi.channel.plugin.android.event.ChannelViewBus;
import com.zoyi.channel.plugin.android.event.CommandBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.etc.InAppPushItem;
import com.zoyi.channel.plugin.android.model.etc.PushEvent;
import com.zoyi.channel.plugin.android.model.rest.Event;
import com.zoyi.channel.plugin.android.model.wrapper.EventRepo;
import com.zoyi.channel.plugin.android.model.wrapper.GuestWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.push_bot.PushBotManager;
import com.zoyi.channel.plugin.android.selector.GuestSelector;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.ListUtils;
import com.zoyi.channel.plugin.android.util.ObjectUtils;
import com.zoyi.channel.plugin.android.util.RequestUtils;
import com.zoyi.channel.plugin.android.wrapper.CHDefaultEvent;
import com.zoyi.okhttp3.RequestBody;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.functions.Action1;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.HashMap;
import java.util.Map;

public class ChannelIOManager {

  @Nullable
  private static ChannelIOManager channelIOManager;

  private Application application;

  private int activityRunningCount = 0;

  @Nullable
  private Subscription socketSubscription;

  @Nullable
  private ChannelPluginListener channelPluginListener;

  static void create(Application application) {
    if (channelIOManager == null) {
      channelIOManager = new ChannelIOManager(application);
    }
  }

  private ChannelIOManager(Application application) {
    this.application = application;
  }

  static void subscribeSocket() {
    if (channelIOManager != null) {
      channelIOManager.startReceiveSocket();
    }
  }

  static void release() {
    if (channelIOManager != null) {
      channelIOManager.stopReceiveSocket();
    }
  }

  static void fetchMe(@Nullable final OnGuestUpdatedListener listener) {
    if (channelIOManager != null) {
      ChannelApiManager.get()
          .touch()
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<GuestWrapper>() {
            @Override
            public void onNext(GuestWrapper repo) {
              if (repo != null) {
                repo.update();

                if (listener != null) {
                  listener.onGuestUpdated();
                }
              }
            }
          });
    }
  }

  private void startReceiveSocket() {
    if (socketSubscription == null || socketSubscription.isUnsubscribed()) {
      socketSubscription = RxBus.observable()
          .subscribe(new Action1<Object>() {
            @Override
            public void call(Object o) {
              if (o instanceof CommandBus) {
                CommandBus commandBus = (CommandBus) o;

                switch (commandBus.getCommand()) {
                  case READY:
                    fetchMe(null);
                    break;
                }
              }

              if (o instanceof BadgeBus) {
                if (channelPluginListener != null) {
                  channelPluginListener.onChangeBadge(((BadgeBus) o).getCount());
                }
              }

              if (o instanceof ChannelViewBus) {
                ChannelViewBus channelViewBus = (ChannelViewBus) o;

                if (channelPluginListener != null) {

                  switch (channelViewBus.getChannelViewEvent()) {
                    case SHOW_IN_APP_PUSH:
                      if (channelViewBus.getData() != null && channelViewBus.getData() instanceof InAppPushItem) {
                        channelPluginListener.onReceivePush(new PushEvent((InAppPushItem) channelViewBus.getData()));
                      }
                      break;

                    default:
                      break;
                  }
                }
              }
            }
          });
    }
  }

  private void stopReceiveSocket() {
    if (socketSubscription != null && !socketSubscription.isUnsubscribed()) {
      socketSubscription.unsubscribe();
    }
  }

  public static void increaseActivityRunningCount() {
    if (channelIOManager != null) {
      channelIOManager.activityRunningCount++;
    }
  }

  public static void decreaseActivityRunningCount() {
    if (channelIOManager != null) {
      channelIOManager.activityRunningCount--;
    }
  }

  public static boolean isChannelIOActivityRunning() {
    return channelIOManager != null && channelIOManager.activityRunningCount > 0;
  }

  @Nullable
  public static ChannelPluginListener getChannelPluginListener() {
    if (channelIOManager != null) {
      return channelIOManager.channelPluginListener;
    }
    return null;
  }

  static void setChannelPluginListener(ChannelPluginListener channelPluginListener) {
    if (channelIOManager != null) {
      channelIOManager.channelPluginListener = channelPluginListener;
    }
  }

  static void clearChannelPluginListener() {
    if (channelIOManager != null) {
      channelIOManager.channelPluginListener = null;
    }
  }

  static void trackPageView(@Nullable Context context, @NonNull String pluginId) {
    Map<String, Object> property = new HashMap<>();
    if (context != null) {
      property.put(Const.PROPERTY_URL, context.getClass().getSimpleName());
    }

    track(pluginId, new Event(CHDefaultEvent.Name.PAGE_VIEW, property));
  }

  static void track(@NonNull String pluginId, @NonNull final Event event) {

    RequestBody body = RequestUtils.form()
        .set(Const.TRACK_EVENT_KEY_EVENT, ObjectUtils.toMap(event))
        .set(Const.TRACK_EVENT_KEY_GUEST, ObjectUtils.toMap(GuestSelector.get()))
        .create();

    ChannelApiManager.call(
        ChannelApiManager.get().trackEvent(pluginId, body),
        new RestSubscriber<EventRepo>() {
          @Override
          public void onError(RetrofitException error) {
            L.e(error.getMessage());
          }

          @Override
          public void onSuccess(@NonNull EventRepo eventRepo) {
            if (ListUtils.hasItems(eventRepo.getPushBotCandidates())) {
              PushBotManager.reachPushBots(eventRepo.getPushBotCandidates());
            }
          }
        });
  }
}