package com.zoyi.channel.plugin.android.activity.chat;

import android.content.DialogInterface;
import android.content.Intent;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.Bundle;
import android.support.annotation.AnimRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.content.ContextCompat;
import android.support.v7.app.AlertDialog;
import android.support.v7.widget.LinearLayoutManager;
import android.text.TextUtils;
import android.view.View;
import android.widget.ImageButton;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.BaseActivity;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageContentClickListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnNewMessageAlertClickListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnProfileBotUpdateListener;
import com.zoyi.channel.plugin.android.activity.chat.model.MobileNumber;
import com.zoyi.channel.plugin.android.activity.chat.model.ProfileBotMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.SendingMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.view.ChatHeaderView;
import com.zoyi.channel.plugin.android.activity.chat.view.ChatNavigationView;
import com.zoyi.channel.plugin.android.activity.chat.view.NewMessageAlertView;
import com.zoyi.channel.plugin.android.activity.chat.view.overlayheader.OnHeaderStateChangeListener;
import com.zoyi.channel.plugin.android.activity.chat.view.overlayheader.OverlayHeaderCallback;
import com.zoyi.channel.plugin.android.activity.chat.view.overlayheader.OverlayHeaderRecyclerView;
import com.zoyi.channel.plugin.android.activity.download.DownloadActivity;
import com.zoyi.channel.plugin.android.activity.photo_album.PhotoAlbumActivity;
import com.zoyi.channel.plugin.android.activity.photo_album.PhotoAlbumStorage;
import com.zoyi.channel.plugin.android.activity.photo_picker.PhotoPickerActivity;
import com.zoyi.channel.plugin.android.activity.settings.SettingsActivity;
import com.zoyi.channel.plugin.android.activity.video.VideoViewerActivity;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.enumerate.StoreType;
import com.zoyi.channel.plugin.android.enumerate.Transition;
import com.zoyi.channel.plugin.android.enumerate.UpdateType;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.ActionInput;
import com.zoyi.channel.plugin.android.model.entity.Entity;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.File;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.model.rest.UserChat;
import com.zoyi.channel.plugin.android.selector.*;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.channel.plugin.android.util.lang.StringUtils;
import com.zoyi.channel.plugin.android.util.message_format.type.LinkType;
import com.zoyi.channel.plugin.android.view.dialog.CountryCodeDialog;
import com.zoyi.channel.plugin.android.view.handler.EditTextChangedListener;
import com.zoyi.channel.plugin.android.view.handler.InfiniteScrollListener;
import com.zoyi.channel.plugin.android.view.layout.BottomFloatingActionLayout;
import com.zoyi.channel.plugin.android.view.layout.CHTextView;
import com.zoyi.channel.plugin.android.view.layout.NewChatButton;
import com.zoyi.channel.plugin.android.view.layout.WatchedEditText;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.functions.Action1;

import java.util.concurrent.TimeUnit;

import static com.zoyi.channel.plugin.android.view.layout.BigBar.MenuPosition;
import static com.zoyi.channel.plugin.android.view.layout.MenuButton.ActionType;
import static com.zoyi.channel.plugin.android.view.layout.MenuButton.MenuState;

/**
 * Created by mika on 2018. 11. 6..
 */

public class ChatActivity2 extends BaseActivity implements
    ChatContract2.View,
    OnNewMessageAlertClickListener,
    EditTextChangedListener,
    OverlayHeaderCallback,
    OnHeaderStateChangeListener,
    OnProfileBotUpdateListener,
    OnMessageContentClickListener {

  private OverlayHeaderRecyclerView overlayHeaderRecyclerView;
  private WatchedEditText editChat;
  private ImageButton buttonSend;
  private View layoutInput;
  private View layoutInputFrame;
  private NewMessageAlertView viewNewMessageAlert;
  private CHTextView textDisabled;
  private BottomFloatingActionLayout viewBottomFloating;
  private NewChatButton buttonNewChat;
  private View buttonPhotoPicker;

  @NonNull
  private ChatNavigationView chatNavigation;
  @NonNull
  private ChatHeaderView chatHeaderView;

  private LinearLayoutManager layoutManager;

  private ChatContract2.Presenter presenter;
  private ChatAdapter2 adapter;

  @Nullable
  private Subscription countSubscription;

  private int exitAnimation = R.anim.ch_plugin_slide_out_right;

  @Initializer
  @Override
  protected void onCreate(@Nullable Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);

    init(R.layout.ch_plugin_activity_chat);

    String chatId = getIntent() != null ? getIntent().getStringExtra(Const.EXTRA_CHAT_ID) : null;
    String pushBotId = getIntent() != null ? getIntent().getStringExtra(Const.EXTRA_PUSH_BOT_ID) : null;

    viewNewMessageAlert = findViewById(R.id.view_new_message_alert);
    buttonPhotoPicker = findViewById(R.id.button_attach);
    buttonSend = findViewById(R.id.iv_chat_send);
    textDisabled = findViewById(R.id.text_disabled_input);
    layoutInputFrame = findViewById(R.id.layout_bottom);
    layoutInput = findViewById(R.id.layout_input);
    viewBottomFloating = findViewById(R.id.ch_layout_bottom_floating_action);
    buttonNewChat = findViewById(R.id.ch_button_new_chat);
    overlayHeaderRecyclerView = findViewById(R.id.recycler_header);
    editChat = findViewById(R.id.edit_chat);

    adapter = new ChatAdapter2();
    adapter.setOnMessageContentClickListener(this);
    adapter.setOnProfileBotUpdateListener(this);

    presenter = new ChatPresenter2(chatId, pushBotId, new TypingManager(editChat));
    presenter.setView(this);
    presenter.setAdapterModel(adapter);
    presenter.setAdapterView(adapter);

    chatNavigation = new ChatNavigationView(this);
    chatHeaderView = new ChatHeaderView(this);

    layoutManager = new LinearLayoutManager(this);
    layoutManager.setStackFromEnd(true);
    viewNewMessageAlert.setListener(this);

    editChat.setWatchedTextChangedListener(this);
    editChat.setOnFocusChangeListener(new View.OnFocusChangeListener() {
      @Override
      public void onFocusChange(View v, boolean hasFocus) {
        adapter.setEditTextFocused(hasFocus);
      }
    });
    editChat.setText(PrefSupervisor.getStoredMessage(this));
    PrefSupervisor.clearStoredMessage(this);

    buttonPhotoPicker.setOnClickListener(new View.OnClickListener() {
      @Override
      public void onClick(View v) {
        hideKeyboard(editChat);
        IntentUtils.setNextActivity(ChatActivity2.this, PhotoPickerActivity.class)
            .startActivityForResult(Const.PHOTO_REQUEST_CODE);
      }
    });

    buttonSend.setOnClickListener(new View.OnClickListener() {
      @Override
      public void onClick(View v) {
        presenter.sendText(StringUtils.stripEnd(editChat.getString(), null));
        editChat.setText("");
      }
    });

    buttonNewChat.setClickListener(new View.OnClickListener() {
      @Override
      public void onClick(View v) {
        setResult(Const.RESULT_CHAT_OPEN_NEW_CHAT);
        finish();
      }
    });

    overlayHeaderRecyclerView.getRecyclerView().setLayoutManager(layoutManager);
    overlayHeaderRecyclerView.getRecyclerView().setAdapter(adapter);
    overlayHeaderRecyclerView.getRecyclerView().setItemAnimator(null);
    overlayHeaderRecyclerView.addBannerView(chatNavigation);
    overlayHeaderRecyclerView.setCallback(this);
    overlayHeaderRecyclerView.setListener(this);
    overlayHeaderRecyclerView.getRecyclerView().addOnScrollListener(
        new InfiniteScrollListener(
            layoutManager,
            InfiniteScrollListener.TOP,
            overlayHeaderRecyclerView.getRecyclerView()) {
          @Override
          public void scrolledInList() {
            viewBottomFloating.hide();
          }

          @Override
          public void scrollAttachedToBottom() {
            viewNewMessageAlert.hide();
            viewBottomFloating.show();
          }

          @Override
          public void refresh() {
            presenter.fetchBackwardMessages();
          }
        });

    viewBottomFloating.setRecyclerView(overlayHeaderRecyclerView.getRecyclerView());

    chatHeaderView.setOnClickListener(new View.OnClickListener() {
      @Override
      public void onClick(View v) {
        overlayHeaderRecyclerView.toggleBanner();
      }
    });
    chatHeaderView.bindChannel(ChannelSelector.getCurrentChannel());

    getBigBar().withActivity(this)
        .addMenu(ActionType.SETTINGS, MenuPosition.LEFT, MenuState.HIDDEN)
        .addMenu(ActionType.BACK, MenuPosition.LEFT, MenuState.HIDDEN)
        .addText(MenuPosition.LEFT)
        .addMenu(ActionType.REFRESH, MenuPosition.RIGHT, MenuState.HIDDEN)
        .addMenu(ActionType.EXIT, MenuPosition.RIGHT)
        .setHeaderView(chatHeaderView)
        .build();

    setInputFrameVisibility(false);
    setInputState(ChannelSelector.getCurrentChannel(), true);

    refreshChatCount(true);
    refreshMenu(chatId == null);

    bindChannel(ChannelSelector.getCurrentChannel());

    presenter.init();
  }

  @Override
  protected void onDestroy() {
    presenter.release();
    if (countSubscription != null && !countSubscription.isUnsubscribed()) {
      countSubscription.unsubscribe();
    }
    super.onDestroy();
  }

  @Override
  public void optionClicked(ActionType actionType) {
    switch (actionType) {
      case REFRESH:
        presenter.refresh();
        break;

      case SETTINGS:
        IntentUtils.setNextActivity(this, SettingsActivity.class)
            .setTransition(Transition.SLIDE_FROM_BOTTOM)
            .startActivityForResult(Const.REQUEST_PLUGIIN_SETTINGS);
        break;

      case BACK:
        onBackPressed();
        break;

      case EXIT:
        exitMessenger();
        break;
    }
  }

  @Override
  public void onBackPressed() {
    if (UserChatSelector.isEmpty() && presenter.isLocalChat()) {
      exitMessenger();
    } else {
      super.onBackPressed();
    }
  }

  private void exitMessenger() {
    exitAnimation = R.anim.ch_plugin_slide_out_bottom;
    setResult(Const.RESULT_CHAT_GO_TO_MAIN);
    finish();
  }

  @AnimRes
  @Override
  protected int getExistAnimOfFinish() {
    return exitAnimation;
  }

  @Override
  protected void onActivityResult(int requestCode, int resultCode, Intent data) {
    super.onActivityResult(requestCode, resultCode, data);
    switch (requestCode) {
      case Const.PHOTO_REQUEST_CODE:
        if (resultCode == Const.PHOTO_RESULT_CODE) {
          presenter.sendFiles(data.getStringArrayListExtra(Const.PHOTO_INTENT_KEY));
        }
        break;

      case Const.REQUEST_PHOTO_ALBUM:
        PhotoAlbumStorage.getInstance().release();
        break;

      case Const.REQUEST_PLUGIIN_SETTINGS:
        if (resultCode == Const.RESULT_CHAT_OPTION_CHANGED) {
          setResult(Const.RESULT_CHAT_OPTION_CHANGED);
        }
        break;
    }
  }

  // initializer

  @Override
  public void refreshChatCount(boolean force) {
    if (force) {
      getBigBar().setText(CounterSelector.getTotalCounter());
    } else {
      if (countSubscription != null && !countSubscription.isUnsubscribed()) {
        countSubscription.unsubscribe();
      }
      countSubscription = Observable.timer(600, TimeUnit.MILLISECONDS)
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new Action1<Long>() {
            @Override
            public void call(Long aLong) {
              getBigBar().setText(CounterSelector.getTotalCounter());
            }
          });
    }
  }

  private void setInputState(@Nullable Channel channel, boolean onInit) {
    if (channel == null || (onInit && !channel.isWorking() && Const.AWAY_OPTION_DISABLED.equals(channel.getAwayOption()))) {
      setInputEnable(false);
    }
    if (channel != null && (channel.isWorking() || Const.AWAY_OPTION_ACTIVE.equals(channel.getAwayOption()))) {
      setInputEnable(true);
    }
  }

  private void setInputEnable(boolean enable) {
    Views.setVisibility(layoutInput, enable);
    Views.setVisibility(textDisabled, !enable);
  }

  public void refreshMenu(boolean isLocalChat) {
    boolean showBack = !isLocalChat || !UserChatSelector.isEmpty();

    getBigBar().setMenuState(ActionType.SETTINGS, !showBack ? MenuState.VISIBLE : MenuState.HIDDEN);
    getBigBar().setMenuState(ActionType.BACK, showBack ? MenuState.VISIBLE : MenuState.HIDDEN);
  }

  // socket, store event receiver

  @Override
  public void receiveStoreEvent(StoreType storeType, UpdateType updateType, @Nullable Entity entity) {
    presenter.receiveStoreEvent(storeType, updateType, entity);
  }

  @Override
  public void receiveCommand(Command command, @Nullable Object object) {
    presenter.receiveCommand(command, object);
  }

  @Override
  public void receiveTyping(Typing typing) {
    presenter.receiveTyping(typing);
  }

  @Override
  public void receiveLanguageEvent() {
    presenter.updateInitMessage();
  }

  // binder

  /**
   * 유저챗 정보가 생성 또는 갱신되었을 때 불리는 함수
   *
   * @param oldUserChat 이전의 유저챗 정보, 처음 불렸을 경우 null일 수 있고, 이전 상태를 알 수 없을 경우 <code>userChat</code>과 동일할 수 있음
   * @param userChat 새롭게 갱신되는 유저챗 정보
   */
  @Override
  public void onUserChatChange(@Nullable UserChat oldUserChat, @NonNull UserChat userChat) {
    if (userChat.isStateRemoved()) {
      finish();
      return;
    }

    chatHeaderView.bindUserChat(userChat);
    chatNavigation.bindUserChat(userChat);

    refreshMenu(false);

    String userChatState = userChat.getState();

    // Input root frame
    setInputFrameVisibility(
        CompareUtils.exists(
            userChatState,
            Const.USER_CHAT_STATE_READY,
            Const.USER_CHAT_STATE_OPEN,
            Const.USER_CHAT_STATE_FOLLOWING,
            Const.USER_CHAT_STATE_HOLDING,
            Const.USER_CHAT_STATE_CONTINUE_CHAT)
    );

    // new chat button
    Views.setVisibility(buttonNewChat, Const.USER_CHAT_STATE_CLOSED.equals(userChatState));

    // When chat is finished, close keyboard
    if (CompareUtils.exists(userChatState, Const.USER_CHAT_STATE_SOLVED, Const.USER_CHAT_STATE_CLOSED)) {
      hideKeyboard(editChat);
    }

    if (oldUserChat != null &&
        !Const.USER_CHAT_STATE_CLOSED.equals(oldUserChat.getState()) &&
        Const.USER_CHAT_STATE_CLOSED.equals(userChat.getState())) {
      tryScrollToBottom();
    }

    viewBottomFloating.bindUserChatState(userChatState);
  }

  @Override
  public void bindChannel(@Nullable Channel channel) {
    chatHeaderView.bindChannel(channel);
    chatNavigation.bindChannel(channel);

    setInputState(channel, false);
  }

  // view callbacks


  @Override
  public void showInitBanner() {
    chatNavigation.bindUserChat(null);
    overlayHeaderRecyclerView.postDelayed(new Runnable() {
      @Override
      public void run() {
        overlayHeaderRecyclerView.showBanner(true);
      }
    }, 400);
  }

  @Override
  public void onError() {
    getBigBar().setMenuState(ActionType.REFRESH, MenuState.VISIBLE);
  }

  @Override
  public void onTryRefresh() {
    getBigBar().setMenuState(ActionType.REFRESH, MenuState.HIDDEN);
  }

  @Override
  public void onNewMessageAlertClick() {
    scrollToBottom();
  }

  @Override
  public boolean canScrollTop() {
    return presenter.hasBackwardMessages();
  }

  @Override
  public void onBannerShow() {
    chatHeaderView.setExpand(true);
  }

  @Override
  public void onBannerHide() {
    chatHeaderView.setExpand(false);
  }

  @Override
  public void scrollToBottom() {
    layoutManager.scrollToPosition(adapter.getItemCount() - 1);
  }

  @Override
  public void tryScrollToBottom() {
    if (layoutManager.findLastCompletelyVisibleItemPosition() >= adapter.getItemCount() - 3) {
      scrollToBottom();
    }
  }

  @Override
  public void onHostMessageArrive(@NonNull Message message) {
    if (layoutManager.findLastCompletelyVisibleItemPosition() >= adapter.getItemCount() - 3) {
      scrollToBottom();
    } else {
      viewNewMessageAlert.show(ProfileSelector.getHostProfile(message.getPersonType(), message.getPersonId()));
    }
  }

  @Override
  public void setInputFrameVisibility(boolean visibility) {
    Views.setVisibility(layoutInputFrame, visibility);
  }

  @Override
  public void onWatchedTextChanged(String s) {
    buttonSend.setEnabled(!TextUtils.isEmpty(s.trim()));
  }

  @Override
  public void onCountrySelectorClick(ProfileBotMessageItem item, String key, MobileNumber mobileNumber) {
    presenter.fetchCountries(item, key, mobileNumber);
  }

  @Override
  public void onCountriesFetch(final ProfileBotMessageItem item, final String key, final MobileNumber mobileNumber) {
    String countryName;

    switch (PrefSupervisor.getPluginLanguage(this)) {
      case "ko":
        countryName = "South Korea";
        break;
      case "ja":
        countryName = "Japan";
        break;
      default:
        countryName = "United States";
        break;
    }
    new CountryCodeDialog(this, countryName, new CountryCodeDialog.OnCountryCodeSelectListener() {
      @Override
      public void onCountryCodeSelected(int position, String countryCode) {
        presenter.selectCountry(item, key, mobileNumber, countryCode);
      }
    }).show();
  }

  @Override
  public void onProfileBotUpdateButtonClick(ProfileBotMessageItem item, String key, Object value) {
    presenter.updateProfile(item, key, value);
  }

  @Override
  public void onProfileBotFocusChange(boolean hasFocus) {
    Drawable background = ContextCompat.getDrawable(
        this, hasFocus ? R.drawable.ch_plugin_chat_disabled_input_box_bg : R.drawable.ch_plugin_chat_input_box_bg);

    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
      layoutInput.setBackground(background);
    } else {
      layoutInput.setBackgroundDrawable(background);
    }
  }

  @Override
  public void onProfileBotUpdateComplete() {
    editChat.requestFocus();
  }

  @Override
  public void onTranslationButtonClick(Message message) {
    if (message != null) {
      presenter.translateMessage(message, PrefSupervisor.getPluginLanguage(this));
    }
  }

  @Override
  public void onImageClick(File file) {
    if (file != null) {
      IntentUtils.setNextActivity(this, PhotoAlbumActivity.class)
          .putExtra(Const.EXTRA_FILE_ID, file.getId())
          .startActivityForResult(Const.REQUEST_PHOTO_ALBUM);
    }
  }

  @Override
  public void onWebPageClick(String url) {
    if (url != null) {
      Executor.executeLinkAction(this, url, LinkType.URL);
    }
  }

  @Override
  public void onFileClick(File file) {
    if (file != null) {
      if (file.getType() != null && file.getType().startsWith("video")) {
        IntentUtils.setNextActivity(this, VideoViewerActivity.class)
            .putExtra(Const.EXTRA_FILE_NAME, file.getName())
            .putExtra(Const.EXTRA_URL, file.getUrl())
            .startActivity();
      } else {
        IntentUtils.setNextActivity(this, DownloadActivity.class)
            .putExtra(Const.EXTRA_URL, file.getUrl())
            .putExtra(Const.EXTRA_FILE_NAME, file.getFilename())
            .putExtra(Const.EXTRA_IMAGE, file.isImage())
            .setTransition(Transition.NONE)
            .startActivity();
      }
    }
  }

  @Override
  public void onFormClick(@Nullable String actionType, ActionInput actionInput) {
    presenter.sendForm(actionType, actionInput);
  }

  @Override
  public void onResendButtonClick(final SendingMessageItem item) {
    final AlertDialog dialog = new AlertDialog.Builder(this)
        .setMessage(ResUtils.getString("ch.chat.resend.description"))
        .setPositiveButton(ResUtils.getString("ch.chat.retry_sending_message"), new DialogInterface.OnClickListener() {
          @Override
          public void onClick(DialogInterface dialog, int which) {
            presenter.resend(item);
          }
        })
        .setNegativeButton(ResUtils.getString("ch.chat.resend.cancel"), null)
        .setNeutralButton(ResUtils.getString("ch.chat.delete"), new DialogInterface.OnClickListener() {
          @Override
          public void onClick(DialogInterface dialog, int which) {
            presenter.cancelSend(item);
          }
        })
        .setCancelable(true)
        .create();

    dialog.setOnShowListener(new DialogInterface.OnShowListener() {
      @Override
      public void onShow(DialogInterface args) {
        int dark = ContextCompat.getColor(ChatActivity2.this, R.color.ch_dark);
        int cobalt = ContextCompat.getColor(ChatActivity2.this, R.color.ch_cobalt);

        dialog.getButton(AlertDialog.BUTTON_POSITIVE).setTextColor(cobalt);
        dialog.getButton(AlertDialog.BUTTON_NEGATIVE).setTextColor(dark);
        dialog.getButton(AlertDialog.BUTTON_NEUTRAL).setTextColor(dark);
      }
    });
    dialog.show();
  }
}
