package com.zoyi.channel.plugin.android.push;

import android.annotation.SuppressLint;
import android.app.Notification;
import android.content.Context;
import android.os.Bundle;
import android.provider.Settings;
import android.support.annotation.Nullable;
import android.support.v4.app.NotificationManagerCompat;
import android.text.TextUtils;
import com.zoyi.channel.plugin.android.ChannelIO;
import com.zoyi.channel.plugin.android.global.CheckInPrefSupervisor;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.User;
import com.zoyi.channel.plugin.android.model.rest.Veil;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.selector.ChannelSelector;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.RequestUtils;
import com.zoyi.okhttp3.RequestBody;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.Map;

public class ChannelPushManager {

  private static final int NOTIFICATION_ID = Integer.MAX_VALUE/2;

  private static Bundle convertMessageMapToBundle(Map<String, String> message) {
    Bundle bundle = new Bundle();

    for (Map.Entry<String, String> entry : message.entrySet()) {
      bundle.putString(entry.getKey(), entry.getValue());
    }
    return bundle;
  }

  public static boolean isChannelPushNotification(Map<String, String> message) {
    return isChannelPushNotification(convertMessageMapToBundle(message));
  }

  public static boolean isChannelPushNotification(Bundle message) {
    if (message == null) {
      return false;
    }
    PushData pushData = new PushData(message);

    return pushData.isChannelPluginMessage();
  }

  public static void sendTokenToChannelIO(@Nullable final Context context) {
    if (context != null) {
      String token = PrefSupervisor.getDeviceToken(context);
      String sentToken = PrefSupervisor.getSentDeviceToken(context);

      if (!TextUtils.isEmpty(token) && !CompareUtils.isSame(token, sentToken)) {
        PrefSupervisor.setSentDeviceToken(context, token);

        RequestBody body = RequestUtils.form()
            .set("platform", "android")
            .set("key", getDeviceId(context))
            .set("token", token)
            .create();

        ChannelIO.getApi().sendToken(body)
            .subscribeOn(Schedulers.newThread())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(new RestSubscriber<Void>());
      }
    }
  }

  @SuppressLint("HardwareIds")
  public static void deleteToken(Context context) {
    Channel channel = ChannelSelector.getCurrentChannel();

    if (channel != null) {
      ChannelIO.getApi().deleteToken(
          channel.getId(),
          "android", getDeviceId(context))
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<Void>());
    }
    
    PrefSupervisor.setSentDeviceToken(context, null);
  }

  @SuppressLint("HardwareIds")
  private static String getDeviceId(Context context) {
    String key = PrefSupervisor.getDeviceId(context);

    if (TextUtils.isEmpty(key)) {
      key = Settings.Secure.getString(context.getContentResolver(), Settings.Secure.ANDROID_ID);
      PrefSupervisor.setDeviceId(context, key);
    }
    return key;
  }

  public static void showPushNotification(Context context, Map<String, String> message) {
    showPushNotification(context, convertMessageMapToBundle(message));
  }

  public static void showPushNotification(Context context, Bundle message) {
    if (CheckInPrefSupervisor.hasCheckInData(context)) {
      String personId = message.getString("personId");
      String personType = message.getString("personType");

      if (CompareUtils.isSame(personId, CheckInPrefSupervisor.getPersonId(context))
          && CompareUtils.exists(personType, User.CLASSNAME, Veil.CLASSNAME)) {
        sendNotification(context, message);
      }
    }
  }

  private static void sendNotification(Context context, Bundle message) {
    Notification notification = NotificationFactory.createPushMessage(context, new PushData(message));

    if (notification != null) {
      NotificationManagerCompat.from(context).notify(NOTIFICATION_ID, notification);
    }
  }

  static void clearNotification(Context context) {
    try {
      NotificationManagerCompat.from(context).cancel(NOTIFICATION_ID);
    } catch (Exception ex) {
      L.e(ex.getMessage());
    }
  }
}
