package com.zoyi.channel.plugin.android.socket;

import android.app.Application;

import com.zoyi.channel.plugin.android.ChannelStore;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.event.ChannelModelBus;
import com.zoyi.channel.plugin.android.event.CommandBus;
import com.zoyi.channel.plugin.android.event.PushBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.event.TypingBus;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.model.rest.Bot;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.File;
import com.zoyi.channel.plugin.android.model.rest.Manager;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.model.interfaces.ProfileEntity;
import com.zoyi.channel.plugin.android.model.rest.Session;
import com.zoyi.channel.plugin.android.model.rest.User;
import com.zoyi.channel.plugin.android.model.rest.UserChat;
import com.zoyi.channel.plugin.android.model.rest.Veil;
import com.zoyi.channel.plugin.android.model.rest.WebPage;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.channel.plugin.android.util.lang.StringUtils;
import com.zoyi.com.google.gson.Gson;
import com.zoyi.io.socket.client.IO;
import com.zoyi.io.socket.client.Socket;
import com.zoyi.io.socket.client.SocketIOException;
import com.zoyi.io.socket.emitter.Emitter;
import com.zoyi.io.socket.engineio.client.EngineIOException;
import com.zoyi.io.socket.engineio.client.transports.WebSocket;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscriber;
import com.zoyi.rx.Subscription;

import org.json.JSONException;
import org.json.JSONObject;

import java.net.URISyntaxException;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Created by mika on 2016. 4. 8..
 */
public class SocketManager {
  private static SocketManager socketManager;

  private Application application;
  private Socket socket;
  private Gson gson;
  private Timer heartbeatTimer, reconnectConsumer;
  private AtomicBoolean forceDisconnect;
  private AtomicBoolean ready;
  private AtomicBoolean error;
  private Subscription heartbeat;
  private BlockingQueue<Integer> reconnectQueue;
  private int[] attemptDelay = {100, 3000, 4000, 5000, 7000, 8000, 10000};
  private AtomicInteger attemptCount = new AtomicInteger();

  private static final int HEARTBEAT_INTERVAL = 30000;
  private static final int RECONNECT_POP_INTERVAL = 500;

  static final String CHAT_SERVER_URL_PRODUCTION = "https://ws.channel.io/app";
  //static final String CHAT_SERVER_URL_PRODUCTION = "http://ws.exp.channel.io/app";

  private String channelId;

  public static void create(Application application) {
    if (socketManager == null) {
      socketManager = new SocketManager();
      socketManager.application = application;
      socketManager.init();
    }
  }

  private void init() {
    gson = new Gson();
    forceDisconnect = new AtomicBoolean(false);
    ready = new AtomicBoolean(false);
    error = new AtomicBoolean(false);
    reconnectQueue = new ArrayBlockingQueue<>(1);

    try {
      IO.Options options = new IO.Options();
      options.reconnection = false;
      options.transports = new String[] { WebSocket.NAME };
      socket = IO.socket(CHAT_SERVER_URL_PRODUCTION, options);
    } catch (URISyntaxException e) {
      throw new RuntimeException(e);
    }

    socket.on(Socket.EVENT_CONNECT, onConnect);                     // void
    socket.on(Socket.EVENT_CONNECT_ERROR, onConnectError);          // EngineIOException
    socket.on(Socket.EVENT_CONNECT_TIMEOUT, onConnectTimeout);
    socket.on(Socket.EVENT_CONNECTING, onConnecting);               // void
    socket.on(Socket.EVENT_DISCONNECT, onDisconnect);                // compare
    socket.on(Socket.EVENT_ERROR, onError);                         // EngineIOException
    //socket.on(Socket.EVENT_MESSAGE, onMessage);
    socket.on(Socket.EVENT_PING, onPing);                           // void
    socket.on(Socket.EVENT_PONG, onPong);                           // long
    socket.on(Socket.EVENT_RECONNECT, onReconnect);                 // int
    socket.on(Socket.EVENT_RECONNECT_ATTEMPT, onReconnectAttempt);  // int
    socket.on(Socket.EVENT_RECONNECT_ERROR, onReconnectError);      // SocketIOException
    socket.on(Socket.EVENT_RECONNECT_FAILED, onReconnectFailed);
    socket.on(Socket.EVENT_RECONNECTING, onReconnecting);           // int

    socket.on(SocketEvent.AUTHENTICATED, onAuthenticated);          // boolean
    socket.on(SocketEvent.READY, onReady);
    socket.on(SocketEvent.CREATE, onCreate);                        // JSONObject
    socket.on(SocketEvent.DELETE, onDelete);                        // JSONObject
    socket.on(SocketEvent.JOINED, onJoined);                        // JSONObject
    socket.on(SocketEvent.LEAVED, onLeaved);                        // JSONObject
    socket.on(SocketEvent.PUSH, onPush);                          // JSONObject
    //socket.on(SocketEvent.RECONNECT_ATTEMPT, onReconnectAttempt2);
    socket.on(SocketEvent.UNAUTHORIZED, onUnauthorized);            // JSONObject
    socket.on(SocketEvent.UPDATE, onUpdate);                        // JSONObject
    socket.on(SocketEvent.TYPING, onTyping);                        // JSONObject
  }

  public static void setChannelId(String channelId) {
    if (socketManager != null) {
      socketManager.channelId = channelId;
    }
  }

  // static functions

  public static boolean isReady() {
    if (socketManager != null) {
      return socketManager.ready.get();
    }
    return false;
  }

  public static boolean isError() {
    if (socketManager != null) {
      return socketManager.error.get();
    }
    return false;
  }

  public static void connect() {
    if (socketManager != null) {
      socketManager.connectSocket();
    }
  }

  public static void reconnect() {
    if (socketManager != null && ChannelStore.isMainRunning()) {
      socketManager.enqueueReconnect();
    }
  }

  public static void joinChat(String chatId) {
    if (socketManager != null) {
      socketManager.chatAction(SocketEvent.ACTION_JOIN, chatId);
    }
  }

  public static void leaveChat(String chatId) {
    if (socketManager != null) {
      socketManager.chatAction(SocketEvent.ACTION_LEAVE, chatId);
    }
  }

  public static void typing(Typing typing) {
    try {
      if (socketManager != null) {
        socketManager.emit(SocketEvent.TYPING, new JSONObject(new Gson().toJson(typing)));
      }
    } catch (JSONException e) {
      e.printStackTrace();
    }
  }

  public static void disconnect() {
    if (socketManager != null) {
      socketManager.disconnect(true);
      socketManager.stopHeartbeat();
    }
  }

  public static void destroy() {
    if (socketManager != null) {
      socketManager.setReconnectConsumer(false);
      socketManager.socket.off();
      socketManager.socket.disconnect();

      socketManager.channelId = null;
      socketManager.forceDisconnect = null;
      socketManager.ready = null;
      socketManager.reconnectQueue = null;
      socketManager.socket = null;
      socketManager.gson = null;

      socketManager = null;
    }
  }

  // internal functions

  private void enqueueReconnect() {
    if (!ready.get()) {
      try {
        reconnectQueue.add(1);
      } catch (Exception ignored) {
      }
    }
  }

  private void clearReconnectQueue() {
    try {
      reconnectQueue.clear();
    } catch (Exception ignored) {
    }
  }

  private void connectSocket() {
    if (socket != null && !socket.connected() && channelId != null) {
      L.d("try connect");
      socket.connect();
      setReconnectConsumer(true);
    }
  }

  private void disconnect(boolean force) {
    forceDisconnect.set(force);

    try {
      clearReconnectQueue();
      socket.disconnect();
    } catch (Exception ex) {
    }

    if (!force) {
      enqueueReconnect();
    } else {
      setReconnectConsumer(false);
    }
  }

  private void authentication() {
    String personType = ChannelStore.getPersonType();
    String personId = ChannelStore.getPersonId();

    if (personType != null && personId != null && channelId != null) {
      String info = String.format("{\n" +
          "  type: \"Plugin\",\n" +
          "  channelId: \"%s\",\n" +
          "  guestType: \"%s\",\n" +
          "  guestId: \"%s\"" +
          "}", channelId, personType, personId);
      try {
        JSONObject jsonObject = new JSONObject(info);
        emit(SocketEvent.ACTION_AUTHENTICATION, jsonObject);
      } catch (JSONException e) {
        e.printStackTrace();
        RxBus.post(new CommandBus(Command.UNAUTHORIZED));
      }
    } else {
      disconnect(true);
    }
  }

  private synchronized void setReconnectConsumer(boolean flag) {
    if (flag) {
      if (reconnectConsumer == null) {
        reconnectConsumer = new Timer();
        reconnectConsumer.schedule(new TimerTask() {
          @Override
          public void run() {
            try {
              Integer data = reconnectQueue.peek();
              if (data != null) {
                int index = Math.min(attemptCount.getAndIncrement(), attemptDelay.length - 1);
                Thread.sleep(attemptDelay[index]);
                reconnectQueue.remove();
                connectSocket();
              }
            } catch (Exception e) {
              L.e(e.getMessage());
            }
          }
        }, RECONNECT_POP_INTERVAL, RECONNECT_POP_INTERVAL);
      }
    } else {
      if (reconnectConsumer != null) {
        try {
          reconnectConsumer.cancel();
        } catch (Exception ignored) {
        } finally {
          reconnectConsumer = null;
        }
      }
    }
  }

  private void startHeartbeat() {
    stopHeartbeat();
    heartbeat = Observable.interval(HEARTBEAT_INTERVAL, TimeUnit.MILLISECONDS)
        .subscribe(new Subscriber<Long>() {
          @Override
          public void onCompleted() {
          }

          @Override
          public void onError(Throwable e) {
          }

          @Override
          public void onNext(Long aLong) {
            try {
//              L.i("Heartbeat");
              socket.emit(SocketEvent.ACTION_HEARTBEAT, "");
            } catch (Exception ex) {
              L.e("socket error");
            }
          }
        });
  }

  private synchronized void stopHeartbeat() {
    if (heartbeat != null && !heartbeat.isUnsubscribed()) {
      heartbeat.unsubscribe();
    }
  }

  private void emit(String event, Object object) {
    if (object == null || socket == null) {
      return;
    }
    socket.emit(event, object);
  }

  private void chatAction(String action, String chatId) {
    if (socket == null) {
      return;
    }
    if (chatId == null || !ready.get()) {
      return;
    }
    String message = String.format("/user_chats/%s", chatId);
    emit(action, message);
  }

  // Events

  private Emitter.Listener onTyping = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onTyping: " + objects[0].toString());
      Typing typing = parseJson(objects[0].toString(), Typing.class);

      if (typing != null) {
        typing.setCreatedAt(System.currentTimeMillis());
        RxBus.post(new TypingBus(typing));
      }
    }
  };

  private Emitter.Listener onConnect = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onConnect: ");
      try {
        error.set(false);
        forceDisconnect.set(false);
        attemptCount.set(0);
        authentication();
      } catch (Exception ex) {
        L.e(ex.getMessage());
      }
      RxBus.post(new CommandBus(Command.SOCKET_CONNECTED));
    }
  };

  private Emitter.Listener onConnectTimeout = new Emitter.Listener() {
    @Override
    public void call(Object... objects){
      L.d("onConnectTimeout");
    }
  };

  private Emitter.Listener onConnectError = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      ready.set(false);
      enqueueReconnect();

      error.set(true);

      RxBus.post(new CommandBus(Command.SOCKET_DISCONNECTED, forceDisconnect.get()));
      RxBus.post(new CommandBus(Command.SOCKET_ERROR));

      try {
        String message = "";
        if (objects[0] instanceof EngineIOException) {
          EngineIOException exception = (EngineIOException) objects[0];
          message = exception.getMessage();
        }
        if (objects[0] instanceof SocketIOException) {
          SocketIOException exception = (SocketIOException) objects[0];
          message = exception.getMessage();
        }
        L.e("onConnectError: " + message);
      } catch (Exception ex) {
        L.e(ex.getMessage());
      }
    }
  };

  private Emitter.Listener onConnecting = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onConnecting");
//      setSocketStatus(SocketStatus.CONNECTING);
    }
  };

  private Emitter.Listener onDisconnect = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onDisconnect: " + objects[0] + " " + forceDisconnect.get());
      ready.set(false);
      stopHeartbeat();
      RxBus.post(new CommandBus(Command.SOCKET_DISCONNECTED, forceDisconnect.get()));

      if (!forceDisconnect.get()) {
        enqueueReconnect();
      }
    }
  };

  private Emitter.Listener onError = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      EngineIOException exception = (EngineIOException) objects[0];
      L.e("onError: " + objects.length + " " + exception.getMessage());

      error.set(true);

      RxBus.post(new CommandBus(Command.SOCKET_ERROR));

      ready.set(false);
      enqueueReconnect();
    }
  };

  private Emitter.Listener onPing = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      //L.d("onPing");
    }
  };

  private Emitter.Listener onPong = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      //L.d("onPong: " + (Long)objects[0]);
    }
  };

  private Emitter.Listener onReconnect = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onReconnect: " + (int) objects[0]);
    }
  };

  private Emitter.Listener onReconnectAttempt = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onReconnectAttempt: " + (int) objects[0]);
    }
  };

  private Emitter.Listener onReconnectError = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      SocketIOException exception = (SocketIOException) objects[0];
      L.e("onReconnectError: " + exception.getMessage());
      enqueueReconnect();
      ready.set(false);
      RxBus.post(new CommandBus(Command.SOCKET_DISCONNECTED, forceDisconnect.get()));
    }
  };

  private Emitter.Listener onReconnectFailed = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.e("onReconnectFailed");
      enqueueReconnect();
      ready.set(false);
      RxBus.post(new CommandBus(Command.SOCKET_DISCONNECTED, forceDisconnect.get()));
    }
  };

  private Emitter.Listener onReconnecting = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onReconnecting: " + (int) objects[0]);
      RxBus.post(new CommandBus(Command.SOCKET_RECONNECTING));
    }
  };

  private Emitter.Listener onAuthenticated = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onAuthenticated: " + objects[0]);
    }
  };

  private Emitter.Listener onReady = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("Ready");
      ready.set(true);
      startHeartbeat();
      clearReconnectQueue();
      RxBus.post(new CommandBus(Command.READY));
    }
  };

  private Emitter.Listener onJoined = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onJoined: " + objects[0]);
      try {
        String[] split = StringUtils.split((String) objects[0], '/');
        RxBus.post(new CommandBus(Command.JOINED, split[1]));
      } catch (Exception ex) {
        L.e(ex.getMessage());
      }
    }
  };

  private Emitter.Listener onLeaved = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onLeaved: " + objects[0]);
      try {
        String[] split = StringUtils.split((String) objects[0], '/');
        RxBus.post(new CommandBus(Command.LEAVED, split[1]));
      } catch (Exception ex) {
        L.e(ex.getMessage());
      }
    }
  };

  private Emitter.Listener onUnauthorized = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.e("onUnauthorized: " + objects[0].toString());
      ready.set(false);
      forceDisconnect.set(true);
      socket.disconnect();
      RxBus.post(new CommandBus(Command.UNAUTHORIZED));
    }
  };

  private Emitter.Listener onPush = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onPush: " + objects[0].toString());

      try {
        JSONObject json = (JSONObject) objects[0];
        UserChat userChat = null;
        ProfileEntity person = null;

        String type = json.getString("type");

        if (json.has("refers")) {
          JSONObject refers = json.getJSONObject("refers");
          if (refers.has("bot")) {
            person = parseJson(refers.getString("bot"), Bot.class);
          }
          if (refers.has("manager")) {
            person = parseJson(refers.getString("manager"), Manager.class);
          }
          if (refers.has("userChat")) {
            userChat = parseJson(refers.getString("userChat"), UserChat.class);
          }
        }

        switch (type) {
          case Message.CLASSNAME:
            Message message = parseJson(json.getString("entity"), Message.class);
            if (message != null) {
              String string = null;

              if (message.getMessage() != null) {
                string = message.getMessage();
              } else if (message.getFile() != null) {
                if (message.getFile().isImage()) {
                  string = ResUtils.getString(
                      application,
                      "ch.notification.upload_image.description");
                } else {
                  string = ResUtils.getString(
                      application,
                      "ch.notification.upload_file.description");
                }
              }

              RxBus.post(new PushBus(string, person, userChat));
            }
            break;
        }
      } catch (Exception e) {
        L.e(e.getMessage());
      }
    }
  };

  private String getTag(JSONObject json) {
    try {
      String type = json.getString("type");
      String id = json.getJSONObject("entity").getString("id");

      return String.format("%s (%s): ", type, id);
    } catch (Exception ex) {
      return "";
    }
  }

  private Emitter.Listener onCreate = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("+ onCreate) " + getTag((JSONObject) objects[0]) + objects[0].toString());
      onMessage((JSONObject) objects[0], true);
    }
  };

  private Emitter.Listener onUpdate = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("~ onUpdate) " + getTag((JSONObject) objects[0]) + objects[0].toString());
      onMessage((JSONObject) objects[0], true);
    }
  };

  private Emitter.Listener onDelete = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("- onDelete) " + getTag((JSONObject) objects[0]) + objects[0].toString());
      onMessage((JSONObject) objects[0], false);
    }
  };

  private void onMessage(JSONObject json, boolean upsert) {
    try {
      if (!json.has("type") || !json.has("entity")) {
        return;
      }

      String entity = json.getString("entity");
      String type = json.getString("type");

      switch (type) {
        case Manager.CLASSNAME:
          Manager manager = parseJson(entity, Manager.class);
          if (manager == null) {
            return;
          }
          RxBus.post(new ChannelModelBus(manager, upsert));
          break;

        case Message.CLASSNAME:
          Message message = parseJson(entity, Message.class);
          if (message == null) {
            return;
          }

          if (json.has("refers")) {
            JSONObject refers = json.getJSONObject("refers");
            if (refers.has("file")) {
              File fileRefers = parseJson(refers.getString("file"), File.class);
              RxBus.post(new ChannelModelBus(fileRefers, upsert));
            }
            if (refers.has("webPage")) {
              WebPage webPageRefers = parseJson(refers.getString("webPage"), WebPage.class);
              RxBus.post(new ChannelModelBus(webPageRefers, upsert));
            }
            if (refers.has("bot")) {
              Bot botRefers = parseJson(refers.getString("bot"), Bot.class);
              RxBus.post(new ChannelModelBus(botRefers, upsert));
            }
          }

          RxBus.post(new ChannelModelBus(message, upsert));
          break;

        case UserChat.CLASSNAME:
          UserChat userChat = parseJson(entity, UserChat.class);
          if (userChat == null) {
            return;
          }

          if (json.has("refers")) {
            JSONObject refers = json.getJSONObject("refers");
            if (refers.has("message")) {
              Message referMessage = parseJson(refers.getString("message"), Message.class);
              RxBus.post(new ChannelModelBus(referMessage, upsert));
            }
            if (refers.has("manager")) {
              Manager referManager = parseJson(refers.getString("manager"), Manager.class);
              RxBus.post(new ChannelModelBus(referManager, upsert));
            }
            if (refers.has("bot")) {
              Bot referBot = parseJson(refers.getString("bot"), Bot.class);
              RxBus.post(new ChannelModelBus(referBot, upsert));
            }
          }
          RxBus.post(new ChannelModelBus(userChat, upsert));
          break;

        case Session.CLASSNAME:
          Session session = parseJson(entity, Session.class);
          if (session == null) {
            return;
          }

          if (json.has("refers")) {
            JSONObject refers = json.getJSONObject("refers");
            if (refers.has("bot")) {
              Bot referBot = parseJson(refers.getString("bot"), Bot.class);
              RxBus.post(new ChannelModelBus(referBot, upsert));
            }
            if (refers.has("manager")) {
              Manager referManager = parseJson(refers.getString("manager"), Manager.class);
              RxBus.post(new ChannelModelBus(referManager, upsert));
            }
          }

          RxBus.post(new ChannelModelBus(session, upsert));

          break;

        case File.CLASSNAME:
          File file = parseJson(entity, File.class);
          if (file != null) {
            RxBus.post(new ChannelModelBus(file, upsert));
          }
          break;

        case Channel.CLASSNAME:
          Channel channel = parseJson(entity, Channel.class);
          if (channel != null) {
            ChannelStore.setChannel(channel);
            RxBus.post(new ChannelModelBus(channel, upsert));
          }
          break;

        case Veil.CLASSNAME:
          Veil veil = parseJson(entity, Veil.class);
          if (veil != null) {
            ChannelStore.setUserVeil(null, veil);
          }
          break;

        case User.CLASSNAME:
          User user = parseJson(entity, User.class);
          if (user != null) {
            ChannelStore.setUserVeil(user, null);
          }
          break;
      }
    } catch (Exception ex) {
      L.e(ex.getMessage());
    }
  }

  private <T> T parseJson(String entity, Class<T> target) {
    if (entity == null) {
      return null;
    }
    try {
      return gson.fromJson(entity, target);
    } catch (Exception ex) {
      ex.printStackTrace();
      return null;
    }
  }
}

