package com.zoyi.channel.plugin.android.activity.chat;

import android.graphics.Color;
import android.support.annotation.ColorInt;
import android.support.v7.util.SortedList;
import android.support.v7.widget.RecyclerView;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.ViewGroup;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.SortedListCallback;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageClickListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageReviewListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnSendingMessageClickListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnUserInfoListener;
import com.zoyi.channel.plugin.android.activity.chat.model.ActionMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.ChatMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.CloseMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.DateItem;
import com.zoyi.channel.plugin.android.activity.chat.model.LogMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.MessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.QuestionMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.ResolveMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.SendingMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.TypingItem;
import com.zoyi.channel.plugin.android.activity.chat.model.UserInfoItem;
import com.zoyi.channel.plugin.android.activity.chat.type.MessageType;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.ActionMessageHolder;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.CloseMessageHolder;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.DateDividerHolder;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.LogMessageHolder;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.ManagerMessageHolder;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.NewMessageHolder;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.QuestionMessageHolder;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.ResolveMessageHolder;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.SendingMessageHolder;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.TypingHolder;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.UserInfoMessageHolder;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.UserMessageHolder;
import com.zoyi.channel.plugin.android.activity.photo_album.PhotoAlbumStorage;
import com.zoyi.channel.plugin.android.model.rest.File;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.model.rest.User;
import com.zoyi.channel.plugin.android.model.rest.UserChat;
import com.zoyi.channel.plugin.android.model.rest.Veil;
import com.zoyi.channel.plugin.android.presenter.chat.ChatAdapterContract;
import com.zoyi.channel.plugin.android.util.CompareUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * Created by mika on 2016. 12. 8..
 */
public class ChatAdapter extends RecyclerView.Adapter<RecyclerView.ViewHolder>
    implements ChatAdapterContract.Model, ChatAdapterContract.View {

  private UserChat userChat;

  private SortedList<MessageItem> items;

  private OnMessageClickListener onMessageClickListener;

  private OnSendingMessageClickListener onSendingMessageClickListener;

  private OnUserInfoListener onUserInfoListener;

  private OnMessageReviewListener onMessageReviewListener;

  private ResolveMessageItem resolveMessageItem = null;
  private CloseMessageItem closeMessageItem = null;

  private int backColor = Color.BLACK;
  private int textColor = Color.WHITE;

  private boolean isNewChat;

  public ChatAdapter(boolean isNewChat) {
    this.isNewChat = isNewChat;
    items = new SortedList<>(MessageItem.class, new SortedListCallback<MessageItem>(this, false));

    // Typing layout must always exists
    items.add(new TypingItem());
  }

  public void setColor(@ColorInt int backColor, @ColorInt int textColor) {
    this.backColor = backColor;
    this.textColor = textColor;
  }

  @Override
  public void setOnMessageClickListener(OnMessageClickListener onMessageClickListener) {
    this.onMessageClickListener = onMessageClickListener;
  }

  @Override
  public void setOnSendingMessageClickListener(OnSendingMessageClickListener onSendingMessageClickListener) {
    this.onSendingMessageClickListener = onSendingMessageClickListener;
  }

  public void setOnUserInfoListener(OnUserInfoListener onUserInfoListener) {
    this.onUserInfoListener = onUserInfoListener;
  }

  @Override
  public void setOnMessageReviewListener(OnMessageReviewListener onMessageReviewListener) {
    this.onMessageReviewListener = onMessageReviewListener;
  }

  @Override
  public void notifyItemChange(int position) {
    notifyItemChanged(position);
  }

  @Override
  public void setUserChat(UserChat userChat) {
    this.userChat = userChat;
  }

  @Override
  public void addMessage(Message message, String chatId) {
    addMessages(Collections.singletonList(message), chatId);
  }

  @Override
  public void addMessages(List<Message> messages, String chatId) {
    if (messages == null || messages.isEmpty()) {
      return;
    }

    List<MessageItem> list = new ArrayList<>();

    for (Message message : messages) {
      if (message == null
          || !CompareUtils.isSame(chatId, message.getChatId())) {
        continue;
      }

      if (message.getBotOption() != null && message.getBotOption().isWelcome() && isNewChat) {
        continue;
      }

      if (message.getLog() != null) {
        list.add(new LogMessageItem(message));
        continue;
      }

      list.add(new ChatMessageItem(message));
      list.add(new DateItem(message.getCreatedAt()));

      if (CompareUtils.exists(message.getPersonType(), User.CLASSNAME, Veil.CLASSNAME)) {
        SendingMessageItem item = SendingMessageItem.create(
            message.getRequestId(),
            message.getChannelId(),
            message.getChatId());
        if (item != null) {
          items.remove(item);
        }
      }
    }

    if (userChat.isStateResolved()) {
      if (resolveMessageItem == null) {
        resolveMessageItem = new ResolveMessageItem(Long.MAX_VALUE);
        list.add(resolveMessageItem);
      }
    } else if (userChat.isStateClosed()) {
      removeMessageItem(resolveMessageItem);

      if (closeMessageItem == null) {
        closeMessageItem = new CloseMessageItem(userChat, Long.MAX_VALUE);
        list.add(closeMessageItem);
      }
    }

    items.addAll(list);
  }

  @Override
  public void addMessageItem(MessageItem item) {
    items.add(item);
  }

  @Override
  public void addMessageItems(Collection<MessageItem> items) {
    this.items.addAll(items);
  }

  @Override
  public void removeMessageItem(MessageItem item) {
    if (item == null) {
      return;
    }
    items.remove(item);
  }

  @Override
  public void addOrUpdateMessageItem(int position, MessageItem item) {
    if (position >= 0) {
      items.updateItemAt(position, item);
    } else {
      addMessageItem(item);
    }
  }

  @Override
  public void addMessageItem(UserChat userChat) {
    if (userChat == null) {
      return;
    }

    synchronized (ChatAdapter.class) {
      if (userChat.isStateResolved()) {
        if (resolveMessageItem == null) {
          resolveMessageItem = new ResolveMessageItem(Long.MAX_VALUE);
          addMessageItem(resolveMessageItem);
        }
      } else if (userChat.isStateClosed()) {
        removeMessageItem(resolveMessageItem);

        if (closeMessageItem == null) {
          closeMessageItem = new CloseMessageItem(userChat, Long.MAX_VALUE);
          addMessageItem(closeMessageItem);
        }
      }
    }
  }

  @Override
  public MessageItem getItem(int index) {
    if (index < 0 || index >= items.size()) {
      return null;
    }
    return items.get(index);
  }

  @Override
  public RecyclerView.ViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
    LayoutInflater i = LayoutInflater.from(parent.getContext());
    MessageType type = MessageType.fromId(viewType);

    switch (type) {
      case ACTION_MESSAGE:
        return new ActionMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_manager, parent, false),
            onMessageClickListener);

      case QUESTION_MESSAGE:
        return new QuestionMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_user, parent, false),
            backColor,
            textColor);

      case MANAGER_MESSAGE:
      case BOT_MESSAGE:
        return new ManagerMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_manager, parent, false),
            onMessageClickListener);

      case SENDING:
        return new SendingMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_user, parent, false),
            onSendingMessageClickListener,
            backColor,
            textColor);

      case USER_MESSAGE:
        return new UserMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_user, parent, false),
            onMessageClickListener,
            backColor,
            textColor);

      case DATE:
        return new DateDividerHolder(
            i.inflate(R.layout.ch_plugin_item_message_date, parent, false));

      case NEW_MESSAGE_DIVIDER:
        return new NewMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_unread_divider, parent, false));

      case USER_INFO:
        return new UserInfoMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_user_info, parent, false),
            onUserInfoListener);

      case RESOLVE:
        return new ResolveMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_resolve, parent, false),
            onMessageReviewListener);

      case CLOSE:
        return new CloseMessageHolder(i.inflate(R.layout.ch_plugin_item_message_close, parent, false));

      case LOG:
        return new LogMessageHolder(i.inflate(R.layout.ch_plugin_item_message_log, parent, false));

      case TYPING:
        return TypingHolder.newInstance(parent);

      default:
        return null;
    }
  }

  @Override
  public void onBindViewHolder(RecyclerView.ViewHolder holder, int position) {
    MessageItem item = items.get(position);
    ChatMessageItem chatMessageItem;

    switch (item.getType()) {
      case ACTION_MESSAGE:
        ActionMessageHolder actionMessageHolder = (ActionMessageHolder) holder;
        actionMessageHolder.set((ActionMessageItem) item);
        break;

      case QUESTION_MESSAGE:
        QuestionMessageHolder questionMessageHolder = (QuestionMessageHolder) holder;
        questionMessageHolder.set((QuestionMessageItem) item);
        break;

      case MANAGER_MESSAGE:
      case BOT_MESSAGE:
        chatMessageItem = (ChatMessageItem) item;

        ManagerMessageHolder managerMessageHolder = (ManagerMessageHolder) holder;
        managerMessageHolder.set(
            chatMessageItem,
            chatMessageItem.isConnected(getItem(position - 1)));
        break;

      case USER_MESSAGE:
        chatMessageItem = (ChatMessageItem) item;

        UserMessageHolder userMessageHolder = (UserMessageHolder) holder;
        userMessageHolder.set(
            chatMessageItem.getMessage(),
            chatMessageItem.isConnected(getItem(position - 1)));
        break;

      case SENDING:
        SendingMessageHolder sendingHolder = (SendingMessageHolder) holder;
        sendingHolder.set((SendingMessageItem) item);
        break;

      case DATE:
        DateDividerHolder dateDividerHolder = (DateDividerHolder) holder;
        dateDividerHolder.set((DateItem) item);
        break;

      case USER_INFO:
        UserInfoMessageHolder userInfoMessageHolder = (UserInfoMessageHolder) holder;
        userInfoMessageHolder.set((UserInfoItem) item);
        break;

      case RESOLVE:
        ResolveMessageHolder resolveMessageHolder = (ResolveMessageHolder) holder;
        resolveMessageHolder.bind((ResolveMessageItem) item);
        break;

      case CLOSE:
        CloseMessageHolder closeMessageHolder = (CloseMessageHolder) holder;
        closeMessageHolder.bind((CloseMessageItem) item);
        break;

      case LOG:
        LogMessageHolder logMessageHolder = (LogMessageHolder) holder;
        logMessageHolder.bind((LogMessageItem) item);
        break;

      case TYPING:
        ((TypingHolder) holder).bind((TypingItem) item);
        break;
    }
  }

  @Override
  public int getItemViewType(int position) {
    return items.get(position).getType().toInt();
  }

  @Override
  public int getItemCount() {
    return items.size();
  }

  @Override
  public int getIndex(MessageItem item) {
    return items.indexOf(item);
  }

  @Override
  public void setImageFilesToStorage() {
    PhotoAlbumStorage instance = PhotoAlbumStorage.getInstance();

    for (int i=0; i<items.size(); i++) {
      MessageItem messageItem = items.get(i);

      if (messageItem.getType() == MessageType.MANAGER_MESSAGE
          || messageItem.getType() == MessageType.USER_MESSAGE) {
        ChatMessageItem chatMessageItem = (ChatMessageItem) messageItem;
        Message message = chatMessageItem.getMessage();
        File file = message.getFile();

        if (file != null && file.isImage()) {
          if (!TextUtils.isEmpty(file.getUrl())
              && !TextUtils.isEmpty(file.getFilename())
              && file.getSize() > 0) {
            instance.addImageFile(file);
          }
        }
      }
    }
  }
}
