package com.zoyi.channel.plugin.android.presenter.userchat;

import android.content.Context;
import android.text.TextUtils;

import com.zoyi.channel.plugin.android.ChannelPlugin;
import com.zoyi.channel.plugin.android.activity.userchat_list.OnUserChatCheckListener;
import com.zoyi.channel.plugin.android.activity.userchat_list.OnUserChatClickListener;
import com.zoyi.channel.plugin.android.activity.userchat_list.UserChatCounter;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.enumerate.userchat.UserChatListState;
import com.zoyi.channel.plugin.android.event.ChatCounterBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.rest.ChannelModel;
import com.zoyi.channel.plugin.android.model.rest.Manager;
import com.zoyi.channel.plugin.android.model.rest.Session;
import com.zoyi.channel.plugin.android.model.rest.User;
import com.zoyi.channel.plugin.android.model.rest.UserChat;
import com.zoyi.channel.plugin.android.model.rest.Veil;
import com.zoyi.channel.plugin.android.model.source.userchat.UserChatDataSource;
import com.zoyi.channel.plugin.android.model.source.userchat.UserChatRepository;
import com.zoyi.channel.plugin.android.model.wrapper.UserChatsWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

public class UserChatPresenter
    implements UserChatContract.Presenter, OnUserChatClickListener, OnUserChatCheckListener {

  private UserChatContract.View view;
  private UserChatAdapterContract.View adapterView;
  private UserChatAdapterContract.Model adapterModel;

  private UserChatRepository userChatRepository;

  private Context context;

  private UserChatCounter userChatCounter;
  private UserChatListState state = UserChatListState.WAITING_SOCKET;
  private boolean init = true;
  private String runningChatId;

  public UserChatPresenter(Context context) {
    this.context = context;
    userChatRepository = new UserChatRepository(new UserChatDataSource());

    userChatCounter = new UserChatCounter();
  }

  @Override
  public void setView(UserChatContract.View view) {
    this.view = view;
  }

  @Override
  public void setAdapterView(UserChatAdapterContract.View adapterView) {
    this.adapterView = adapterView;
    this.adapterView.setOnUserChatClickListener(this);
    this.adapterView.setOnUserChatCheckListener(this);
  }

  @Override
  public void setAdapterModel(UserChatAdapterContract.Model adapterModel) {
    this.adapterModel = adapterModel;
  }

  @Override
  public void setRunningChatId(String runningChatId) {
    this.runningChatId = runningChatId;
  }

  @Override
  public void setInit(boolean init) {
    this.init = init;
  }

  @Override
  public int getChatCount(String chatId) {
    return userChatCounter.getCounter(chatId);
  }

  @Override
  public void refresh() {
    switch (state) {
      case WAITING_SOCKET:
        SocketManager.connect();
        break;

      case USER_CHAT_LIST_NOT_LOADED:
        getUserChats();
        break;
    }
  }

  @Override
  public void setState(UserChatListState state) {
    this.state = state;
  }

  @Override
  public void getUserChats() {
    if (SocketManager.isReady()) {
      if (init) {
        view.showProgress(ResUtils.getString(context, "ch.loading_information"));
        adapterModel.clear();
      }
      view.setReconnectVisibility(false);

      setState(UserChatListState.USER_CHAT_LIST_LOADING);
      view.setRefreshVisibility(false);

      fetchUserChats();
    } else {
      setState(UserChatListState.WAITING_SOCKET);
      view.setRefreshVisibility(false);

      SocketManager.reconnect();
    }
  }

  @Override
  public void receiveCommand(Command command, Object object) {
    switch (command) {
      case READY:
        view.setReconnectVisibility(false);

        if (UserChatListState.WAITING_SOCKET.equals(state)) {
          getUserChats();
        } else {
          refresh();
        }
        break;

      case JOINED:
        runningChatId = (String) object;
        break;

      case SOCKET_DISCONNECTED:
        setState(UserChatListState.WAITING_SOCKET);

        view.setRefreshVisibility(true);

        adapterModel.stopRefreshBySocket();
        break;

      case SOCKET_ERROR:
        view.setReconnectVisibility(true);
        break;
    }
  }

  @Override
  public void receiveData(ChannelModel channelModel, boolean upsert) {
    adapterModel.updateDictionary(channelModel, upsert);

    if (channelModel instanceof UserChat) {
      updateUserChat((UserChat) channelModel);
    } else if (channelModel instanceof Session) {
      updateSession((Session) channelModel, upsert);
    } else if (channelModel instanceof Manager) {
      adapterModel.updateManager(channelModel.getId());
    }
  }

  private void updateUserChat(UserChat userChat) {
    boolean isShowClosedChat = PrefSupervisor.isShownClosedChat(context);
    String userChatId = userChat.getId();

    if (TextUtils.isEmpty(userChat.getLastMessageId())) {
      return;
    }

    if (userChat.isStateRemoved()) {
      userChatCounter.delete(userChatId);
      adapterModel.removeItem(userChat);
      return;
    }

    if (userChat.isStateClosed()) {
      if (isShowClosedChat) {
        adapterModel.addOrUpdateItem(userChat);
      } else {
        userChatCounter.delete(userChatId);
        adapterModel.removeItem(userChat);
      }
    } else {
      adapterModel.addOrUpdateItem(userChat);
    }
  }

  private void updateSession(Session session, boolean upsert) {
    adapterModel.updateSession(session);

    if (!CompareUtils.exists(session.getPersonType(), User.CLASSNAME, Veil.CLASSNAME)) {
      return;
    }

    if (upsert) {
      userChatCounter.upsert(session);
    } else {
      userChatCounter.delete(session.getChatId());
    }

    if (session.getId() != null && !CompareUtils.isSame(session.getChatId(), runningChatId)) {
      RxBus.post(new ChatCounterBus(userChatCounter.getCounter(runningChatId)));
    }
  }

  private void fetchUserChats() {
    String[] states = PrefSupervisor.isShownClosedChat(context) ?
        Const.USER_CHAT_STATE_CONTAIN_CLOSED.split("&")
        : Const.USER_CHAT_STATE_EXCEPT_CLOSED.split("&");

    userChatRepository.getUserChats(new RestSubscriber<UserChatsWrapper>() {
      @Override
      public void onError(RetrofitException error) {
        view.hideProgress();

        handleError();
      }

      @Override
      public void onNext(UserChatsWrapper userChatsWrapper) {
        view.hideProgress();

        if (userChatsWrapper == null) {
          handleError();
        } else {
          init = false;

          setState(UserChatListState.READY);
          view.setRefreshVisibility(false);

          adapterModel.init(userChatsWrapper);
          userChatCounter.setList(userChatsWrapper.getSessions());

          RxBus.post(new ChatCounterBus(userChatCounter.getCounter(runningChatId)));

          view.userChatsFetched(userChatsWrapper);
        }
      }
    }, states);
  }

  public void handleError() {
    if (UserChatListState.USER_CHAT_LIST_LOADING.equals(state)) {
      setState(UserChatListState.USER_CHAT_LIST_NOT_LOADED);

      view.setRefreshVisibility(true);
    }
  }

  @Override
  public void requestRemoveChat(final String chatId) {
    view.showProgress(ResUtils.getString(context, "ch.chat.delete_progress"));

    ChannelPlugin.getApi().removeUserChat(chatId)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<Void>() {
          @Override
          public void onError(RetrofitException error) {
            requestRemoveChats();
          }

          @Override
          public void onNext(Void aVoid) {
            userChatCounter.delete(chatId);

            adapterView.removeUserChat(chatId);

            requestRemoveChats();
          }
        });
  }

  @Override
  public void requestRemoveChats() {
    if (adapterModel.hasCheckedChatIds()) {
      requestRemoveChat(adapterModel.getCheckedChatId());
    } else {
      view.completeRemoveChat();
    }
  }

  @Override
  public void setEditMode(boolean editMode) {
    adapterModel.clearCheckedChatId();
    adapterView.setEditMode(editMode);
  }

  @Override
  public void onUserChatClick(String chatId) {
    if (!TextUtils.isEmpty(chatId)) {
      runningChatId = chatId;

      view.startChat(chatId, userChatCounter.getCounter(chatId), false);
    }
  }

  @Override
  public void onUserChatLongClick(String chatId) {
    if (!TextUtils.isEmpty(chatId)) {
      view.deleteChat(chatId);
    }
  }

  @Override
  public void onUserChatCheck(String chatId) {
    if (!TextUtils.isEmpty(chatId)) {
      adapterView.checkUserChat(chatId);

      view.checkUserChat(adapterModel.hasCheckedChatIds());
    }
  }
}
