package com.zoyi.channel.plugin.android.activity.userchat_list;

import android.content.Context;
import android.support.v7.util.SortedList;
import android.support.v7.widget.RecyclerView;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.SortedListCallback;
import com.zoyi.channel.plugin.android.model.rest.ChannelModel;
import com.zoyi.channel.plugin.android.model.rest.Session;
import com.zoyi.channel.plugin.android.model.rest.UserChat;
import com.zoyi.channel.plugin.android.model.source.userchat.UserChatItem;
import com.zoyi.channel.plugin.android.model.wrapper.UserChatsWrapper;
import com.zoyi.channel.plugin.android.presenter.userchat.UserChatAdapterContract;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;

/**
 * Created by mika on 2016. 12. 6..
 */
class UserChatListAdapter extends RecyclerView.Adapter<UserChatHolder>
    implements UserChatAdapterContract.View, UserChatAdapterContract.Model {

  private OnUserChatClickListener clickListener;
  private OnUserChatCheckListener checkListener;

  private Map<String, UserChatItem> map;
  private UserChatListDataDictionary dictionary;
  private SortedList<UserChatItem> items;
  private Queue<String> checkedChatIds;
  private boolean refreshBySocket = false;
  private boolean editMode = false;

  UserChatListAdapter() {
    map = new HashMap<>();
    dictionary = new UserChatListDataDictionary();
    items = new SortedList<>(UserChatItem.class, new SortedListCallback(this, true));
    checkedChatIds = new LinkedList<>();
  }

  @Override
  public void setOnUserChatClickListener(OnUserChatClickListener clickListener) {
    this.clickListener = clickListener;
  }

  @Override
  public void setOnUserChatCheckListener(OnUserChatCheckListener checkListener) {
    this.checkListener = checkListener;
  }

  @Override
  public void setEditMode(boolean editMode) {
    this.editMode = editMode;

    notifyDataSetChanged();
  }

  @Override
  public void removeUserChat(String chatId) {
    if (map.containsKey(chatId)) {
      items.remove(map.get(chatId));
    }
  }

  @Override
  public void checkUserChat(String chatId) {
    if (checkedChatIds.contains(chatId)) {
      removeCheckedChatId(chatId);
    } else {
      addCheckedChatId(chatId);
    }
  }

  private void removeCheckedChatId(String chatId) {
    checkedChatIds.remove(chatId);

    if (map.containsKey(chatId)) {
      int position = items.indexOf(map.get(chatId));

      notifyItemChanged(position);
    }
  }

  private void addCheckedChatId(String chatId) {
    checkedChatIds.add(chatId);

    if (map.containsKey(chatId)) {
      int position = items.indexOf(map.get(chatId));

      notifyItemChanged(position);
    }
  }

  @Override
  public void clear() {
    if (items != null) {
      items.clear();
    }
    if (map != null) {
      map.clear();
    }
    dictionary = new UserChatListDataDictionary();
  }

  @Override
  public void init(UserChatsWrapper wrapper) {
    dictionary.add(wrapper.getManagers());
    dictionary.add(wrapper.getSessions());
    dictionary.add(wrapper.getMessages());

    List<UserChatItem> delta = new ArrayList<>();

    for (UserChat userChat : wrapper.getUserChats()) {
      if (!TextUtils.isEmpty(userChat.getLastMessageId())) {
        UserChatItem item = new UserChatItem(userChat, dictionary);

        if (map.containsKey(userChat.getId())) {
          items.remove(map.get(userChat.getId()));
        }
        delta.add(item);
      }
    }

    for (UserChatItem item : delta) {
      map.put(item.getId(), item);
    }
    items.clear();
    items.addAll(delta);

    refreshBySocket = true;
  }

  @Override
  public void stopRefreshBySocket() {
    this.refreshBySocket = false;
  }

  @Override
  public void clearCheckedChatId() {
    checkedChatIds.clear();
  }

  @Override
  public boolean hasCheckedChatIds() {
    return checkedChatIds.size() > 0;
  }

  @Override
  public String getCheckedChatId() {
    return checkedChatIds.poll();
  }

  @Override
  public void removeItem(UserChat userChat) {
    if (refreshBySocket && userChat != null) {
      String chatId = userChat.getId();

      if (map.containsKey(chatId)) {
        int position = items.indexOf(map.get(chatId));

        if (position >= 0) {
          map.remove(chatId);
          items.removeItemAt(position);
        }
      }
    }
  }

  @Override
  public void updateItem(UserChatItem item) {
    if (refreshBySocket) {
      int position = items.indexOf(item);

      if (position >= 0) {
        items.updateItemAt(position, item);
      }
    }
  }

  @Override
  public void addOrUpdateItem(UserChat userChat) {
    if (refreshBySocket && userChat != null) {
      String chatId = userChat.getId();
      UserChatItem item;

      if (map.containsKey(chatId)) {
        item = map.get(chatId);
        item.setUserChat(userChat);

        updateItem(item);
      } else {
        item = new UserChatItem(userChat, dictionary);

        map.put(item.getId(), item);
        items.add(item);
      }
    }
  }

  @Override
  public void updateDictionary(ChannelModel channelModel, boolean upsert) {
    if (refreshBySocket && channelModel != null) {
      if (upsert) {
        dictionary.add(channelModel);
      } else {
        dictionary.remove(channelModel);
      }
    }
  }

  @Override
  public UserChatHolder onCreateViewHolder(ViewGroup parent, int viewType) {
    Context c = parent.getContext();
    View view = LayoutInflater.from(c).inflate(R.layout.ch_plugin_item_user_chat, parent, false);

    return new UserChatHolder(view, clickListener, checkListener);
  }

  @Override
  public void onBindViewHolder(UserChatHolder holder, int position) {
    holder.setUserChatItem(
        items.get(position),
        checkedChatIds.contains(items.get(position).getId()),
        editMode);
  }

  @Override
  public int getItemCount() {
    return items.size();
  }

  @Override
  public void updateManager(String managerId) {
    if (refreshBySocket) {
      for (int i = 0; i < items.size(); i++) {
        UserChatItem item = items.get(i);

        if (item.containsManager(managerId)) {
          items.updateItemAt(i, item);
        }
      }
    }
  }

  @Override
  public void updateSession(Session session) {
    if (refreshBySocket && session != null) {
      for (int i = 0; i < items.size(); i++) {
        UserChatItem item = items.get(i);

        if (item.getSessionId().equals(session.getId())) {
          item.setSession(session);
          items.updateItemAt(i, item);
          break;
        }
      }
    }
  }
}
