package com.zoyi.channel.plugin.android.activity.userchat_list;

import android.content.Intent;
import android.graphics.Color;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.view.View;

import com.zoyi.channel.plugin.android.ChannelStore;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.BaseActivity;
import com.zoyi.channel.plugin.android.activity.chat.ChatActivity;
import com.zoyi.channel.plugin.android.activity.settings.SettingsActivity;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.ChannelModel;
import com.zoyi.channel.plugin.android.model.rest.Plugin;
import com.zoyi.channel.plugin.android.model.wrapper.UserChatsWrapper;
import com.zoyi.channel.plugin.android.presenter.userchat.UserChatContract;
import com.zoyi.channel.plugin.android.presenter.userchat.UserChatPresenter;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.util.IntentUtils;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.channel.plugin.android.util.UIUtils;
import com.zoyi.channel.plugin.android.util.Utils;
import com.zoyi.channel.plugin.android.view.external.fab.FloatingActionButton;
import com.zoyi.channel.plugin.android.view.handler.AlertDialogListener;
import com.zoyi.channel.plugin.android.view.handler.InfiniteScrollListener;
import com.zoyi.channel.plugin.android.view.layout.BigBar.MenuPosition;
import com.zoyi.channel.plugin.android.view.layout.MenuButton.MenuState;

import static com.zoyi.channel.plugin.android.view.layout.MenuButton.ActionType;
import static com.zoyi.channel.plugin.android.view.layout.MenuButton.OnClickListener;

/**
 * Created by mika on 2016. 12. 5..
 */
public class UserChatListActivity
    extends BaseActivity
    implements OnClickListener, UserChatContract.View {

  private View emptyLayout;
  private RecyclerView recyclerView;
  private FloatingActionButton fab;
  private View reconnectLayout;

  private String tempChatId;

  private UserChatListAdapter adapter;

  private boolean editMode = false;

  private UserChatContract.Presenter presenter;

  @Override
  protected void onCreate(@Nullable Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);
    init(R.layout.ch_plugin_activity_user_chat_list);

    if (getIntent() != null) {
      tempChatId = getIntent().getStringExtra(Const.EXTRA_CHAT_ID);
    }

    emptyLayout = findViewById(R.id.layout_empty);
    reconnectLayout = findViewById(R.id.layout_reconnect);
    fab = (FloatingActionButton) findViewById(R.id.fab);
    fab.setOnClickListener(this);
    fab.hide(false);

    getBigBar().withActivity(this)
        .addMenu(ActionType.EXIT, MenuPosition.LEFT)
        .addMenu(ActionType.REFRESH, MenuPosition.RIGHT, MenuState.HIDDEN)
        .addMenu(ActionType.MORE, MenuPosition.RIGHT, MenuState.HIDDEN)
        .addMenu(ActionType.DELETE, MenuPosition.RIGHT, MenuState.HIDDEN)
        .build();

    adapter = new UserChatListAdapter();

    presenter = new UserChatPresenter(this);
    presenter.setView(this);
    presenter.setAdapterView(adapter);
    presenter.setAdapterModel(adapter);

    LinearLayoutManager manager = new LinearLayoutManager(this);
    recyclerView = (RecyclerView) findViewById(R.id.recycler_view);
    recyclerView.setLayoutManager(manager);
    recyclerView.setAdapter(adapter);
    recyclerView.setItemAnimator(null);
    recyclerView.getRecycledViewPool().setMaxRecycledViews(0, 40);
    recyclerView.addOnScrollListener(new InfiniteScrollListener(manager, InfiniteScrollListener.BOTTOM) {
      @Override
      public void refresh() {
        // not yet.
      }
    });

    if (SocketManager.isError()) {
      setRefreshVisibility(true);
      setReconnectVisibility(true);
    }

    setStyle();
    presenter.getUserChats();
  }

  @Override
  protected void onActivityResult(int requestCode, int resultCode, Intent data) {
    if (resultCode == Const.RESULT_CHAT_NOT_EXISTS && adapter.getItemCount() == 0) {
      finish();
    }

    switch (requestCode) {
      case Const.REQUEST_PLUGIIN_MENU:
        if (resultCode == Const.RESULT_CHAT_CLOSE) {
          if (adapter.getItemCount() > 0) {
            setEditMode(true);
          }
        }
        break;
    }
  }

  @Override
  protected void onStart() {
    presenter.setRunningChatId(null);
    super.onStart();
  }

  @Override
  public void onBackPressed() {
    if (editMode) {
      setEditMode(false);
    } else {
      super.onBackPressed();
    }
  }

  @Override
  public void onClick(View view) {
    int id = view.getId();

    if (id == R.id.fab) {
      startChat(null, presenter.getChatCount(null), false);
    }
  }

  @Override
  public void optionClicked(ActionType actionType) {
    switch (actionType) {
      case EXIT:
        if (editMode) {
          setEditMode(false);
          return;
        }
        break;

      case REFRESH:
        getBigBar().setMenuState(ActionType.REFRESH, MenuState.HIDDEN);
        presenter.refresh();
        break;

      case MORE:
        IntentUtils.setNextActivity(this, SettingsActivity.class)
            .startActivityForResult(Const.REQUEST_PLUGIIN_MENU);
        break;

      case DELETE:
        UIUtils.showAlertDialog(this, ResUtils.getString(this, "ch.chat.delete_description"), true, new AlertDialogListener() {
          @Override
          public void onPositive() {
            presenter.requestDeleteChats();
          }
        });
        break;
    }
    super.optionClicked(actionType);
  }

  private void setStyle() {
    Channel channel = ChannelStore.getChannel();

    Plugin plugin = ChannelStore.getPlugin();

    if (channel == null || plugin == null) {
      finish();
    } else {
      setStatusBarColor(plugin.getBackgroundColor());
      getBigBar().setTheme(plugin.getBackgroundColor(), plugin.getTextColor());

      getBigBar().setTitle(channel.getName());

      fab.setColorNormal(plugin.getBackgroundColor());
      fab.setColorPressed(Utils.addBlackMask(plugin.getBackgroundColor()));
      fab.setColorRipple(Color.WHITE);
      fab.setImageResource(
          plugin.getTextColor() == Color.BLACK
              ? R.drawable.ch_plugin_ic_add_black_24dp
              : R.drawable.ch_plugin_ic_add_white_24dp);

      fab.attachToRecyclerView(recyclerView);
    }
  }

  private void setEditMode(boolean editMode) {
    this.editMode = editMode;

    if (editMode) {
      getBigBar().setTitle(ResUtils.getString(this, "ch.chat.delete_title"));
      getBigBar().setMenuState(ActionType.MORE, MenuState.HIDDEN);
      getBigBar().setMenuState(ActionType.DELETE, MenuState.DISABLED);

      fab.setVisibility(View.GONE);
    } else {
      Channel channel = ChannelStore.getChannel();

      if (channel != null) {
        getBigBar().setTitle(channel.getName());
      }
      getBigBar().setMenuState(ActionType.DELETE, MenuState.HIDDEN);
      getBigBar().setMenuState(ActionType.MORE, MenuState.VISIBLE);

      fab.setVisibility(View.VISIBLE);
    }
    presenter.setEditMode(editMode);
  }

  @Override
  public void userChatsFetched(UserChatsWrapper userChatsWrapper) {
    getBigBar().setMenuState(ActionType.MORE, MenuState.VISIBLE);

    fab.show();

    if (tempChatId != null) {
      startChat(tempChatId, presenter.getChatCount(tempChatId), false);
      tempChatId = null;
    } else {
      if (adapter.getItemCount() == 0) {
        startChat(null, 0, true);
      }
    }

    setViewVisibility();
  }

  @Override
  public void startChat(String chatId, int chatCount, boolean chatEmpty) {
    IntentUtils.setNextActivity(this, ChatActivity.class)
        .putExtra(Const.EXTRA_CHAT_ID, chatId)
        .putExtra(Const.EXTRA_CHAT_COUNT, chatCount)
        .putExtra(Const.EXTRA_CHAT_EMPTY, chatEmpty)
        .startActivityForResult(Const.REQUEST_CHAT);
  }

  @Override
  public void setRefreshVisibility(boolean showRefresh) {
    getBigBar().setMenuState(ActionType.REFRESH, showRefresh ? MenuState.VISIBLE : MenuState.HIDDEN);
  }

  @Override
  public void setReconnectVisibility(boolean show) {
    reconnectLayout.setVisibility(UIUtils.getVisible(show, true));
  }

  @Override
  public void checkUserChat(boolean hasCheckedChat) {
    getBigBar().setMenuState(ActionType.DELETE, hasCheckedChat ? MenuState.VISIBLE : MenuState.DISABLED);
  }

  @Override
  public void deleteChat(final String chatId) {
    UIUtils.showAlertDialog(this, ResUtils.getString(this, "ch.chat.delete_description"), true, new AlertDialogListener() {
      @Override
      public void onPositive() {
        presenter.requestDeleteChat(chatId);
      }
    });
  }

  @Override
  public void completeDeleteChat() {
    presenter.hideProgress();
    setEditMode(false);

    setViewVisibility();
  }

  @Override
  public void setPresenter(UserChatContract.Presenter presenter) {}

  @Override
  public void receiveCommand(Command command, Object object) {
    presenter.receiveCommand(command, object);
  }

  @Override
  public void receiveData(ChannelModel channelModel, boolean upsert) {
    presenter.receiveData(channelModel, upsert);

    setViewVisibility();
  }

  private void setViewVisibility() {
    emptyLayout.setVisibility(UIUtils.getVisible(!hasUserChat(), true));
    recyclerView.setVisibility(UIUtils.getVisible(hasUserChat(), true));
  }

  private boolean hasUserChat() {
    return adapter.getItemCount() > 0;
  }
}
