package com.zoyi.channel.plugin.android.presenter.userchat;

import android.app.ProgressDialog;
import android.content.Context;
import android.text.TextUtils;

import com.zoyi.channel.plugin.android.ChannelPlugin;
import com.zoyi.channel.plugin.android.activity.userchat_list.OnUserChatCheckListener;
import com.zoyi.channel.plugin.android.activity.userchat_list.OnUserChatClickListener;
import com.zoyi.channel.plugin.android.activity.userchat_list.UserChatCounter;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.enumerate.userchat.UserChatListState;
import com.zoyi.channel.plugin.android.event.ChatCounterBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.model.rest.ChannelModel;
import com.zoyi.channel.plugin.android.model.rest.Session;
import com.zoyi.channel.plugin.android.model.rest.User;
import com.zoyi.channel.plugin.android.model.rest.Veil;
import com.zoyi.channel.plugin.android.model.source.userchat.UserChatDataSource;
import com.zoyi.channel.plugin.android.model.source.userchat.UserChatRepository;
import com.zoyi.channel.plugin.android.model.wrapper.UserChatWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.UserChatsWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.ProgressHelper;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

public class UserChatPresenter
    implements UserChatContract.Presenter, OnUserChatClickListener, OnUserChatCheckListener {

  private UserChatContract.View userchatView;
  private UserChatAdapterContract.View adapterView;
  private UserChatAdapterContract.Model adapterModel;

  private UserChatRepository userChatRepository;

  private Context context;
  private ProgressDialog dialog;

  private UserChatCounter userChatCounter;
  private UserChatListState state = UserChatListState.WAITING_SOCKET;
  private boolean init = true;
  private String runningChatId;

  public UserChatPresenter(Context context) {
    this.context = context;
    userChatRepository = new UserChatRepository(new UserChatDataSource());

    userChatCounter = new UserChatCounter();
  }

  @Override
  public void showProgress(String message) {
    if (dialog == null || !dialog.isShowing()) {
      dialog = ProgressHelper.show(context, ResUtils.getString(context, "ch.loading_information"));
    }
  }

  @Override
  public void showProgress() {
    showProgress(ResUtils.getString(context, "ch.loading_information"));
  }

  @Override
  public void hideProgress() {
    if (dialog != null && dialog.isShowing()) {
      dialog.dismiss();
    }
  }

  @Override
  public void setView(UserChatContract.View view) {
    this.userchatView = view;
  }

  @Override
  public void setAdapterView(UserChatAdapterContract.View adapterView) {
    this.adapterView = adapterView;
    this.adapterView.setOnUserChatClickListener(this);
    this.adapterView.setOnUserChatCheckListener(this);
  }

  @Override
  public void setAdapterModel(UserChatAdapterContract.Model adapterModel) {
    this.adapterModel = adapterModel;
  }

  @Override
  public void setRunningChatId(String runningChatId) {
    this.runningChatId = runningChatId;
  }

  @Override
  public int getChatCount(String chatId) {
    return userChatCounter.getCounter(chatId);
  }

  @Override
  public void refresh() {
    switch (state) {
      case WAITING_SOCKET:
        SocketManager.connect();
        break;

      case USER_CHAT_LIST_NOT_LOADED:
        getUserChats();
        break;
    }
  }

  @Override
  public void setState(UserChatListState state) {
    this.state = state;
  }

  @Override
  public void getUserChats() {
    if (SocketManager.isReady()) {
      if (init) {
        showProgress();
      }
      userchatView.setReconnectVisibility(false);

      setState(UserChatListState.USER_CHAT_LIST_LOADING);
      userchatView.setRefreshVisibility(false);

      fetchUserChats();
    } else {
      setState(UserChatListState.WAITING_SOCKET);
      userchatView.setRefreshVisibility(false);

      SocketManager.reconnect();
    }
  }

  @Override
  public void receiveCommand(Command command, Object object) {
    switch (command) {
      case READY:
        userchatView.setReconnectVisibility(false);

        if (UserChatListState.WAITING_SOCKET.equals(state)) {
          getUserChats();
        } else {
          refresh();
        }
        break;

      case JOINED:
        runningChatId = (String) object;
        break;

      case SOCKET_DISCONNECTED:
        setState(UserChatListState.WAITING_SOCKET);

        userchatView.setRefreshVisibility(true);

        adapterModel.stopRefreshBySocket();
        break;

      case SOCKET_ERROR:
        userchatView.setReconnectVisibility(true);
        break;
    }
  }

  @Override
  public void receiveData(ChannelModel channelModel, boolean upsert) {
    adapterModel.receiveChannelModel(channelModel, upsert);

    if (channelModel instanceof Session) {
      Session session = (Session) channelModel;

      if (!CompareUtils.exists(session.getPersonType(), User.CLASSNAME, Veil.CLASSNAME)) {
        return;
      }

      if (upsert) {
        userChatCounter.upsert(session);
      } else {
        userChatCounter.delete(session);
      }

      if (session.getId() != null && !CompareUtils.isSame(session.getChatId(), runningChatId)) {
        RxBus.post(new ChatCounterBus(userChatCounter.getCounter(runningChatId)));
      }
    }
  }

  private void fetchUserChats() {
    userChatRepository.getUserChats(new RestSubscriber<UserChatsWrapper>() {
      @Override
      public void onError(RetrofitException error) {
        hideProgress();

        handleError();
      }

      @Override
      public void onNext(UserChatsWrapper userChatsWrapper) {
        hideProgress();

        if (userChatsWrapper == null) {
          handleError();
        } else {
          init = false;

          setState(UserChatListState.READY);
          userchatView.setRefreshVisibility(false);

          adapterModel.init(userChatsWrapper);
          userChatCounter.setList(userChatsWrapper.getSessions());

          RxBus.post(new ChatCounterBus(userChatCounter.getCounter(runningChatId)));

          userchatView.userChatsFetched(userChatsWrapper);
        }
      }
    });
  }

  public void handleError() {
    if (UserChatListState.USER_CHAT_LIST_LOADING.equals(state)) {
      setState(UserChatListState.USER_CHAT_LIST_NOT_LOADED);

      userchatView.setRefreshVisibility(true);
    }
  }

  @Override
  public void requestDeleteChat(final String chatId) {
    showProgress(ResUtils.getString(context, "ch.chat.delete_progress"));

    ChannelPlugin.getApi().closeUserChat(chatId)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<UserChatWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            requestDeleteChats();
          }

          @Override
          public void onNext(UserChatWrapper userChatWrapper) {
            if (userChatWrapper != null) {
              userChatCounter.delete(userChatWrapper.getSession());

              adapterView.removeUserChat(userChatWrapper.getUserChat());
            }
            requestDeleteChats();
          }
        });
  }

  @Override
  public void requestDeleteChats() {
    if (adapterModel.hasCheckedChatIds()) {
      requestDeleteChat(adapterModel.getCheckedChatId());
    } else {
      userchatView.completeDeleteChat();
    }
  }

  @Override
  public void setEditMode(boolean editMode) {
    adapterModel.clearCheckedChatId();
    adapterView.setEditMode(editMode);
  }

  @Override
  public void onUserChatClick(String chatId) {
    if (!TextUtils.isEmpty(chatId)) {
      runningChatId = chatId;

      userchatView.startChat(chatId, userChatCounter.getCounter(chatId), false);
    }
  }

  @Override
  public void onUserChatLongClick(String chatId) {
    if (!TextUtils.isEmpty(chatId)) {
      userchatView.deleteChat(chatId);
    }
  }

  @Override
  public void onUserChatCheck(String chatId) {
    if (!TextUtils.isEmpty(chatId)) {
      adapterView.checkUserChat(chatId);

      userchatView.checkUserChat(adapterModel.hasCheckedChatIds());
    }
  }
}
