package com.zoyi.channel.plugin.android;

import android.app.Application;
import com.zoyi.channel.plugin.android.event.BadgeBus;
import com.zoyi.channel.plugin.android.event.CommandBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.model.wrapper.GuestWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.PluginWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.ScriptsWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscriber;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.functions.Action1;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * Created by mika on 2017. 2. 7..
 */
public class ChannelStore {
  private static ChannelStore channelStore;

  private Application application;
  private String pluginId;

  private Plugin plugin;
  private Channel channel;
  private User user;
  private Veil veil;

  private boolean mainRunning = false;

  private Subscription pluginFetchSubscription;
  private Subscription socketSubscription;
  private Map<String, Script> scriptMap;

  private ArrayList<OnChannelPluginChangedListener> listeners;

  static void create(Application application, String pluginId) {
    if (channelStore == null) {
      channelStore = new ChannelStore(application, pluginId);
    }
  }

  private ChannelStore(Application application, String pluginId) {
    this.application = application;
    this.pluginId = pluginId;
    this.listeners = new ArrayList<>();
  }

  static void checkIn(PluginWrapper pluginWrapper) {
    if (channelStore != null) {
      PrefSupervisor.setVeilId(channelStore.application, pluginWrapper.getVeilId());

      if (pluginWrapper.getUser() != null) {
        channelStore.user = pluginWrapper.getUser();
      }
      channelStore.plugin = pluginWrapper.getPlugin();
      channelStore.channel = pluginWrapper.getChannel();
      channelStore.veil = pluginWrapper.getVeil();

      channelStore.startReceiveSocket();
      channelStore.startFetchPlugin();
      channelStore.fetchScripts();
    }
  }

  static void clear() {
    L.e("clear()");
    if (channelStore != null) {
      channelStore.stopFetchPlugin();
      channelStore.stopReceiveSocket();
      channelStore.plugin = null;
      channelStore.channel = null;
      channelStore.user = null;
      channelStore.veil = null;
    }
  }

  private void startReceiveSocket() {
    if (socketSubscription == null || socketSubscription.isUnsubscribed()) {
      socketSubscription = RxBus.observable()
          .onBackpressureBuffer()
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new Action1<Object>() {
            @Override
            public void call(Object o) {
              if (o instanceof CommandBus) {
                CommandBus commandBus = (CommandBus) o;

                switch (commandBus.getCommand()) {
                  case READY:
                    fetchMe();
                    break;
                }
              }

              if (o instanceof BadgeBus) {
                for (OnChannelPluginChangedListener listener : listeners) {
                  if (listener != null) {
                    listener.badgeChanged(((BadgeBus) o).getCount());
                  }
                }
              }
            }
          });
    }
  }

  private void stopReceiveSocket() {
    if (socketSubscription != null && !socketSubscription.isUnsubscribed()) {
      socketSubscription.unsubscribe();
      socketSubscription = null;
    }
  }

  private void fetchMe() {
    ChannelPlugin.getApi().getMe()
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<GuestWrapper>() {
          @Override
          public void onNext(GuestWrapper guestWrapper) {
            if (guestWrapper != null) {
              setUserVeil(guestWrapper.getUser(), guestWrapper.getVeil());
            }
          }
        });
  }

  private void startFetchPlugin() {
    stopFetchPlugin();
    pluginFetchSubscription = Observable.interval(3, TimeUnit.MINUTES)
        .subscribe(new Subscriber<Long>() {
          @Override
          public void onCompleted() {
          }

          @Override
          public void onError(Throwable e) {
          }

          @Override
          public void onNext(Long aLong) {
            if (channelStore != null) {
              channelStore.fetchPlugin();
            }
          }
        });
  }

  private void stopFetchPlugin() {
    if (pluginFetchSubscription != null && !pluginFetchSubscription.isUnsubscribed()) {
      pluginFetchSubscription.unsubscribe();
      pluginFetchSubscription = null;
    }
  }

  private void fetchPlugin() {
    if (pluginId != null) {
      ChannelPlugin.getApi().getPlugin(pluginId)
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<PluginWrapper>() {
            @Override
            public void onNext(PluginWrapper pluginWrapper) {
              if (pluginWrapper != null) {
                plugin = pluginWrapper.getPlugin();
                channel = pluginWrapper.getChannel();
              }
            }
          });
    }
  }

  private void fetchScripts() {
    ChannelPlugin.getApi().getScripts()
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<ScriptsWrapper>() {
          @Override
          public void onNext(ScriptsWrapper scriptsWrapper) {
            setScripts(scriptsWrapper.getScripts());
          }
        });
  }

  private void setScripts(List<Script> scripts) {
    if (scriptMap == null) {
      scriptMap = new HashMap<>();
    }

    if (scripts != null) {
      for (Script script : scripts) {
        if (script.getKey() != null && script.getMessage() != null) {
          scriptMap.put(script.getKey(), script);
        }
      }
    }
  }

  public static String getScript(String key, String defaultResourceKey) {
    if (channelStore != null) {
      if (channelStore.scriptMap != null && channelStore.scriptMap.containsKey(key)) {
        String locale = Locale.getDefault().getLanguage();
        if (channelStore.scriptMap.get(key).getMessage(locale) != null) {
          return channelStore.scriptMap.get(key).getMessage(locale);
        }
      }
      if (channelStore.application != null) {
        return ResUtils.getString(channelStore.application, defaultResourceKey);
      }
    }
    return "";
  }

  public static Channel getChannel() {
    L.e("getChannel()");
    return channelStore != null ? channelStore.channel : null;
  }

  public static User getUser() {
    return channelStore != null ? channelStore.user : null;
  }

  public static String getName() {
    if (channelStore != null) {
      if (channelStore.user != null && !channelStore.user.isGhost()) {
        return channelStore.user.getName();
      }
      if (channelStore.veil != null && !channelStore.veil.isGhost()) {
        return channelStore.veil.getName();
      }
    }
    return null;
  }

  public static String getMobileNumber() {
    if (channelStore != null) {
      if (channelStore.user != null) {
        return channelStore.user.getMobileNumber();
      }
      if (channelStore.veil != null) {
        return channelStore.veil.getMobileNumber();
      }
    }
    return null;
  }

  public static String getPersonType() {
    if (channelStore != null && channelStore.user != null) {
      return User.CLASSNAME;
    }
    return Veil.CLASSNAME;
  }

  public static String getPersonId() {
    if (channelStore != null) {
      if (channelStore.user != null) {
        return channelStore.user.getId();
      }
      return PrefSupervisor.getVeilId(channelStore.application);
    }
    return null;
  }

  public static String getVeilId() {
    if (channelStore != null && channelStore.application != null) {
      return PrefSupervisor.getVeilId(channelStore.application);
    }
    return null;
  }

  public static String getUserId() {
    if (channelStore != null && channelStore.user != null) {
      return channelStore.user.getId();
    }
    return null;
  }

  public static boolean isWorking() {
    return channelStore != null && channelStore.channel != null && channelStore.channel.isWorking();
  }

  public static boolean hasName() {
    if (channelStore != null) {
      if (channelStore.user != null) {
        return !channelStore.user.isGhost();
      }
      if (channelStore.veil != null) {
        return !channelStore.veil.isGhost();
      }
    }
    return true;
  }

  public static void setUserVeil(User user, Veil veil) {
    if (channelStore != null) {
      if (user != null) {
        channelStore.user = user;
      }
      if (veil != null) {
        channelStore.veil = veil;
      }
      RxBus.post(new BadgeBus(getCount()));
    }
  }

  public static Plugin getPlugin() {
    if (channelStore != null) {
      return channelStore.plugin;
    }
    return null;
  }

  public static String getPluginId() {
    if (channelStore != null) {
      return channelStore.pluginId;
    }
    return null;
  }

  static void setMainRunning(boolean flag) {
    if (channelStore != null) {
      channelStore.mainRunning = flag;
    }
  }

  public static boolean isMainRunning() {
    return channelStore != null && channelStore.mainRunning;
  }

  static boolean isDataStored() {
    return channelStore != null && channelStore.channel != null && channelStore.plugin != null;
  }

  static int getCount() {
    if (channelStore != null) {
      if (channelStore.user != null) {
        return channelStore.user.getAlert();
      }
      if (channelStore.veil != null) {
        return channelStore.veil.getAlert();
      }
    }
    return 0;
  }

  static void addOnChannelPluginChangedListener(OnChannelPluginChangedListener listener) {
    if (channelStore != null && channelStore.listeners != null) {
      channelStore.listeners.add(listener);
    }
  }

  static void removeOnChannelPluginChangedListener(OnChannelPluginChangedListener listener) {
    if (channelStore != null && channelStore.listeners != null) {
      channelStore.listeners.remove(listener);
    }
  }
}
