package com.zoyi.channel.plugin.android.util;

import android.app.Activity;
import android.os.Environment;
import android.text.TextUtils;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.util.io.FilenameUtils;
import com.zoyi.retrofit2.okhttp3.*;
import com.zoyi.retrofit2.okio.*;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.util.Locale;
import java.util.Random;

/**
 * Created by mika on 8/22/16.
 */
public class DownloadUtils {
  private Activity activity;
  private Call call;

  public static Call downloadFile(
      final Activity activity,
      String url,
      String fileName,
      boolean image,
      ProgressListener listener) {
    DownloadUtils utils = new DownloadUtils();
    utils.activity = activity;
    String savePath;
    if (image) {
      File dir = new File(Environment.getExternalStoragePublicDirectory(
          Environment.DIRECTORY_PICTURES),
          Const.EXTERNAL_DOWNLOAD_PATH);

      boolean created = FileUtils.createDirectory(dir);

      if (!created) {
        return null;
      }

      savePath = String.format("%s/%s", dir.getAbsolutePath(), fileName);
    } else {
      savePath = String.format("%s/%s",
          Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_DOWNLOADS).getAbsolutePath(),
          fileName);
    }

    return utils.download(url, savePath, image, listener);
  }

  private Call download(
      String url,
      final String savePath,
      final boolean isImage,
      final ProgressListener listener) {
    final String tempPath;

    try {
      tempPath = activity.getExternalCacheDir() + "/" + getRandomString(16);
    } catch (Exception ex) {
      activity.runOnUiThread(new Runnable() {
        @Override
        public void run() {
          listener.error(null, "Cannot find path");
        }
      });
      return null;
    }

    Request request = new Request.Builder()
        .url(url)
        .build();

    OkHttpClient client = new OkHttpClient.Builder()
        .addNetworkInterceptor(new Interceptor() {
          @Override
          public Response intercept(Chain chain) throws IOException {
            Response originalResponse = chain.proceed(chain.request());
            return originalResponse.newBuilder()
                .body(new ProgressResponseBody(activity, originalResponse.body(), listener))
                .build();
          }
        })
        .build();

    call = client.newCall(request);
    call.enqueue(new Callback() {
      @Override
      public void onFailure(final Call call, final IOException e) {
        L.e(e.getMessage());
        if (listener != null) {
          activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
              listener.error(call, e.getMessage());
            }
          });
        }
      }

      @Override
      public void onResponse(final Call call, Response response) throws IOException {
        try {
          File downloadedFile = new File(tempPath);
          BufferedSink sink = Okio.buffer(Okio.sink(downloadedFile));
          sink.writeAll(response.body().source());
          sink.close();

          FileChannel inChannel = new FileInputStream(tempPath).getChannel();

          for (int count = 1; count < Integer.MAX_VALUE; count++) {
            String newSavePath = getSavePath(savePath, count);
            File file = new File(newSavePath);
            if (!file.exists()) {
              FileChannel outChannel = new FileOutputStream(newSavePath).getChannel();

              inChannel.transferTo(0, inChannel.size(), outChannel);
              inChannel.close();
              outChannel.close();

              FileUtils.deleteFile(new File(tempPath));

              if (isImage) {
                Executor.startFileMediaScan(activity, newSavePath);
              }
              activity.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                  listener.done(savePath);
                }
              });
              break;
            }
          }
        } catch (final Exception ex) {
          response.close();
          activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
              listener.error(call, ex.getMessage());
            }
          });
        }
      }
    });

    return call;
  }

  private static class ProgressResponseBody extends ResponseBody {
    private final Activity activity;
    private final ResponseBody responseBody;
    private final ProgressListener progressListener;
    private BufferedSource bufferedSource;

    ProgressResponseBody(
        Activity activity,
        ResponseBody responseBody,
        ProgressListener progressListener) {
      this.activity = activity;
      this.responseBody = responseBody;
      this.progressListener = progressListener;
    }

    @Override
    public MediaType contentType() {
      return responseBody.contentType();
    }

    @Override
    public long contentLength() {
      return responseBody.contentLength();
    }

    @Override
    public BufferedSource source() {
      if (bufferedSource == null) {
        bufferedSource = Okio.buffer(source(responseBody.source()));
      }
      return bufferedSource;
    }

    private Source source(Source source) {
      return new ForwardingSource(source) {
        long totalBytesRead = 0L;
        int progress = 0;

        @Override
        public long read(Buffer sink, long byteCount) throws IOException {
          long bytesRead = super.read(sink, byteCount);
          // read() returns the number of bytes read, or -1 if this source is exhausted.
          totalBytesRead += bytesRead != -1 ? bytesRead : 0;
          try {
            final int p = (int) ((100 * totalBytesRead) / responseBody.contentLength());
            if (p != progress) {
              progress = p;
              activity.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                  progressListener.update(p);
                }
              });
            }
          } catch (Exception ex) {
          }

          return bytesRead;
        }
      };
    }
  }

  private static String getSavePath(String path, int count) {
    if (count == 1) {
      return path;
    }

    String parent = FilenameUtils.getFullPathNoEndSeparator(path);
    String filename = FilenameUtils.getBaseName(path);
    String ext = FilenameUtils.getExtension(path);

    ext = !TextUtils.isEmpty(ext) ? String.format(".%s", ext) : "";
    return String.format(
        Locale.US,
        "%s/%s (%d)%s",
        parent,
        filename,
        count,
        ext);
  }

  private static final String ALLOWED_CHARACTERS ="0123456789qwertyuiopasdfghjklzxcvbnm";

  private static String getRandomString(final int sizeOfRandomString) {
    final Random random = new Random();
    final StringBuilder sb = new StringBuilder(sizeOfRandomString);
    for (int i = 0; i < sizeOfRandomString; ++i) {
      sb.append(ALLOWED_CHARACTERS.charAt(random.nextInt(ALLOWED_CHARACTERS.length())));
    }
    return sb.toString();
  }

  public interface ProgressListener {
    void update(int progress);
    void error(Call call, String message);
    void done(String savePath);
  }
}
