package com.zoyi.channel.plugin.android.activity.userchat_list;

import android.content.Context;
import android.support.v7.util.SortedList;
import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.SortedListCallback;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.ChannelModel;
import com.zoyi.channel.plugin.android.model.rest.Manager;
import com.zoyi.channel.plugin.android.model.rest.ReferMessage;
import com.zoyi.channel.plugin.android.model.rest.Session;
import com.zoyi.channel.plugin.android.model.rest.UserChat;
import com.zoyi.channel.plugin.android.model.source.userchat.UserChatItem;
import com.zoyi.channel.plugin.android.model.wrapper.UserChatsWrapper;
import com.zoyi.channel.plugin.android.presenter.userchat.UserChatAdapterContract;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;

/**
 * Created by mika on 2016. 12. 6..
 */
class UserChatListAdapter extends RecyclerView.Adapter<UserChatHolder>
    implements UserChatAdapterContract.View, UserChatAdapterContract.Model {

  private OnUserChatClickListener clickListener;
  private OnUserChatCheckListener checkListener;

  private Map<String, UserChatItem> map;
  private UserChatListDataDictionary dictionary;
  private SortedList<UserChatItem> items;
  private Queue<String> checkedChatIds;
  private boolean refreshBySocket = false;
  private boolean editMode = false;

  UserChatListAdapter() {
    map = new HashMap<>();
    dictionary = new UserChatListDataDictionary();
    items = new SortedList<>(UserChatItem.class, new SortedListCallback(this, true));
    checkedChatIds = new LinkedList<>();
  }

  @Override
  public void setOnUserChatClickListener(OnUserChatClickListener clickListener) {
    this.clickListener = clickListener;
  }

  @Override
  public void setOnUserChatCheckListener(OnUserChatCheckListener checkListener) {
    this.checkListener = checkListener;
  }

  @Override
  public void setEditMode(boolean editMode) {
    this.editMode = editMode;

    notifyDataSetChanged();
  }

  @Override
  public void removeUserChat(UserChat userChat) {
    String userChatId = userChat.getId();

    if (map.containsKey(userChatId)) {
      items.remove(map.get(userChatId));
    }
  }

  @Override
  public void checkUserChat(String chatId) {
    if (checkedChatIds.contains(chatId)) {
      removeCheckedChatId(chatId);
    } else {
      addCheckedChatId(chatId);
    }
  }

  private void removeCheckedChatId(String chatId) {
    checkedChatIds.remove(chatId);

    if (map.containsKey(chatId)) {
      int position = items.indexOf(map.get(chatId));

      notifyItemChanged(position);
    }
  }

  private void addCheckedChatId(String chatId) {
    checkedChatIds.add(chatId);

    if (map.containsKey(chatId)) {
      int position = items.indexOf(map.get(chatId));

      notifyItemChanged(position);
    }
  }

  @Override
  public void init(UserChatsWrapper wrapper) {
    dictionary.add(wrapper.getManagers());
    dictionary.add(wrapper.getSessions());
    dictionary.add(wrapper.getMessages());

    List<UserChatItem> delta = new ArrayList<>();

    for (UserChat userChat : wrapper.getUserChats()) {
      UserChatItem item = new UserChatItem(userChat, dictionary);

      if (map.containsKey(userChat.getId())) {
        items.remove(map.get(userChat.getId()));
      }
      delta.add(item);
    }

    for (UserChatItem item : delta) {
      map.put(item.getId(), item);
    }
    items.clear();
    items.addAll(delta);

    refreshBySocket = true;
  }

  @Override
  public void stopRefreshBySocket() {
    this.refreshBySocket = false;
  }

  @Override
  public void clearCheckedChatId() {
    checkedChatIds.clear();
  }

  @Override
  public boolean hasCheckedChatIds() {
    return checkedChatIds.size() > 0;
  }

  @Override
  public String getCheckedChatId() {
    return checkedChatIds.poll();
  }

  @Override
  public UserChatHolder onCreateViewHolder(ViewGroup parent, int viewType) {
    Context c = parent.getContext();
    View view = LayoutInflater.from(c).inflate(R.layout.ch_plugin_item_user_chat, parent, false);

    return new UserChatHolder(view, clickListener, checkListener);
  }

  @Override
  public void onBindViewHolder(UserChatHolder holder, int position) {
    holder.setUserChatItem(
        items.get(position),
        checkedChatIds.contains(items.get(position).getId()),
        editMode);
  }

  @Override
  public int getItemCount() {
    return items.size();
  }

  @Override
  public void receiveChannelModel(ChannelModel channelModel, boolean upsert) {
    if (!refreshBySocket) {
      return;
    }
    if (upsert) {
      dictionary.add(channelModel);
    } else {
      dictionary.remove(channelModel);
    }

    switch (channelModel.getClass().getSimpleName()) {
      case UserChat.CLASSNAME:
        updateUserChat((UserChat) channelModel);
        break;

      case Session.CLASSNAME:
        updateSession((Session) channelModel);
        break;

      case Manager.CLASSNAME:
        updateManager(channelModel.getId());
        break;

      case ReferMessage.CLASSNAME:
        // do nothing
        break;
    }
  }

  private void updateUserChat(UserChat userChat) {
    UserChatItem item;
    String userChatId = userChat.getId();
    boolean closed = Const.USER_CHAT_STATE_CLOSED.equals(userChat.getState());

    if (map.containsKey(userChatId)) {
      item = map.get(userChatId);

      int position = items.indexOf(item);

      if (position >= 0) {
        if (closed) {
          items.removeItemAt(position);
        } else {
          item.setUserChat(userChat);
          items.updateItemAt(position, item);
        }
      }
    } else {
      item = new UserChatItem(userChat, dictionary);

      map.put(userChatId, item);

      items.add(item);
    }
  }

  private void updateManager(String managerId) {
    for (int i = 0; i < items.size(); i++) {
      UserChatItem item = items.get(i);

      if (item.containsManager(managerId)) {
        items.updateItemAt(i, item);
      }
    }
  }

  private void updateSession(Session session) {
    for (int i = 0; i < items.size(); i++) {
      UserChatItem item = items.get(i);

      if (item.getSessionId().equals(session.getId())) {
        item.setSession(session);
        items.updateItemAt(i, item);
        break;
      }
    }
  }
}
