package com.zhuge.analysis.stat;

import android.content.Context;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;
import android.util.Base64;

import com.zhuge.analysis.util.HttpServices;
import com.zhuge.analysis.util.Utils;
import com.zhuge.analysis.util.ZGJSONObject;
import com.zhuge.analysis.util.ZGLogger;

import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;

/**
 * 逻辑处理，事件分发
 * Created by Omen on 16/9/9.
 */
/*package*/ class ZGCore {

    private static final int MESSAGE_DEVICE_INFO = 0;
    private static final int MESSAGE_CHECK_SESSION = 1;
    private static final int MESSAGE_CUSTOM_EVENT = 2;
    private static final int MESSAGE_IDENTIFY_USER = 3;
    private static final int MESSAGE_FLUSH = 4;
    private static final int MESSAGE_NOTIFICATION = 5;
    private static final int MESSAGE_UPDATE_SESSION = 6;
    private static final int MESSAGE_START_TRACK = 7;
    private static final int MESSAGE_END_TRACK = 8;
    private static final int MESSAGE_SET_EVENT_INFO = 9;
    private static final int MESSAGE_SET_DEVICE_INFO = 10;

    private static final int CODE_NEED_FLUSH = 0;
    private ZGWorker  zgWorker;
    private Context   mContext;
    private ZGAppInfo appInfo;
    private final Map<String, Long> mEventTimings = new HashMap<>();
    private static final String TAG = "com.zhuge.ZGCore";
    int forgroundActivities = 0;

    /*package*/ ZGCore(ZGAppInfo appInfo){
        this.appInfo = appInfo;
        HandlerThread workerThread =  new HandlerThread("com.zhuge.worker");
        workerThread.start();
        zgWorker = new ZGWorker(workerThread.getLooper());
    }
    /**
     * 初始化设备信息，用户自定义字段
     * @param context 上下文
     */
    /*package*/ void init(Context context) {
        mContext = context.getApplicationContext();
        appInfo.initGlobalSettingFile(mContext);//初始化配置文件
        appInfo.initAppInfo(mContext);//初始化应用信息
        appInfo.upgradeSharedPrefs(mContext); //查看是否需要更新
        appInfo.initDeviceInfo(mContext);//上传设备id
        if (appInfo.did == null){
            ZGLogger.logError(TAG,"did生成失败。");
        }
        appInfo.logInitInfo();//输出初始化信息
        zgWorker.sendEmptyMessage(MESSAGE_DEVICE_INFO);//发送设备信息
    }

    /*package*/ void onEnterForeground(String name) {
        forgroundActivities++;
        Message message = zgWorker.obtainMessage(MESSAGE_CHECK_SESSION,name);
        message.sendToTarget();
    }

    /*package*/ void onExitForeground() {
        forgroundActivities--;
        zgWorker.sendEmptyMessage(MESSAGE_UPDATE_SESSION);
    }

    /*package*/ void startTrack(String event_name) {
        Message message = zgWorker.obtainMessage(MESSAGE_START_TRACK);
        message.obj = event_name;
        message.sendToTarget();
    }

    /*package*/ void endTrack(String event_name, JSONObject properties) {
        EventDescription eventDescription = new EventDescription(event_name,properties);
        Message message = zgWorker.obtainMessage(MESSAGE_END_TRACK);
        message.obj = eventDescription;
        message.sendToTarget();
    }

    /*package*/ void track(String eventName, JSONObject json) {
        EventDescription eventDescription = new EventDescription(eventName,json);
        Message message = zgWorker.obtainMessage(MESSAGE_CUSTOM_EVENT);
        message.obj = eventDescription;
        message.sendToTarget();
    }

    /*package*/ void identify(String uid, JSONObject json) {
        EventDescription eventDescription = new EventDescription(uid,json);
        Message message = zgWorker.obtainMessage(MESSAGE_IDENTIFY_USER);
        message.obj = eventDescription;
        message.sendToTarget();
    }

    /*package*/ void flush() {
        Message message = zgWorker.obtainMessage(MESSAGE_FLUSH);
        message.sendToTarget();
    }

    /*package*/ void sendNotification(ZGJSONObject json) {
        Message message = zgWorker.obtainMessage(MESSAGE_NOTIFICATION);
        message.obj = json;
        message.sendToTarget();
    }

    /*package*/ void userSetEventInfo(JSONObject object) {
        Message message = zgWorker.obtainMessage(MESSAGE_SET_EVENT_INFO);
        message.obj = object;
        message.sendToTarget();
    }

    /*package*/ void userSetDeviceInfo(JSONObject object) {
        Message message = zgWorker.obtainMessage(MESSAGE_SET_DEVICE_INFO);
        message.obj = object;
        message.sendToTarget();
    }

    public void onException(Thread thread, Throwable ex) {
        boolean isForgound = forgroundActivities>0;
        ZGJSONObject object = appInfo.buildException(thread, ex, isForgound);
        sendNotification(object);
    }

    private static class EventDescription {
        EventDescription(String eventName, JSONObject properties) {
            this.eventName = eventName;
            this.properties = properties;
        }

        private final String eventName;
        private final JSONObject properties;
    }


    private class ZGWorker extends Handler{
        private ZhugeDbAdapter dbAdapter;
        private HttpServices httpService;
        private long localEventSize = 0;
        private long today_send_count = 0;
        public ZGWorker(Looper looper){
            super(looper);
        }

        @Override
        public void handleMessage(Message msg) {
            if (mContext == null){
                ZGLogger.logError(TAG,"未正确初始化，请在应用入口调用ZhugeSDK.getInstance().init();");
                return;
            }
            if (dbAdapter == null){
                dbAdapter = new ZhugeDbAdapter(mContext);
                localEventSize = dbAdapter.getEventCount();
                String todayInfo = appInfo.getGlobalSP().getString(Constants.SP_TODAY_COUNT, "");
                if (!"".equals(todayInfo)) {
                    String[] toInfo = todayInfo.split("\\|");
                    if ((System.currentTimeMillis() / 1000 / 86400 - Integer.parseInt(toInfo[0])) != 0) {
                        today_send_count = 0;
                    } else {
                        today_send_count = Integer.parseInt(toInfo[1]);
                    }
                }
            }
            int stateCode = -1;
            switch (msg.what){
                case MESSAGE_DEVICE_INFO:
                    ZGJSONObject deviceInfo = appInfo.buildDeviceInfo(mContext);
                    if (deviceInfo == null){
                        break;
                    }
                    addEvent(deviceInfo);
                    updateDeviceInfoTime();
                    stateCode = CODE_NEED_FLUSH;
                    break;
                case MESSAGE_CHECK_SESSION:
                    String sessionName = msg.obj.toString();
                    stateCode = startNewSessionIfNeed(sessionName);
                    break;
                case MESSAGE_CUSTOM_EVENT:
                    EventDescription description = (EventDescription) msg.obj;
                    String name = description.eventName;
                    JSONObject pro = description.properties;
                    if (appInfo.sessionID<0 && appInfo.isInMainThread){
                        startNewSessionIfNeed("et_in_main_thread");
                    }
                    ZGJSONObject info = appInfo.buildCustomEvent(name,pro);
                    updateSessionActivity("自定义事件更新会话");
                    stateCode = addEvent(info);
                    break;
                case MESSAGE_IDENTIFY_USER:
                    EventDescription userDes = (EventDescription) msg.obj;
                    String uid = userDes.eventName;
                    JSONObject uPro = userDes.properties;
                    ZGJSONObject userPro = appInfo.buildIdentify(uid,uPro);
                    updateCUID(uid);
                    updateSessionActivity("标记用户更新事件");
                    addEvent(userPro);
                    stateCode = CODE_NEED_FLUSH;
                    break;
                case MESSAGE_FLUSH:
                    int i = flushEvent();
                    stateCode = i == 0? -1: i;
                    break;
                case MESSAGE_NOTIFICATION:
                    ZGJSONObject data = (ZGJSONObject) msg.obj;
                    addEvent(data);
//                    stateCode = CODE_NEED_FLUSH;
                    flushEvent();
                    break;
                case MESSAGE_UPDATE_SESSION:
                    updateSessionActivity("退出前台，更新会话时间");
                    break;
                case MESSAGE_START_TRACK:
                    String event = (String) msg.obj;
                    mEventTimings.put(event,System.currentTimeMillis());
                    break;
                case MESSAGE_END_TRACK:
                    EventDescription obj = (EventDescription) msg.obj;
                    if (appInfo.sessionID<0 && appInfo.isInMainThread){
                        startNewSessionIfNeed("et_in_main_thread");
                    }
                    stateCode = endTrack(obj);
                    break;
                case MESSAGE_SET_EVENT_INFO:
                    JSONObject set = (JSONObject) msg.obj;
                    ZGLogger.logVerbose("设置事件全局信息:"+set.toString());
                    appInfo.getGlobalSP().edit().putString(Constants.SP_USER_DEFINE_EVENT,set.toString()).apply();
                    break;
                case MESSAGE_SET_DEVICE_INFO:
                    JSONObject device = (JSONObject) msg.obj;
                    ZGLogger.logVerbose("设置自定义设备环境:"+device.toString());
                    appInfo.getGlobalSP().edit().putString(Constants.SP_USER_DEFINE_DEVICE,device.toString()).apply();
                    break;
            }
            if ((stateCode == CODE_NEED_FLUSH || stateCode >=Constants.UPLOAD_LIMIT_SIZE) && !hasMessages(MESSAGE_FLUSH)){
                sendEmptyMessageDelayed(MESSAGE_FLUSH,Constants.FLUSH_INTERVAL);
            }
        }

        private void updateDeviceInfoTime() {
            appInfo.getGlobalSP().edit().putLong(Constants.SP_UPDATE_DEVICE_TIME,System.currentTimeMillis()).apply();
        }

        private void updateCUID(String uid) {
            appInfo.getGlobalSP().edit().putString(Constants.SP_CUID,uid).apply();
        }

        private int endTrack(EventDescription description) {
            String event_name = description.eventName;
            JSONObject properties = description.properties;
            Long begin = mEventTimings.get(event_name);
            if (null == begin)
                return -1;
            mEventTimings.remove(event_name);
            long now = System.currentTimeMillis();
            long dru = now - begin;
            try {
                if (properties == null){
                    JSONObject obj = new JSONObject();
                    obj.put("$dru",dru);
                    ZGJSONObject zgjsonObject = appInfo.buildCustomEvent(event_name, obj);
                    return addEvent(zgjsonObject);
                }else {
                    properties.put("$dru",dru);
                    ZGJSONObject zgjsonObject = appInfo.buildCustomEvent(event_name,properties);
                    return addEvent(zgjsonObject);
                }
            }catch (Exception e){
                ZGLogger.handleException(TAG,"时长追踪事件错误",e);
                return -1;
            }
        }

        private int startNewSessionIfNeed(String name) {
            long timestamp = System.currentTimeMillis();
            if (appInfo.sessionID>0){
                if (timestamp - appInfo.lastSessionActivityTime < Constants.SESSION_EXCEED){
                    updateSessionActivity("session ID>0");
                    ZGLogger.logVerbose("已经初始化，更新会话时间");
                    return -1;
                }
                startSession(name);
                ZGLogger.logVerbose("已经初始化，开始新的会话");
                return  CODE_NEED_FLUSH;
            }else {
                String lastSession = appInfo.getGlobalSP().getString(Constants.SP_LAST_SESSION_TIME, "");
                if (!lastSession.equals("")){
                    String[] last = lastSession.split("\\|");
                    long session = Long.parseLong(last[0]);
                    long lastTime = Long.parseLong(last[1]);
                    if (session <=0 ||timestamp - lastTime >Constants.SESSION_EXCEED){
                        ZGLogger.logVerbose("第一次进入，距离上次超时，开始新的会话");
                        startSession(name);
                        return CODE_NEED_FLUSH;
                    }else {
                        //继承上次的会话ID
                        ZGLogger.logVerbose("第一次进入，继承上次会话");
                        appInfo.sessionID = session;
                        updateSessionActivity("继承上次会话");
                        return -1;
                    }
                }else {
                    ZGLogger.logVerbose("第一次进入，没有上次，开始新的会话");
                    startSession(name);
                    return CODE_NEED_FLUSH;
                }
            }
        }

        private void startSession(String name) {
            appInfo.sessionID = System.currentTimeMillis();
            if (Constants.ENABLE_SESSION_TRACK){
                completeLastSession();
                ZGJSONObject st = appInfo.buildSessionStart(name);
                if (null == st)
                    return;
                addEvent(st);
            }
            updateSessionActivity("会话开始，更新会话时间");
        }

        private void completeLastSession() {
            ZGJSONObject se = appInfo.buildSessionEnd();
            if (null == se)
                return;
            addEvent(se);
        }

        private void updateSessionActivity(String s) {
            ZGLogger.logVerbose("updateSessionActivity "+s);
            long now = System.currentTimeMillis();
            appInfo.lastSessionActivityTime = now;
            String updateSession = appInfo.sessionID + "|" + now;
            appInfo.getGlobalSP().edit().putString(Constants.SP_LAST_SESSION_TIME, updateSession).apply();
        }

        private int addEvent(ZGJSONObject event){
            if (localEventSize >= Constants.MAX_LOCAL_SIZE){
                ZGLogger.logError(TAG,"本地存储事件超过最大值，事件将被丢弃。");
                return -1;
            }
            if (null == event){
                return -1;
            }
            ZGLogger.logVerbose("添加事件\n"+event.toString());
            int count =  dbAdapter.addEvent(event);
            localEventSize = count;
            if (appInfo.debug){
                return Constants.UPLOAD_LIMIT_SIZE;//实时调试开启之后，要立即上传事件
            }
            return count;
        }

        private int flushEvent() {
            if (!appInfo.connectivityUtils.isOnline()){
                ZGLogger.logMessage(TAG,"网络不可用，暂停发送。");
                sendEmptyMessageDelayed(MESSAGE_FLUSH,30*1000);//30秒后重试
                return -1;
            }
            if (today_send_count >= Constants.SEND_SIZE){
                ZGLogger.logMessage(TAG,"当日已达最大上传数，暂停发送事件。");
                return -1;
            }
            if (null == httpService){
                httpService = new HttpServices();
            }
            String[] mDbData = dbAdapter.getData();
            if (null == mDbData){
                return -1;
            }
            try {
                final Map<String, Object> postMap = new HashMap<>();
                JSONObject postData = appInfo.wrapData(mDbData[1]);
                String data = Base64.encodeToString(Utils.compress(postData.toString().getBytes("UTF-8")),
                        Base64.DEFAULT).replace("\r", "").replace("\n", "");
                postMap.put("method", "event_statis_srv.upload");
                postMap.put("compress", "1");
                postMap.put("event", data);
                String url , backUrl;
                if (appInfo.api_path != null){
                    url = appInfo.api_path;
                    backUrl = appInfo.api_path_back;
                }else {
                    url = Constants.API_PATH;
                    backUrl = Constants.API_PATH_BACKUP;
                }
                byte[] returnByte = httpService.requestApi(url, backUrl,postMap);
                if (returnByte == null){
                    ZGLogger.logMessage(TAG,"发送失败，未获得服务端返回数据。");
                    return -1;
                }
                String s = new String(returnByte, "UTF-8");
                JSONObject responseDict = new JSONObject(s);
                int return_code = responseDict.optInt("return_code");
                if (return_code == 0){
                    int count = Integer.parseInt(mDbData[2]);
                    String id = mDbData[0];
                    today_send_count += count;
                    localEventSize -= count;
                    dbAdapter.removeEvent(id);
                    updateTodayCount();
                    ZGLogger.logMessage(TAG,"发送成功，今日已发送"+today_send_count+"条数据。");
                }else {
                    ZGLogger.logMessage(TAG,"发送失败，返回信息："+responseDict.toString());
                }
                return (int) localEventSize;
            } catch (Exception e) {
                ZGLogger.handleException(TAG,"发送数据出错。",e);
                return -1;
            }
        }

        private void updateTodayCount() {
            long day = System.currentTimeMillis()/1000/86400;
            String value = day+"|"+today_send_count;
            appInfo.getGlobalSP().edit().putString(Constants.SP_TODAY_COUNT,value).apply();
        }
    }
}
