package com.zhuge.analysis.stat;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteException;
import android.database.sqlite.SQLiteOpenHelper;
import android.database.sqlite.SQLiteStatement;

import com.zhuge.analysis.util.ZGLogger;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.File;

/**
 * SQL Lite Database Adapter for Zhuge.
 * Created by Omen on 15/11/17.
 */
/*package*/ class ZhugeDbAdapter {

    private static final String TAG = "Zhuge.Database";
    private static final String TABLE_NAME = "events";

    public static final String KEY_DATA = "data";
    public static final String KEY_CREATED_AT = "created_at";
    private static final String DATABASE_NAME = "zhuge";
    private static final int DATABASE_VERSION = 1;


    private static final String CREATE_EVENTS_TABLE =
            "CREATE TABLE " + TABLE_NAME + " (_id INTEGER PRIMARY KEY AUTOINCREMENT, " +
                    KEY_DATA + " TEXT NOT NULL, " +
                    KEY_CREATED_AT + " INTEGER NOT NULL);";
    private static final String EVENTS_TIME_INDEX =
            "CREATE INDEX IF NOT EXISTS time_idx ON " + TABLE_NAME +
                    " (" + KEY_CREATED_AT + ");";



    private final ZhugeDbHelper mDbHelper;


    /*package*/ ZhugeDbAdapter(Context context){
        mDbHelper = new ZhugeDbHelper(context,DATABASE_NAME);
    }

    /**
     *
     * @param jsonObject 要添加的事件信息
     * @return 当前database中的记录数
     */
    /*package*/ int addEvent(JSONObject jsonObject){
        Cursor cursor = null;
        int count=-1;
        try {
            SQLiteDatabase db = mDbHelper.getWritableDatabase();
            ContentValues contentValues = new ContentValues();
            contentValues.put(KEY_DATA,jsonObject.toString());
            contentValues.put(KEY_CREATED_AT, System.currentTimeMillis());
            db.insert(TABLE_NAME, null, contentValues);

            cursor = db.rawQuery("SELECT COUNT(*) FROM "+TABLE_NAME,null);
            cursor.moveToFirst();
            count = cursor.getInt(0);
        }catch (Exception  e){
            ZGLogger.handleException(TAG,"向zhuge数据库中写入数据时出错，重新初始化zhuge数据库。",e);
            mDbHelper.deleteDatabase();
        }finally {
            if (cursor!=null){
                cursor.close();
            }
            mDbHelper.close();
        }
        return count;
    }

    /**
     * 从数据库按照发生时间获取最近的50条事件。
     * @return String[3]的数组，第一个是这次取得事件的最后一个索引，第二个为事件数据，第三个为这次取得的事件数。
     */
    /*package*/ String[] getData(){
        Cursor cursor = null;
        String data = null;
        String last_id = null; //当次获得事件的索引
        int size = 0;//这次上传的事件数
        try {
            SQLiteDatabase db = mDbHelper.getReadableDatabase();
            cursor = db.rawQuery("SELECT * FROM "+TABLE_NAME+" ORDER BY "+KEY_CREATED_AT +
                    " ASC LIMIT 50",null);
            JSONArray array = new JSONArray();
            while (cursor.moveToNext()){
                if (cursor.isLast()){
                    last_id = cursor.getString(cursor.getColumnIndex("_id"));
                }
                String s = cursor.getString(cursor.getColumnIndex(KEY_DATA));
                JSONObject jsonObject = new JSONObject(s);
                array.put(jsonObject);
                size++;
            }
            if (array.length()>0){
                data = array.toString();
            }
        }catch (Exception e){
            ZGLogger.handleException(TAG,"无法从zhuge数据库中读取数据。",e);
            //在没有对DB进行修改的情况下出现的错误，我们可以假定他可以自己恢复。在写操作时，出现错误，则删除文件。
            last_id = null;
            data = null;
        }finally {
          mDbHelper.close();
            if (cursor != null){
                cursor.close();
            }
        }
        if (data!=null && last_id!=null){
            return new String[]{last_id,data,Integer.toString(size)};
        }
        return null;
    }

    /**
     * 从数据库中移除数据
     * @param lastID 移除lastID之前的数据
     */
    /*package*/ void removeEvent(String lastID){
        try {
            final SQLiteDatabase db = mDbHelper.getWritableDatabase();
            db.delete(TABLE_NAME, "_id <= " + lastID, null);
        } catch (final SQLiteException e) {
            ZGLogger.handleException(TAG,"无法从zhuge数据库中删除数据，重新初始化数据库。",e);
            // 修改SQL出错的情况下，删除DB。
            mDbHelper.deleteDatabase();
        } finally {
            mDbHelper.close();
        }
    }

    /*package*/ long getEventCount() {
        long numberRows = 0;
        SQLiteStatement statement = null;
        try {
            SQLiteDatabase db = mDbHelper.getReadableDatabase();
            String query = "SELECT COUNT(*) FROM " + TABLE_NAME;
            statement = db.compileStatement(query);
            numberRows = statement.simpleQueryForLong();
        } catch (SQLiteException e) {
            ZGLogger.handleException(TAG,"查询事件数时出错。",e);
        } finally {
            if (statement != null) {
                statement.close();
            }
            mDbHelper.close();
        }
        return numberRows;
    }

    private static class ZhugeDbHelper extends SQLiteOpenHelper{
        private final File mFile;


        public ZhugeDbHelper(Context context, String name) {
            super(context, name, null, DATABASE_VERSION);
            mFile = context.getDatabasePath(DATABASE_NAME);
        }

        @Override
        public void onCreate(SQLiteDatabase db) {
            db.execSQL(CREATE_EVENTS_TABLE);
            db.execSQL(EVENTS_TIME_INDEX);
            ZGLogger.logMessage(TAG,"create zhuge database");
        }

        @Override
        public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
            db.execSQL("DROP TABLE IF EXISTS " + TABLE_NAME);
            db.execSQL(CREATE_EVENTS_TABLE);
            db.execSQL(EVENTS_TIME_INDEX);
            ZGLogger.logMessage(TAG,"upgrade zhuge database");

        }

        /*package*/ void deleteDatabase() {
            close();
            mFile.delete();
        }
    }
}
