package com.zhuge.analysis.stat;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteException;
import android.database.sqlite.SQLiteOpenHelper;
import android.util.Log;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.File;

/**
 * SQL Lite Database Adapter for Zhuge.
 * Created by jiaokang on 15/11/17.
 */
/*package*/ class ZhugeDbAdapter {

    private static final String LOGTAG = "Zhuge.Database";
    private static final String TABLE_NAME = "events";

    public static final String KEY_DATA = "data";
    public static final String KEY_CREATED_AT = "created_at";
    private static final String DATABASE_NAME = "zhuge";
    private static final int DATABASE_VERSION = 1;


    private static final String CREATE_EVENTS_TABLE =
            "CREATE TABLE " + TABLE_NAME + " (_id INTEGER PRIMARY KEY AUTOINCREMENT, " +
                    KEY_DATA + " TEXT NOT NULL, " +
                    KEY_CREATED_AT + " INTEGER NOT NULL);";
    private static final String EVENTS_TIME_INDEX =
            "CREATE INDEX IF NOT EXISTS time_idx ON " + TABLE_NAME +
                    " (" + KEY_CREATED_AT + ");";



    private final ZhugeDbHelper mDbHelper;


    /*package*/ ZhugeDbAdapter(Context context){
        mDbHelper = new ZhugeDbHelper(context,DATABASE_NAME);
    }

    /**
     *
     * @param jsonObject 要添加的事件信息
     * @return 当前database中的记录数
     */
    /*package*/ int addEvent(JSONObject jsonObject){
        Cursor cursor = null;
        int count=-1;
        try {
            SQLiteDatabase db = mDbHelper.getWritableDatabase();
            ContentValues contentValues = new ContentValues();
            contentValues.put(KEY_DATA,jsonObject.toString());
            contentValues.put(KEY_CREATED_AT, System.currentTimeMillis());
            db.insert(TABLE_NAME, null, contentValues);

            cursor = db.rawQuery("SELECT COUNT(*) FROM "+TABLE_NAME,null);
            cursor.moveToFirst();
            count = cursor.getInt(0);
        }catch (Exception  e){
            Log.e(LOGTAG,"Zhuge 向数据库插入数据出错"+e.getMessage());
            mDbHelper.deleteDatabase();
        }finally {
            if (cursor!=null){
                cursor.close();
            }
            mDbHelper.close();
        }
        return count;
    }

    /*package*/ String[] getData(){
        Cursor cursor = null;
        String data = null;
        String last_id = null;
        int size = 0;
        try {
            SQLiteDatabase db = mDbHelper.getReadableDatabase();
            cursor = db.rawQuery("SELECT * FROM "+TABLE_NAME+" ORDER BY "+KEY_CREATED_AT +
                    " ASC LIMIT 50",null);
            JSONArray array = new JSONArray();
            while (cursor.moveToNext()){
                if (cursor.isLast()){
                    last_id = cursor.getString(cursor.getColumnIndex("_id"));
                }
                String s = cursor.getString(cursor.getColumnIndex(KEY_DATA));
                JSONObject jsonObject = new JSONObject(s);
                array.put(jsonObject);
                size++;
            }
            if (array.length()>0){
                data = array.toString();
            }
        }catch (Exception e){
            Log.e(LOGTAG, "Could not pull records for Zhuge out of database " + TABLE_NAME + ". Waiting to send.", e);

            // We'll dump the DB on write failures, but with reads we can
            // let things ride in hopes the issue clears up.
            // (A bit more likely, since we're opening the DB for read and not write.)
            // A corrupted or disk-full DB will be cleaned up on the next write or clear call.
            last_id = null;
            data = null;
        }finally {
          mDbHelper.close();
            if (cursor != null){
                cursor.close();
            }
        }
        if (data!=null && last_id!=null){
            return new String[]{last_id,data,size+""};
        }
        return null;
    }

    /**
     * 从数据库中移除数据
     * @param lastID 移除lastID之前的数据
     */
    /*package*/ void removeEvent(String lastID){
        try {
            final SQLiteDatabase db = mDbHelper.getWritableDatabase();
            db.delete(TABLE_NAME, "_id <= " + lastID, null);
        } catch (final SQLiteException e) {
            Log.e(LOGTAG, "Could not clean sent Zhuge records from " + TABLE_NAME + ". Re-initializing database.", e);

            // We assume that in general, the results of a SQL exception are
            // unrecoverable, and could be associated with an oversized or
            // otherwise unusable DB. Better to bomb it and get back on track
            // than to leave it junked up (and maybe filling up the disk.)
            mDbHelper.deleteDatabase();
        } finally {
            mDbHelper.close();
        }
    }
    private static class ZhugeDbHelper extends SQLiteOpenHelper{
        private final File mFile;


        public ZhugeDbHelper(Context context, String name) {
            super(context, name, null, DATABASE_VERSION);
            mFile = context.getDatabasePath(DATABASE_NAME);
        }

        @Override
        public void onCreate(SQLiteDatabase db) {
            db.execSQL(CREATE_EVENTS_TABLE);
            db.execSQL(EVENTS_TIME_INDEX);
            Log.v(LOGTAG,"create zhuge database.");
        }

        @Override
        public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
            db.execSQL("DROP TABLE IF EXISTS " + TABLE_NAME);
            db.execSQL(CREATE_EVENTS_TABLE);
            db.execSQL(EVENTS_TIME_INDEX);
            Log.v(LOGTAG, "upgrade zhuge database.");

        }

        /*package*/ void deleteDatabase() {
            close();
            mFile.delete();
        }
    }
}
