package com.zhuge.analysis.util_svs;

import android.content.Context;
import android.content.SharedPreferences;
import android.util.Base64;

import com.zhuge.analysis.stat.ZhugeConfig;
import com.zhuge.analysis.stat.ZhugeSDK;
import com.zhuge.analysis.sys_svs.ActivityServicesUtils;
import com.zhuge.analysis.sys_svs.ConnectivityUtils;
import com.zhuge.analysis.sys_svs.TimeUtils;

import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.params.CoreConnectionPNames;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.zip.Deflater;

/**
 * Event 管理类 Created by kongmiao on 14-10-13.
 */
public class EventStore {

    private SharedPreferences zhugeSettings;

    private SharedPreferences getZhugeSettings() {
        return zhugeSettings;
    }

    private static final String EVENT_KEY = "events";
    private static final String TOTALDAY_KEY = "total";
    private static final String DELIMITER = "===";
    private static final String REQUEST_TIME = "rTime";
    private static ExecutorService executor_;
    private static ExecutorService debugExecutor;
    public ScheduledExecutorService timerService_;
    private ScheduledFuture timerFuture;
    private static ActivityServicesUtils activityServicesUtils;
    private static ConnectivityUtils connectivityUtils;

    private int todayTotal = 0;
    private static final int ALLEVENTS = 0;
    private static final int SPLITEVENTS = 1;

    private synchronized void ensureExecutor() {
        if (executor_ == null) {
            executor_ = Executors.newSingleThreadExecutor();
        }
    }

    private synchronized void ensureDebugExecutor() {
        if (debugExecutor == null) {
            debugExecutor = Executors.newCachedThreadPool();
        }
    }

    /**
     * 获取当前事件数
     * 
     * @return 事件数
     */
    public int getTodayTotal() {
        return todayTotal;
    }

    public EventStore(final Context context, String pf, String spName) {
        if (null == connectivityUtils )
            connectivityUtils = new ConnectivityUtils(context);
        zhugeSettings = context.getSharedPreferences(spName, Context.MODE_PRIVATE);
        startUploadService(context);
        String todayInfo = getZhugeSettings().getString(TOTALDAY_KEY, "");
        if ("".equals(todayInfo)) {
            todayTotal = 0;
        } else {
            String[] tiInfo = todayInfo.split("\\|");
            if (System.currentTimeMillis() / 1000 / 86400 - Integer.parseInt(tiInfo[0]) > 0) {
                todayTotal = 0;
            } else {
                todayTotal = Integer.parseInt(tiInfo[1]);
            }
        }
    }

    public boolean stopTimeService() {
        try {
            timerFuture.cancel(false);
            timerService_.shutdown();
        } catch (Exception e) {
            ZhugeSDK.getInstance().getConfig().debug("线程池关闭失败");
        }
        return false;
    }

    private synchronized void updateEventCount() {
        getZhugeSettings().edit().putString(TOTALDAY_KEY, (System.currentTimeMillis() / 1000 / 86400) + "|" + todayTotal).commit();
    }

    /**
     * 存储当次请求用时
     * @param time 请求用时
     */
    private synchronized void saveRequestTime(long time){
        getZhugeSettings().edit().putLong(REQUEST_TIME, time).apply();
    }

    public synchronized void addEvent(ZGJSONObject event) {
        final List<ZGJSONObject> events = eventsList(ALLEVENTS);
        events.add(event);
        getZhugeSettings().edit().putString(EVENT_KEY, joinEvents(events, DELIMITER)).commit();
    }

    private String[] events(int state) {
        final String joinedEventsStr = getZhugeSettings().getString(EVENT_KEY, "");
        String[] allEvents =  joinedEventsStr.length() == 0 ? new String[0] : joinedEventsStr.split(DELIMITER);
        switch (state){
            case ALLEVENTS:
                return allEvents;
            case SPLITEVENTS:
                if (allEvents.length >100){
                    String[] events = new String[100];
                    System.arraycopy(allEvents, 0, events, 0, 100);
                    return events;
                }else {
                    return allEvents;
                }
            default:
                return allEvents;
        }
    }

    public synchronized void removeEvents(final Collection<ZGJSONObject> eventsToRemove) {
        if (eventsToRemove != null && eventsToRemove.size() > 0) {
            final List<ZGJSONObject> events = eventsList(ALLEVENTS);
            if (null == events)
                return;
            if (events.removeAll(eventsToRemove)) {
                getZhugeSettings().edit().putString(EVENT_KEY, joinEvents(events, DELIMITER)).commit();
                ZhugeSDK.getInstance().getConfig().debug("本地数据删除成功");
            }
        }
    }

    public  List<ZGJSONObject> eventsList(int states) {
        final String[] array = events(states);
        final List<ZGJSONObject> events = new ArrayList<ZGJSONObject>(array.length);
        for (String s : array) {
            try {
                final ZGJSONObject event = new ZGJSONObject(s);
                events.add(event);
            } catch (JSONException ignored) {
                // should not happen since JSONObject is being constructed from
                // previously stringified JSONObject
                // events -> json objects -> json strings -> storage -> json
                // strings -> here
            }
        }
        // order the events from least to most recent
        Collections.sort(events, new Comparator<JSONObject>() {
            @Override
            public int compare(final JSONObject e1, final JSONObject e2) {
                Long delta = (e1.optLong("ts", -1) - e2.optLong("ts", -1));
                return delta.intValue();
            }
        });
        return events;
    }

    /**
     * Converts a collection of Event objects to URL-encoded JSON to a string,
     * with each event JSON string delimited by the specified delimiter.
     *
     * @param collection
     *            events to join into a delimited string
     * @param delimiter
     *            delimiter to use, should not be something that can be found in
     *            URL-encoded JSON string
     */
    static String joinEvents(final Collection<JSONObject> collection, final String delimiter) {
        final List<String> strings = new ArrayList<String>();
        for (JSONObject e : collection) {
            strings.add(e.toString());
        }
        return join(strings, delimiter);
    }

    static String joinEvents(final List<ZGJSONObject> collection, final String delimiter) {
        final List<String> strings = new ArrayList<String>();
        for (JSONObject e : collection) {
            strings.add(e.toString());
        }
        return join(strings, delimiter);
    }

    /**
     * Joins all the strings in the specified collection into a single string
     * with the specified delimiter.
     */
    static String join(final Collection<String> collection, final String delimiter) {
        final StringBuilder builder = new StringBuilder();
        int i = 0;
        for (String s : collection) {
            builder.append(s);
            if (++i < collection.size()) {
                builder.append(delimiter);
            }
        }
        String s;
        try {
            s = builder.toString();
        } catch (OutOfMemoryError e) {
            e.printStackTrace();
            return null;
        }
        return s;
    }

    // for unit testing
    synchronized void clear() {
        final SharedPreferences.Editor prefsEditor = getZhugeSettings().edit();
        prefsEditor.remove(EVENT_KEY);
        prefsEditor.commit();
    }

    public String getEventsString() {
        return getZhugeSettings().getString(EVENT_KEY, "");
    }

    public synchronized void startUploadService(final Context context) {
        if (timerService_ == null) {
            timerService_ = Executors.newSingleThreadScheduledExecutor();
        }
        timerFuture = timerService_.scheduleWithFixedDelay(new Runnable() {
            @Override
            public void run() {
                if (activityServicesUtils == null) {
                    activityServicesUtils = new ActivityServicesUtils(context);
                }
                String presentPackage = activityServicesUtils.getForegroundPackage();
                if (!presentPackage.contains(context.getPackageName())) {
                    try {
                        ZhugeSDK.getInstance().getConfig().debug("会话暂停");
                        Thread.sleep(20000);
                        presentPackage = activityServicesUtils.getForegroundPackage();
                        if (!presentPackage.contains(context.getPackageName())) {
                            ZhugeSDK.getInstance().getConfig().completeLastSession(context, 0);
                            send(true);
                            ZhugeSDK.getInstance().getConfig().debug("会话结束");
                            timerService_.schedule(new Runnable() {
                                @Override
                                public void run() {
                                    stopTimeService();
                                }
                            }, 1, TimeUnit.SECONDS);
                        } else {
                            ZhugeSDK.getInstance().getConfig().debug("会话继续");
                            ZhugeSDK.getInstance().updateSessionEnd();
                            send(false);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                } else {
                    ZhugeSDK.getInstance().updateSessionEnd();
                    send(false);
                }
            }
        }, ZhugeSDK.getInstance().getConfig().getPeriodTime(), ZhugeSDK.getInstance().getConfig().getPeriodTime(), TimeUnit.SECONDS);
    }

    public void send(final boolean flush) {
        try {
            ensureExecutor();
            executor_.submit(new Runnable() {
                @Override
                public void run() {
                    try {
                        if (connectivityUtils.getNetworkType() == -100){
                            ZhugeSDK.getInstance().getConfig().debug("网络状态不佳" + "|" + connectivityUtils.getNetworkType());
                            return;
                        }
                        List<ZGJSONObject> eventList = eventsList(SPLITEVENTS);
                        if (eventList.size() <= 0) {
                            return;
                        }
                        if (todayTotal > ZhugeSDK.getInstance().getConfig().getUpload_per_day()) {
                            ZhugeSDK.getInstance().getConfig().debug("数据超过每日上传的限制：" + todayTotal);
                            return;
                        }
                        if (!flush && eventList.size() < ZhugeSDK.getInstance().getConfig().getLocalMax()) {
                            return;
                        }
                        JSONObject jsonObject = sendData(eventList);
                        boolean send_status = requestAPI(jsonObject);
                        if (send_status) {
                            ZhugeSDK.getInstance().getConfig().debug("上传数据成功");
                            todayTotal += eventList.size();
                            if (!ZhugeSDK.getInstance().getConfig().isDebug()) {
                                updateEventCount();
                            }
                            removeEvents(eventList);
                        }
                    } catch (Exception e) {
                        ZhugeSDK.getInstance().getConfig().debug("发送数据线程出错");
                    }
                }
            });
        } catch (Exception e) {
            ZhugeSDK.getInstance().getConfig().debug("上传过程出错");
        }
    }

    private synchronized JSONObject sendData( List<ZGJSONObject> events) {
        try {
            JSONObject postData = new JSONObject();
            ZhugeConfig tmpConfig = ZhugeSDK.getInstance().getConfig();
            postData.put("ak", tmpConfig.getAppkey());
            postData.put("sdkv", ZhugeConfig.SDK_V);
            postData.put("sdk", "and");
            postData.put("did", tmpConfig.getDid());
            postData.put("cn", tmpConfig.getChannel());
            long lTime = getZhugeSettings().getLong(REQUEST_TIME,0);
            if (lTime != 0){
                postData.put("lt",lTime);
            }
            if (ZhugeSDK.getInstance().getConfig().isDebug()) {
                postData.put("debug", 1);
            }
            String cuid = getZhugeSettings().getString(ZhugeConfig.CUID_KEY, null);
            if (cuid != null) {
                postData.put("cuid", cuid);
            }
            long now_t = System.currentTimeMillis();
            postData.put("ts", TimeUtils.formatTime(now_t));
            postData.put("type", "statis");
            JSONArray jsonArray = new JSONArray();
            for (ZGJSONObject zgjsonObject : events) {
                jsonArray.put(zgjsonObject);
            }
            postData.put("data", jsonArray);
            return postData;
        } catch (Exception e) {
            ZhugeSDK.getInstance().getConfig().debug("数据组装出错" + e.getMessage());
        }
        return null;
    }

    private boolean requestAPI(JSONObject postData) {
        try {
            String path = "http://apipool.37degree.com/APIPOOL/";
            HttpClient client = new DefaultHttpClient();
            client.getParams().setParameter(CoreConnectionPNames.CONNECTION_TIMEOUT, ZhugeSDK.getInstance().getConfig().getConnectTimeout());
            client.getParams().setParameter(CoreConnectionPNames.SO_TIMEOUT, ZhugeSDK.getInstance().getConfig().getReadTimeout());
            HttpPost httpPost = new HttpPost(path);
            List<NameValuePair> params = new ArrayList<>();
            params.add(new BasicNameValuePair("method", "event_statis_srv.upload"));
            params.add(new BasicNameValuePair("compress", "1"));
            params.add(new BasicNameValuePair("event", Base64.encodeToString(compress(postData.toString().getBytes()), Base64.DEFAULT).replace("\r", "")
                    .replace("\n", "")));
            httpPost.setEntity(new UrlEncodedFormEntity(params, "utf-8"));
            long befor = System.currentTimeMillis();
            HttpResponse response = client.execute(httpPost);
            long resp = System.currentTimeMillis();
            saveRequestTime(resp - befor);
            if (200 == response.getStatusLine().getStatusCode()) {
                BufferedInputStream responseStream = new BufferedInputStream(response.getEntity().getContent());
                ByteArrayOutputStream responseData = new ByteArrayOutputStream(1024);
                int c;
                while ((c = responseStream.read()) != -1) {
                    responseData.write(c);
                }
                JSONObject responseDict = new JSONObject(responseData.toString("UTF-8"));
                boolean success = responseDict.optInt("return_code") == 0;
                if (success) {
                    return true;
                } else {
                    ZhugeSDK.getInstance().getConfig().debug("请求服务器失败，服务器响应码" + responseDict.optInt("return_code"));
                }
            }
        } catch (Exception e) {
            ZhugeSDK.getInstance().getConfig().debug("连接服务器出错" + e);
        }
        return false;
    }

    public byte[] compress(byte[] data) {
        byte[] output = new byte[0];

        Deflater compresser = new Deflater();
        compresser.reset();
        compresser.setInput(data);
        compresser.finish();
        ByteArrayOutputStream bos = new ByteArrayOutputStream(data.length);
        try {
            int dataLen = data.length;
            byte[] buf = new byte[dataLen + 1];
            while (!compresser.finished()) {
                int i = compresser.deflate(buf);
                bos.write(buf, 0, i);
            }
            output = bos.toByteArray();
        } catch (Exception e) {
            output = data;
            e.printStackTrace();
        } finally {
            try {
                bos.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        compresser.end();
        return output;
    }

    public void sendDataImmediately(final ZGJSONObject data) {
        final List<ZGJSONObject> eventList = new ArrayList<>(1);
        eventList.add(data);
        ensureDebugExecutor();
        debugExecutor.submit(new Runnable() {
            @Override
            public void run() {
                JSONObject jsonObject = sendData(eventList);
                boolean send_status = requestAPI(jsonObject);
                if (send_status) {
                    ZhugeSDK.getInstance().getConfig().debug("上传数据成功");
                }
            }
        });
    }
}