package com.zhuge.analysis.stat;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.telephony.TelephonyManager;
import android.util.Log;

import com.zhuge.analysis.sys_svs.AccountInfoUtils;
import com.zhuge.analysis.sys_svs.ConnectivityUtils;
import com.zhuge.analysis.sys_svs.DeviceInfoUtils;
import com.zhuge.analysis.sys_svs.ManifestUtils;
import com.zhuge.analysis.sys_svs.PackageInfosUtils;
import com.zhuge.analysis.sys_svs.TelephonyUtils;
import com.zhuge.analysis.sys_svs.TimeUtils;
import com.zhuge.analysis.sys_svs.WifiInfoUtils;
import com.zhuge.analysis.util_svs.EventStore;
import com.zhuge.analysis.util_svs.UpdateOnlineConfig;
import com.zhuge.analysis.util_svs.ZGJSONObject;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * zhuge配置 Created by kongmiao on 14-10-11.
 */
public class ZhugeConfig {

    /**
     * SDK属性相关 SDK_V SDK版本 TAG LOG的标识
     */
    public final static String SDK_V = "v1.5.1";
    public final static String TAG = "ZhugeSDK";
    public final static String CUID_KEY = "cuid";

    private final static String SESSION_EVENT_COUNT = "session_event_count";
    /**
     * 每天上传事件数
     */
    private final static int PER_EVENT_MAX = 1000;
    /**
     * 设备应用相关 did 用户唯一标识 appkey appkey channel 渠道
     */
    private String did = null;
    private String imei = null;
    private String mac = null;
    private String appkey = "";
    private String channel = "";
    private String appName = "";
    private String appVersion = "";

    private String sharedPreferencesName = "";
    private String cr = "";
    private int net = -100;
    private int mnet = -100;

    /**
     * 上传规则预定义 INSTANT 实时上传 ONLY_WIFI 只在WiFi下上传 APP_LAUNCH 应用打开时（default） PERIOD
     * 间隔时间
     */
    // public final static int INSTANT = 1;
    // public final static int ONLY_WIFI = 2;
    // public final static int APP_LAUNCH = 3;
    // public final static int PERIOD = 4;

    /**
     * 存储发送规则相关 upload_per_day 每天上传次数 upload_retry 上传重试次数 local_max 本地最大缓存数
     * timeout 连接超时
     */
    private int read_timeout = 30000;
    private int upload_per_day = 500;
    private int connect_timeout = 30000;
    private int period_time = 10;
    private int localMax = 5;

    final int MSG_RECV = 0;
    final int MSG_READ = 1;

    /**
     * 会话规则相关 debug 是否调试 disable_upload 是否禁止上传 session_exceed 会话过期时间
     * disable_accounts 禁止上传账户信息 disable_applist 禁用用户列表 upload_method
     * 上传规则（默认启动时上传）
     */
    private boolean debug = false;
    private boolean disable_upload = false;
    private int session_exceed = 30;
    private boolean disable_accounts = false;
    private boolean disable_phonenum = false;
    private boolean disable_applist = false;

    /**
     * 常量配置 executor_ 线程相关 connectionProcessorFuture_
     */
    private ExecutorService executor_;
    private EventStore eventStore;
    private long session_time;
    private SharedPreferences zhugeSettings;

    private synchronized void ensureExecutor() {
        if (executor_ == null) {
            executor_ = Executors.newSingleThreadExecutor();
        }
    }

    private int getTimeZone() {
        TimeZone tz = TimeZone.getDefault();
        return tz.getOffset(System.currentTimeMillis());
    }

    /**
     * 设置调试状态
     *
     * @param debug
     */
    public void setDebug(boolean debug) {
        this.debug = debug;
    }

    public boolean isDebug() {
        return this.debug;
    }

    /**
     * 设置SharedPreferences文件名称
     */
    void setSharedPreferencesName(String sharedPreferencesName) {
        this.sharedPreferencesName = sharedPreferencesName;
    }

    /**
     * 设置禁止上传
     *
     * @param disable_upload
     */
    private void setDisableUpload(boolean disable_upload) {
        this.disable_upload = disable_upload;
    }

    /**
     * 获取会话的时长限制
     *
     * @return
     */
    public int getSessionExceed() {
        return session_exceed;
    }

    /**
     * 设置会话的时长限制
     *
     * @param session_exceed
     */
    private void setSessionExceed(int session_exceed) {
        this.session_exceed = session_exceed;
    }

    /**
     * 是否禁用账户中心数据
     *
     * @return
     */
    public boolean isDisableAccounts() {
        return disable_accounts;
    }

    /**
     * 设置禁用账户中心数据
     *
     * @param disable_accounts
     */
    public void setDisableAccounts(boolean disable_accounts) {
        this.disable_accounts = disable_accounts;
    }

    /**
     * 是否禁用app列表
     *
     * @return
     */
    private boolean isDisableApplist() {
        return disable_applist;
    }

    /**
     * 设置禁用app列表
     *
     * @param disable_applist
     */
    private void setDisableApplist(boolean disable_applist) {
        this.disable_applist = disable_applist;
    }

    /**
     * 获取连接超时限制
     *
     * @return
     */
    public int getConnectTimeout() {
        return connect_timeout;
    }

    /**
     * 获取读取超时的限制
     *
     * @return
     */
    public int getReadTimeout() {
        return read_timeout;
    }

    /**
     * 获取设备id
     *
     * @return
     */
    public String getDid() {
        return did;
    }

    /**
     * 获取app的版本
     *
     * @return
     */
    public String getAppVersion() {
        return appVersion;
    }

    /**
     * 获取app key
     *
     * @return
     */
    public String getAppkey() {
        return appkey;
    }

    /**
     * 获取渠道名称
     *
     * @return
     */
    public String getChannel() {
        return channel;
    }

    /**
     * 获取应用名称
     *
     * @return
     */
    public String getAppName() {
        return appName;
    }

    /**
     * 获取会话时长
     *
     * @return
     */
    public String getSessionTime() {
        return TimeUtils.formatTime(session_time);
    }

    /**
     * 设置会话时长
     *
     * @param session_time
     */
    private void setSessionTime(long session_time) {
        this.session_time = session_time;
    }

    /**
     * 设置渠道名
     *
     * @param channel
     */
    public void setChannel(String channel) {
        if (this.channel.equals("") && !channel.equals(""))
            this.channel = channel;
    }

    /**
     * 设置appkey
     *
     * @param appkey
     */
    public void setAppkey(String appkey) {
        if (this.appkey.equals("") && !appkey.equals("")) {
            this.appkey = appkey;
        }
    }

    /**
     * 设置连接超时
     *
     * @param connect_timeout
     */
    private void setConnectTimeout(int connect_timeout) {
        this.connect_timeout = connect_timeout;
    }

    /**
     * 设置读取超时
     *
     * @param read_timeout
     */
    private void setReadTimeout(int read_timeout) {
        this.read_timeout = read_timeout;
    }

    /**
     * 设置是否禁用手机号
     *
     * @param disable_phonenum
     */
    public void setDisablePhonenum(boolean disable_phonenum) {
        this.disable_phonenum = disable_phonenum;
    }

    /**
     * 获取诸葛的SharePrefrence设置
     *
     * @return
     */
    private SharedPreferences getZhugeSettings() {
        return zhugeSettings;
    }

    /**
     * 是否手机号禁用
     *
     * @return
     */
    public boolean isDisablePhonenum() {
        return disable_phonenum;
    }

    /**
     * 获取网络类型
     *
     * @return
     */
    public int getNet() {
        return net;
    }

    /**
     * 获取间隔上传时间
     *
     * @return
     */
    public int getPeriodTime() {
        return period_time;
    }

    /**
     * 设置间隔上传时间
     *
     * @param period_time
     */
    private void setPeriodTime(int period_time) {
        this.period_time = period_time;
    }

    public int getLocalMax() {
        return localMax;
    }

    private void setLocalMax(int localMax) {
        this.localMax = localMax;
    }

    /**
     * 计算设备唯一ID
     *
     * @param context
     */
    void initDeviceInfo(Context context) {
        try {
            String mac = null;
            String imei = null;

            try {
                TelephonyManager tm = new TelephonyUtils(context).getTM();
                imei = tm.getDeviceId();
                this.cr = tm.getSimOperator();
                this.mnet = tm.getNetworkType();
            } catch (Exception e) {
                debug("获取IMEI和运营商失败，检查权限READ_PHONE_STATE");
            }
            try {
                mac = new WifiInfoUtils(context).getMacAddress();
            } catch (Exception e) {
                debug("获取MAC失败，检查权限ACCESS_WIFI_STATE");
            }
            if (mac == null && imei == null)
                return;
            try {
                this.mac = mac;
                this.imei = imei;
                this.did = md5(imei + mac);
            } catch (Exception e) {
                debug("计算用户唯一ID失败");
            }
            try {
                this.net = new ConnectivityUtils(context).getNetworkType();
            } catch (Exception e) {
                debug("获取网络类型失败，检查权限ACCESS_NETWORK_STATE，或者网络连接状态");
            }
            eventStore = new EventStore(context, appkey, sharedPreferencesName);
            zhugeSettings = context.getSharedPreferences(sharedPreferencesName, Context.MODE_PRIVATE);
        } catch (Exception e) {
            debug("初始化设备信息错误" + e.getMessage());
        }
    }

    /**
     * 获取开发者信息
     */
    void initDevInfo(Context context) {
        String[] devInfo = ManifestUtils.getDevInfo(context);

        setAppkey(devInfo[0]);
        this.channel = devInfo[1];
    }

    /**
     * 获取应用信息
     */
    void initAppinfo(Context context) {
        try {
            PackageInfo devInfo = ManifestUtils.getAppInfo(context);
            if (devInfo != null) {
                this.appName = devInfo.applicationInfo.loadLabel(context.getPackageManager()).toString();
                this.appVersion = devInfo.versionName;
            } else {
                debug("获取应用信息错误");
            }
        } catch (Exception e) {
            debug("初始化应用信息错误");
        }

    }

    /**
     * 后台获取最新配置
     */
    public void updateConfig(Context context) {
        try {

            final String config_key = "upload";
            ensureExecutor();
            executor_.submit(new Runnable() {
                @Override
                public void run() {
                    try {
                        final String save_key = "config_ts";
                        JSONObject config_info = null;
                        String config;
                        long info_ts = getZhugeSettings().getLong(save_key, -1);
                        long now_t = System.currentTimeMillis();
                        if (info_ts == -1 || (now_t / 86400000 - info_ts / 86400000) > 0) {
                            config_info = UpdateOnlineConfig.getServerConfig();
                        }
                        if (config_info == null) {
                            config = getZhugeSettings().getString(config_key, null);
                        } else {
                            config = config_info.optString("config");
                            if (config != null) {
                                SharedPreferences.Editor editor = getZhugeSettings().edit();
                                editor.putString(config_key, config);
                                editor.putLong(save_key, now_t);
                                editor.commit();
                            }
                        }
                        if (config != null) {
                            setConfigInfo(config);
                        }
                        debug(getConfigInfo());
                    } catch (Exception e) {
                        debug("解析配置出错" + e.getMessage());
                    }
                }
            });
        } catch (Exception e) {
            debug("初始线程请求配置出错");
        }
    }

    /**
     * md5加密
     *
     * @param string
     *            待加密字符
     * @return 加密字符
     */
    public static String md5(String string) {
        byte[] hash;
        try {
            hash = MessageDigest.getInstance("MD5").digest(string.getBytes("UTF-8"));
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("Huh, MD5 should be supported?", e);
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Huh, UTF-8 should be supported?", e);
        }

        StringBuilder hex = new StringBuilder(hash.length * 2);
        for (byte b : hash) {
            if ((b & 0xFF) < 0x10)
                hex.append("0");
            hex.append(Integer.toHexString(b & 0xFF));
        }
        return hex.toString();
    }

    /**
     * 获取应用和开发者信息
     */
    String getInitDebugInfo(Context context) {
        StringBuilder info = new StringBuilder();
        info.append("appkey：" + this.appkey + "\n");
        info.append("渠道名称：" + this.channel + "\n");
        info.append("应用名称：" + this.appName + "\n");
        info.append("应用版本：" + this.appVersion + "\n");
        info.append("用户标识：" + this.did + "\n");
        info.append("运营商代号：" + this.cr + "\n");
        info.append("网络连接类型：" + this.net + "\n");
        info.append("移动网络类型：" + this.mnet + "\n");
        info.append("系统版本：" + DeviceInfoUtils.getOSVersion() + "\n");
        info.append("手机型号：" + DeviceInfoUtils.getDevice() + "\n");
        info.append("系统分辨率：" + DeviceInfoUtils.getResolution(context) + "\n");
        return info.toString();
    }

    /**
     * 获取配置信息
     *
     * @return 信息
     */
    public String getConfigInfo() {
        StringBuilder info = new StringBuilder();
        info.append("禁止收集该设备：" + this.disable_upload + "\n");
        info.append("禁止上传应用列表：" + this.disable_applist + "\n");
        info.append("禁用账户中心：" + this.disable_accounts + "\n");
        // info.append("上传方式："+this.upload_method + "\n");
        info.append("会话超期时间（秒）：" + this.session_exceed + "\n");
        return info.toString();
    }

    /**
     * 设置配置信息
     *
     * @param config
     *            配置字符串
     */
    private void setConfigInfo(String config) {
        String[] configArray = config.split(":");
        boolean disableUpload = Boolean.valueOf(configArray[0]);
        String[] serviceDisabled = configArray[1].split("\\|");
        boolean disableApplist = Boolean.valueOf(serviceDisabled[0]);
        boolean disableAccounts = Boolean.valueOf(serviceDisabled[1]);
        boolean disablePhoneNum = Boolean.valueOf(serviceDisabled[2]);
        String[] uploadConfig = configArray[2].split("\\|");
        int uploadMethod = Integer.valueOf(uploadConfig[0]);
        int sessionExceed = Integer.valueOf(uploadConfig[1]);
        int localMax = Integer.valueOf(uploadConfig[3]);
        int connectTimeout = Integer.valueOf(uploadConfig[5]);
        int readTimeout = Integer.valueOf(uploadConfig[6]);
        int periodTime = Integer.valueOf(uploadConfig[7]);
        setDisableUpload(disableUpload);
        setDisableApplist(disableApplist);
        setDisableAccounts(disableAccounts);
        setDisablePhonenum(disablePhoneNum);
        setLocalMax(localMax);
        setSessionExceed(sessionExceed);
        setReadTimeout(readTimeout);
        setPeriodTime(periodTime);
        setConnectTimeout(connectTimeout);
    }

    /**
     * 初始化个人信息
     *
     * @param context
     *            context
     */
    private void initInfo(Context context) {
        try {
            final String config_key = "info_ts";
            long info_ts = getZhugeSettings().getLong(config_key, -1);
            long now_t = System.currentTimeMillis();
            if (info_ts == -1 || (now_t / 86400000 - info_ts / 86400000) > 7) {
                ZGJSONObject infoObject = new ZGJSONObject();
                infoObject.put("et", "info");
                infoObject.put("an", getAppName());
                infoObject.put("vn", getAppVersion());
                infoObject.put("ov", DeviceInfoUtils.getOSVersion());
                infoObject.put("rs", DeviceInfoUtils.getResolution(context));
                infoObject.put("dv", DeviceInfoUtils.getDevice());
                infoObject.put("maker", DeviceInfoUtils.getManfacturer());
                infoObject.put("br", DeviceInfoUtils.getBrand());
                infoObject.put("cr", this.cr);
                infoObject.put("net", this.net);
                infoObject.put("imei", this.imei);
                infoObject.put("mac", this.mac);
                infoObject.put("cn", getChannel());
                infoObject.put("tz", "" + getTimeZone());
                infoObject.put("ts", TimeUtils.formatTime(now_t));
                infoObject.put("sid", getSessionTime());
                infoObject.put("mb", isDisablePhonenum() ? "" : new TelephonyUtils(context).getPhoneNumber());
                eventStore.addEvent(infoObject);
                getZhugeSettings().edit().putLong(config_key, now_t).commit();
            }
        } catch (Exception e) {
            debug("初始化个人信息出错" + e.getMessage());
        }
    }

    /**
     * 初始化会话信息
     *
     * @param context
     *            context
     */
    private void initSession(Context context) {
        try {
            long now_t = System.currentTimeMillis();
            try {
                this.net = new ConnectivityUtils(context).getNetworkType();
            } catch (Exception e) {
                debug("获取网络类型失败，检查权限ACCESS_NETWORK_STATE，或者网络连接状态");
            }
            setSessionTime(now_t);
            ZGJSONObject infoObject = new ZGJSONObject();
            infoObject.put("et", "ss");
            infoObject.put("an", getAppName());
            infoObject.put("vn", getAppVersion());
            infoObject.put("ov", DeviceInfoUtils.getOSVersion());
            infoObject.put("cr", this.cr);
            infoObject.put("net", this.net);
            infoObject.put("mnet", this.mnet);
            infoObject.put("tz", "" + getTimeZone());
            infoObject.put("ts", TimeUtils.formatTime(now_t));
            infoObject.put("sid", getSessionTime());
            eventStore.addEvent(infoObject);
        } catch (Exception e) {
            debug("初始化会话出错。" + e.getMessage());
        }
    }

    /**
     * 初始化事件
     *
     * @param context
     *            context
     */
    void initEvent(final Context context) {
        if (eventStore.timerService_ == null) {
            eventStore.startUploadService(context);
        }
        completeLastSession(context, 1);
        initSession(context);
        ensureExecutor();
        executor_.submit(new Runnable() {
            @Override
            public void run() {
                initInfo(context);

                if (!isDisableAccounts())
                    initAcs(context);
                if (!isDisableApplist())
                    initPkgs(context);
            }
        });
    }

    /**
     * 存储session最后会话的时间
     *
     * @param end_time
     */
    public synchronized void saveSessionEnd(long end_time) {
        getZhugeSettings().edit().putString("pre_end", session_time + "|" + end_time).commit();
    }

    /**
     * 完成上次session记录
     *
     * @param context
     */
    public synchronized void completeLastSession(Context context, int sr) {
        try {
            String last_end = getZhugeSettings().getString("pre_end", "");
            if ("".equals(last_end)) {
                return;
            }
            String[] last_info = last_end.split("\\|");
            long sid = Long.valueOf(last_info[0]);
            long ts = Long.valueOf(last_info[1]);
            ZGJSONObject infoObject = new ZGJSONObject();
            infoObject.put("et", "se");
            infoObject.put("sr", sr);
            infoObject.put("ec", getEventCount());
            infoObject.put("tz", "" + getTimeZone());
            infoObject.put("ts", TimeUtils.formatTime(ts));
            infoObject.put("sid", TimeUtils.formatTime(sid));
            infoObject.put("dr", TimeUtils.formatTime(ts - sid));
            getZhugeSettings().edit().putString("pre_end", "").commit();
            eventStore.addEvent(infoObject);
            resetSessionEvent();
            ZhugeSDK.getInstance().halt();
        } catch (Exception e) {
            debug("完成上次会话出错" + e.getMessage());
        }
    }

    /**
     * 自定义事件
     *
     * @param context
     * @param event_name
     * @param kv
     */
    void onCustomEvent(Context context, String event_name, JSONObject kv) {
        try {
            if (!debug && isOverCount()) {
                return;
            }
            long now_t = System.currentTimeMillis();
            ZGJSONObject infoObject = new ZGJSONObject();
            infoObject.put("et", "cus");
            infoObject.put("tz", "" + getTimeZone());
            infoObject.put("ts", TimeUtils.formatTime(now_t));
            infoObject.put("eid", event_name);
            infoObject.put("pr", kv);
            infoObject.put("sid", getSessionTime());
            if (debug) {
                saveSessionEnd(now_t);
                eventStore.sendDataImmediately(infoObject);
                return;
            }
            eventStore.addEvent(infoObject);
            saveSessionEnd(now_t);
            addEventCount();
        } catch (Exception e) {
            debug("自定义事件出错" + e.getMessage());
        }
    }

    /**
     * 自定义事件
     *
     * @param context
     * @param event_name
     */
    void onCustomEvent(Context context, String event_name) {
        try {
            if (!debug && isOverCount()) {
                return;
            }
            long now_t = System.currentTimeMillis();
            ZGJSONObject infoObject = new ZGJSONObject();
            infoObject.put("et", "cus");
            infoObject.put("tz", "" + getTimeZone());
            infoObject.put("ts", TimeUtils.formatTime(now_t));
            infoObject.put("eid", event_name);
            infoObject.put("sid", getSessionTime());
            if (debug) {
                saveSessionEnd(now_t);
                eventStore.sendDataImmediately(infoObject);
                return;
            }
            eventStore.addEvent(infoObject);
            saveSessionEnd(now_t);
            addEventCount();
        } catch (Exception e) {
            debug("自定义事件出错" + e.getMessage());
        }
    }

    /**
     * 自定义事件
     *
     * @param context
     * @param event_name
     * @param kv
     */
    void onCustomEvent(Context context, String event_name, HashMap<String, Object> kv) {
        try {
            if (!debug && isOverCount()) {
                return;
            }
            long now_t = System.currentTimeMillis();
            ZGJSONObject infoObject = new ZGJSONObject();
            infoObject.put("et", "cus");
            infoObject.put("tz", "" + getTimeZone());
            infoObject.put("ts", TimeUtils.formatTime(now_t));
            infoObject.put("eid", event_name);
            infoObject.put("pr", new JSONObject(kv));
            infoObject.put("sid", getSessionTime());
            if (debug) {
                saveSessionEnd(now_t);
                eventStore.sendDataImmediately(infoObject);
                return;
            }
            eventStore.addEvent(infoObject);
            saveSessionEnd(now_t);
            addEventCount();
        } catch (Exception e) {
            debug("自定义事件出错" + e.getMessage());
        }
    }

    /**
     * 标注用户
     *
     * @param context
     * @param uid
     * @param kv
     */
    void identifyPerson(Context context, String uid, HashMap<String, Object> kv) {
        try {
            long now_t = System.currentTimeMillis();
            ZGJSONObject infoObject = new ZGJSONObject();
            infoObject.put("et", "idf");
            infoObject.put("tz", "" + getTimeZone());
            infoObject.put("ts", TimeUtils.formatTime(now_t));
            infoObject.put("cuid", uid);
            infoObject.put("pr", new JSONObject(kv));
            infoObject.put("sid", getSessionTime());
            getZhugeSettings().edit().putString(CUID_KEY, uid).commit();
            eventStore.addEvent(infoObject);
            eventStore.send(true);
        } catch (Exception e) {
            debug("标识用户出错" + e.getMessage());
        }
    }

    /**
     * 标注用户
     *
     * @param context
     * @param uid
     * @param kv
     */
    void identifyPerson(Context context, String uid, JSONObject kv) {
        try {
            long now_t = System.currentTimeMillis();
            ZGJSONObject infoObject = new ZGJSONObject();
            infoObject.put("et", "idf");
            infoObject.put("tz", "" + getTimeZone());
            infoObject.put("ts", TimeUtils.formatTime(now_t));
            infoObject.put("cuid", uid);
            infoObject.put("pr", kv);
            infoObject.put("sid", getSessionTime());
            getZhugeSettings().edit().putString(CUID_KEY, uid).commit();
            eventStore.addEvent(infoObject);
            eventStore.send(true);
        } catch (Exception e) {
            debug("标识用户出错" + e.getMessage());
        }
    }

    void channelData(String chan, String userId) {

        try {
            long now_t = System.currentTimeMillis();
            ZGJSONObject infoObject = new ZGJSONObject();
            infoObject.put("et", "_usermap_");
            infoObject.put("tz", "" + getTimeZone());
            infoObject.put("ts", TimeUtils.formatTime(now_t));
            JSONObject ch = new JSONObject();
            ch.put("channel", chan);
            ch.put("user_id", userId);
            infoObject.put("pr", ch);
            eventStore.addEvent(infoObject);
            eventStore.send(true);
        } catch (Exception e) {
            debug("第三方渠道上传错误" + e.getMessage());
        }
    }

    private void sendMid(int msgState, String chan, String mid) {

        if (null == mid || "".equals(mid) || "null".equals(mid) || mid.length() < 1)
            return;
        try {
            long now_t = System.currentTimeMillis();
            ZGJSONObject infoObject = new ZGJSONObject();
            switch (msgState) {
            case MSG_RECV:
                infoObject.put("et", "_msgrecv_");
                break;
            case MSG_READ:
                infoObject.put("et", "_msgread_");
                break;
            default:
                return;
            }
            infoObject.put("tz", "" + getTimeZone());
            infoObject.put("ts", TimeUtils.formatTime(now_t));
            JSONObject ch = new JSONObject();
            ch.put("channel", chan);
            ch.put("mid", mid);
            infoObject.put("pr", ch);
            eventStore.addEvent(infoObject);
            eventStore.send(true);
        } catch (Exception e) {
            debug("通知上传错误" + e.getMessage());
        }
    }

    /**
     * 获取应用列表
     *
     * @param context
     */
    private void initPkgs(Context context) {
        try {
            final String config_key = "pkg_ts";
            long info_ts = getZhugeSettings().getLong(config_key, -1);

            long now_t = System.currentTimeMillis();
            if (info_ts == -1 || (now_t / 86400000 - info_ts / 86400000) > 14) {
                ZGJSONObject infoObject = new ZGJSONObject();
                infoObject.put("et", "pkgs");
                infoObject.put("list", new PackageInfosUtils(context).getUserAppInfos());
                infoObject.put("tz", "" + getTimeZone());
                infoObject.put("ts", TimeUtils.formatTime(now_t));
                infoObject.put("sid", getSessionTime());
                eventStore.addEvent(infoObject);
                getZhugeSettings().edit().putLong(config_key, now_t).commit();
            }
        } catch (Exception e) {
            debug("获取应用列表出错。" + e.getMessage());
        }
    }

    /**
     * 获取账户中心数据
     *
     * @param context
     */
    private void initAcs(Context context) {
        try {
            final String config_key = "acs_ts";
            long info_ts = getZhugeSettings().getLong(config_key, -1);

            long now_t = System.currentTimeMillis();
            if (info_ts == -1 || (now_t / 86400000 - info_ts / 86400000) > 14) {
                ZGJSONObject infoObject = new ZGJSONObject();
                infoObject.put("et", "acs");
                infoObject.put("list", new AccountInfoUtils(context).getAccountInfos());
                infoObject.put("tz", "" + getTimeZone());
                infoObject.put("ts", TimeUtils.formatTime(now_t));
                infoObject.put("sid", getSessionTime());
                eventStore.addEvent(infoObject);
                getZhugeSettings().edit().putLong(config_key, now_t).commit();
            }
        } catch (Exception e) {
            debug("没有添加权限GET_ACCOUNTS");
        }
    }

    public void debug(String info) {
        if (this.debug) {
            Log.e(ZhugeConfig.TAG, info);
        }
    }

    void sendData() {
        eventStore.send(true);
    }

    public int getUpload_per_day() {
        return upload_per_day;
    }

    public void setUpload_per_day(int upload_per_day) {
        this.upload_per_day = upload_per_day;
    }

    public boolean stopUpload() {
        return eventStore.stopTimeService();
    }

    /**
     * 事件数有没有超出
     */

    private boolean isOverCount() {
        return eventStore.getTodayTotal() > PER_EVENT_MAX;
    }

    /**
     * 获取当前会话事件数
     *
     * @return 事件数
     */
    private int getEventCount() {
        return getZhugeSettings().getInt(SESSION_EVENT_COUNT, 0);
    }

    /**
     * 修改当次会话事件数
     */
    private synchronized void addEventCount() {
        int eventCount = getZhugeSettings().getInt(SESSION_EVENT_COUNT, 0);
        getZhugeSettings().edit().putInt(SESSION_EVENT_COUNT, ++eventCount).apply();
    }

    /**
     * 重置会话事件数
     */
    private synchronized void resetSessionEvent() {
        getZhugeSettings().edit().putInt(SESSION_EVENT_COUNT, 0).apply();
    }

    void parseMid(int msgState, ZhugeSDK.PushChannel channel, Object t) {
        JSONObject js;
        String mid = "";
        try {
            switch (channel) {
            case BAIDU:
                String baidu = (String) t;
                if (null == baidu)
                    return;
                js = new JSONObject(baidu);
                mid = js.getString("mid");
                break;

            case GETUI:
                break;

            case JPUSH:
                if (t instanceof String) {
                    String jpush = (String) t;
                    js = new JSONObject(jpush);
                    mid = js.getString("mid");
                } else if (t instanceof JSONObject) {
                    js = (JSONObject) t;

                    mid = js.getString("mid");
                }
                break;
            case UMENG:
                js = (JSONObject) t;
                JSONObject midJson = js.getJSONObject("extra");
                mid = midJson.getString("mid");
                break;
            case XIAOMI:
                if (t instanceof Map) {
                    Map<String, String> map = (Map<String, String>) t;
                    mid = map.get("mid");
                }
                break;
            case XINGE:

                if (t instanceof String) {
                    String xinge = (String) t;
                    js = new JSONObject(xinge);
                    mid = js.getString("mid");
                }
                break;
            }
        } catch (ClassCastException e) {
            debug("Object参数传递错误");
        } catch (JSONException e) {
            debug("json转换出错" + e.getMessage());
        }
        sendMid(msgState, channel.toString(), mid);
    }
}
