package com.zhuge.analysis.util_svs;

import android.content.Context;
import android.content.SharedPreferences;
import android.util.Base64;

import com.zhuge.analysis.stat.ZhugeConfig;
import com.zhuge.analysis.stat.ZhugeSDK;
import com.zhuge.analysis.sys_svs.ActivityServicesUtils;

import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.params.CoreConnectionPNames;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.zip.Deflater;

/**
 * Event 管理类
 * Created by kongmiao on 14-10-13.
 */
public class EventStore {

    private SharedPreferences zhugeSettings;

    private SharedPreferences getZhugeSettings() {
        return zhugeSettings;
    }

    private String PF_KEY;
    private static final String EVENT_KEY = "events";
    private static final String TOTALDAY_KEY = "total";
    private static final String DELIMITER = "===";
    private static ExecutorService executor_;
    private static ExecutorService debugExecutor;
    private long now_t = -100;
    public ScheduledExecutorService timerService_;
    private ScheduledFuture timerFuture;
    private static ActivityServicesUtils activityServicesUtils;
    private int todayTotal = 0;
    private Context mContext = null;

    private synchronized void ensureExecutor() {
        if (executor_ == null) {
            executor_ = Executors.newSingleThreadExecutor();
        }
    }

    private synchronized void ensureDebugExecutor() {
        if (debugExecutor == null) {
            debugExecutor = Executors.newCachedThreadPool();
        }
    }


    public EventStore(final Context context, String pf) {
        this.PF_KEY = pf;
        mContext = context;
        zhugeSettings = context.getSharedPreferences(PF_KEY, Context.MODE_PRIVATE);
        startUploadService(context);
        String todayInfo = getZhugeSettings().getString(TOTALDAY_KEY, "");
        if ("".equals(todayInfo)) {
            todayTotal = 0;
        } else {
            String[] tiInfo = todayInfo.split("\\|");
            if ((int) System.currentTimeMillis() / 86400000 - Integer.parseInt(tiInfo[0]) / 86400000 > 0) {
                todayTotal = 0;
            } else {
                todayTotal = Integer.parseInt(tiInfo[1]);
            }
        }
    }

    public synchronized void startUploadService(final Context context) {
        if (timerService_ == null) {
            timerService_ = Executors.newSingleThreadScheduledExecutor();
        }
        timerFuture = timerService_.scheduleWithFixedDelay(new Runnable() {
            @Override
            public void run() {
                if (activityServicesUtils == null) {
                    activityServicesUtils = new ActivityServicesUtils(context);
                }
                String presentPackage = activityServicesUtils.getForegroundPackage();
                if (!presentPackage.contains(context.getPackageName())) {
                    try {
                        ZhugeSDK.getInstance().getConfig().debug("会话暂停");
                        ZhugeSDK.getInstance().updateSessionEnd();
                        Thread.sleep(20000);
                        presentPackage = activityServicesUtils.getForegroundPackage();
                        if (!presentPackage.contains(context.getPackageName())) {
                            ZhugeSDK.getInstance().getConfig().completeLastSession(context);
                            send(true);
                            ZhugeSDK.getInstance().getConfig().debug("会话结束");
                            timerService_.schedule(new Runnable() {
                                @Override
                                public void run() {
                                    stopTimeService();
                                }
                            }, 1, TimeUnit.SECONDS);

                        } else {
                            ZhugeSDK.getInstance().getConfig().debug("会话继续");
                            ZhugeSDK.getInstance().updateSessionEnd();
                            send(false);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                } else {
                    ZhugeSDK.getInstance().updateSessionEnd();
                    send(false);
                }
            }
        }, ZhugeSDK.getInstance().getConfig().getPeriodTime(), ZhugeSDK.getInstance().getConfig().getPeriodTime(), TimeUnit.SECONDS);
    }

    public synchronized boolean stopTimeService() {
        try {
            getZhugeSettings().edit().putString(TOTALDAY_KEY, (int) System.currentTimeMillis() + "|" + todayTotal).commit();
            timerFuture.cancel(false);
        } catch (Exception e) {
            ZhugeSDK.getInstance().getConfig().debug("线程池关闭失败");
        }
        return false;
    }

    public synchronized void addEvent(ZGJSONObject event) {

        final List<ZGJSONObject> events = eventsList().get(PF_KEY);
        if (null == events)
            return;
        events.add(event);
        getZhugeSettings().edit().putString(EVENT_KEY, joinEvents(events, DELIMITER)).commit();
    }

    private Map<String, String[]> events() {
        final String joinedEventsStr = mContext.getSharedPreferences(PF_KEY, Context.MODE_PRIVATE).getString(EVENT_KEY, "");
        String[] events = joinedEventsStr.length() == 0 ? new String[0] : joinedEventsStr.split(DELIMITER);
        Map<String, String[]> map = new Hashtable<>(3);
        map.put(PF_KEY, events);

        return map;
    }

    public synchronized void removeEvents(final Collection<ZGJSONObject> eventsToRemove) {
        if (eventsToRemove != null && eventsToRemove.size() > 0) {

            final List<ZGJSONObject> events = eventsList().get(PF_KEY);
            if (null == events)
                return;
            if (events.removeAll(eventsToRemove)) {
                getZhugeSettings().edit().putString(EVENT_KEY, joinEvents(events, DELIMITER)).commit();
                ZhugeSDK.getInstance().getConfig().debug("本地数据删除成功");
            }
        }
    }

    public Map<String, List<ZGJSONObject>> eventsList() {
        Map<String, String[]> map = events();
        String[] eventArray = map.get(PF_KEY);
        if (null == eventArray)
            return null;
        final List<ZGJSONObject> events = new ArrayList<>(eventArray.length);
        for (String s : eventArray) {
            try {
                final ZGJSONObject event = new ZGJSONObject(s);
                events.add(event);

            } catch (JSONException ignored) {
                // should not happen since JSONObject is being constructed from previously stringified JSONObject
                // events -> json objects -> json strings -> storage -> json strings -> here
            }
        }
        // order the events from least to most recent
        Collections.sort(events, new Comparator<JSONObject>() {
            @Override
            public int compare(final JSONObject e1, final JSONObject e2) {
                Long delta = (e1.optLong("ts", -1) - e2.optLong("ts", -1));
                return delta.intValue();
            }
        });
        Map<String, List<ZGJSONObject>> mapToUse = new Hashtable<>(3);
        mapToUse.put(PF_KEY, events);
        return mapToUse;
    }

    /**
     * Converts a collection of Event objects to URL-encoded JSON to a string, with each
     * event JSON string delimited by the specified delimiter.
     *
     * @param collection events to join into a delimited string
     * @param delimiter  delimiter to use, should not be something that can be found in URL-encoded JSON string
     */
    static String joinEvents(final Collection<JSONObject> collection, final String delimiter) {
        final List<String> strings = new ArrayList<String>();
        for (JSONObject e : collection) {
            strings.add(e.toString());
        }
        return join(strings, delimiter);
    }

    static String joinEvents(final List<ZGJSONObject> collection, final String delimiter) {
        final List<String> strings = new ArrayList<String>();
        for (JSONObject e : collection) {
            strings.add(e.toString());
        }
        return join(strings, delimiter);
    }

    /**
     * Joins all the strings in the specified collection into a single string with the specified delimiter.
     */
    static String join(final Collection<String> collection, final String delimiter) {
        final StringBuilder builder = new StringBuilder();
        int i = 0;
        for (String s : collection) {
            builder.append(s);
            if (++i < collection.size()) {
                builder.append(delimiter);
            }
        }
        String s;
        try {
            s = builder.toString();
        } catch (OutOfMemoryError e) {
            e.printStackTrace();
            return null;
        }
        return s;
    }

    // for unit testing
    synchronized void clear() {
        final SharedPreferences.Editor prefsEditor = getZhugeSettings().edit();
        prefsEditor.remove(EVENT_KEY);
        prefsEditor.commit();
    }

    public String getEventsString() {
        return getZhugeSettings().getString(EVENT_KEY, "");
    }

    public void send(final boolean flush) {
        try {
            ensureExecutor();
            executor_.submit(new Runnable() {
                @Override
                public void run() {
                    try {
                        if (ZhugeSDK.getInstance().getConfig().getNet() == -100) {
                            ZhugeSDK.getInstance().getConfig().debug("网络状态不佳" + "|" + ZhugeSDK.getInstance().getConfig().getNet());
                            return;
                        }
                        Map<String, List<ZGJSONObject>> eventMap = eventsList();
                        List<ZGJSONObject> eventList = eventMap.get(PF_KEY);
                        if (null == eventList) {
                            ZhugeSDK.getInstance().getConfig().debug("发送的应用出错");
                            return;
                        }
                        if (eventList.size() <= 0) {

                            return;
                        }
                        if (todayTotal > ZhugeSDK.getInstance().getConfig().getUpload_per_day()) {
                            ZhugeSDK.getInstance().getConfig().debug("数据超过每日上传的限制：" + todayTotal);
                            return;
                        }
                        if (!flush && eventList.size() < ZhugeSDK.getInstance().getConfig().getLocalMax()) {

                            return;
                        }
                        JSONObject jsonObject = sendData(PF_KEY, eventList);
                        if (null == jsonObject)
                            return;
                        boolean send_status = requestAPI(jsonObject);
                        if (send_status) {
                            ZhugeSDK.getInstance().getConfig().debug("上传数据成功");
                            todayTotal += eventList.size();
                            removeEvents(eventList);
                        }
                    } catch (Exception e) {
                        ZhugeSDK.getInstance().getConfig().debug("发送数据线程出错");
                    }
                }
            });
        } catch (Exception e) {
            ZhugeSDK.getInstance().getConfig().debug("上传过程出错");
        }
    }

    private synchronized JSONObject sendData(String pf_key, List<ZGJSONObject> events) {
        try {

            JSONObject postData = new JSONObject();
            ZhugeConfig tmpConfig = ZhugeSDK.getInstance().getConfig();
            if (!pf_key.equals(tmpConfig.getAppkey())) {
                ZhugeSDK.getInstance().getConfig().debug("发送的事件不属于本应用！");
                return null;
            }
            postData.put("sdkv", ZhugeConfig.SDK_V);
            postData.put("sdk", "and");
            postData.put("did", tmpConfig.getDid());
            postData.put("cn", tmpConfig.getChannel());
            postData.put("ak", tmpConfig.getAppkey());
            postData.put("pid", android.os.Process.myPid() + "," + android.os.Process.myTid());

            if (ZhugeSDK.getInstance().getConfig().isDebug()) {
                postData.put("debug", 1);
            }
            String cuid = getZhugeSettings().getString(ZhugeConfig.CUID_KEY, null);
            if (cuid != null) {
                postData.put("cuid", cuid);
            }
            now_t = System.currentTimeMillis();
            postData.put("ts", new DecimalFormat("0.000").format((double) now_t / 1000));
            postData.put("type", "statis");


            JSONArray jsonArray = new JSONArray();
            for (ZGJSONObject zgjsonObject : events) {
                jsonArray.put(zgjsonObject);
            }
            if (!this.PF_KEY.equals(tmpConfig.getAppkey())) {
                removeEvents(events);
                ZhugeSDK.getInstance().getConfig().debug("要发送的不是本应用的数据");
                return null;
            }
            postData.put("data", jsonArray);
            return postData;
        } catch (Exception e) {
            ZhugeSDK.getInstance().getConfig().debug("数据组装出错" + e.getMessage());
        }
        return null;
    }

    private boolean requestAPI(JSONObject postData) {
        try {
            String path = "http://apipool.37degree.com/APIPOOL/";
            HttpClient client = new DefaultHttpClient();
            client.getParams().setParameter(CoreConnectionPNames.CONNECTION_TIMEOUT, ZhugeSDK.getInstance().getConfig().getConnectTimeout());
            client.getParams().setParameter(CoreConnectionPNames.SO_TIMEOUT, ZhugeSDK.getInstance().getConfig().getReadTimeout());
            HttpPost httpPost = new HttpPost(path);
            List<NameValuePair> params = new ArrayList<>();
            params.add(new BasicNameValuePair("method", "event_statis_srv.upload"));
            params.add(new BasicNameValuePair("compress", "1"));
            params.add(new BasicNameValuePair("event", Base64.encodeToString(compress(postData.toString().getBytes()), Base64.DEFAULT).replace("\r", "").replace("\n", "")));
            httpPost.setEntity(new UrlEncodedFormEntity(params, "utf-8"));

            HttpResponse response = client.execute(httpPost);
            if (200 == response.getStatusLine().getStatusCode()) {
                BufferedInputStream responseStream = new BufferedInputStream(response.getEntity().getContent());
                ByteArrayOutputStream responseData = new ByteArrayOutputStream(1024);
                int c;
                while ((c = responseStream.read()) != -1) {
                    responseData.write(c);
                }
                JSONObject responseDict = new JSONObject(responseData.toString("UTF-8"));
                boolean success = responseDict.optInt("return_code") == 0;
                if (success) {
                    return true;
                } else {
                    ZhugeSDK.getInstance().getConfig().debug("请求服务器失败，服务器响应码" + responseDict.optInt("return_code"));
                }
            }
        } catch (Exception e) {
            ZhugeSDK.getInstance().getConfig().debug("连接服务器出错" + e);
        }
        return false;
    }


    public byte[] compress(byte[] data) {
        byte[] output = new byte[0];

        Deflater compresser = new Deflater();
        compresser.reset();
        compresser.setInput(data);
        compresser.finish();
        ByteArrayOutputStream bos = new ByteArrayOutputStream(data.length);
        try {
            int dataLen = data.length;
            byte[] buf = new byte[dataLen + 1];
            while (!compresser.finished()) {
                int i = compresser.deflate(buf);
                bos.write(buf, 0, i);
            }
            output = bos.toByteArray();
        } catch (Exception e) {
            output = data;
            e.printStackTrace();
        } finally {
            try {
                bos.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        compresser.end();
        return output;
    }

    public void sendDataImmediately(final ZGJSONObject data) {
        final List<ZGJSONObject> eventList = new ArrayList<>(1);
        eventList.add(data);
        ensureDebugExecutor();
        debugExecutor.submit(new Runnable() {
            @Override
            public void run() {
                JSONObject jsonObject = sendData(PF_KEY, eventList);
                if (null == jsonObject)
                    return;
                boolean send_status = requestAPI(jsonObject);
                if (send_status) {
                    ZhugeSDK.getInstance().getConfig().debug("上传数据成功");
                }
            }
        });
    }
}