package com.yodo1.sdk.kit.cache;

import android.graphics.Bitmap;
import android.util.LruCache;

import java.io.Serializable;

/**
 * KCache
 * <p>
 * Usage：github Kcache项目，移除了diskLruCache的部分。
 * <p>
 * {@link #put(String, String)}
 * {@link #put(String, String, long)}
 * {@link #put(String, byte[])}
 * {@link #put(String, byte[], long)}
 * {@link #put(String, Serializable)}
 * {@link #put(String, Serializable, long)}
 * {@link #put(String, Bitmap)}
 * {@link #put(String, Bitmap, long)}
 * <p>
 * {@link #getAsString(String)}
 * {@link #getAsBytes(String)}
 * {@link #getAsSerializable(String)}
 * {@link #getAsBitmap(String)}
 * <p>
 * {@link #remove(String)}
 * {@link #clear()}
 * {@link #size()}
 * <p>
 * Created by KasoGG on 2017/2/17.
 */
public class Yodo1Cache {

    private static final int INDEX = 0;
    private static Yodo1Cache mInstanche;
    private LruCache<String, CacheEntity> memoryLruCache;

    private Yodo1Cache() {
        initMemoryLruCache();
    }

    public static Yodo1Cache getmInstanche() {
        if (mInstanche == null) {
            mInstanche = new Yodo1Cache();
        }
        return mInstanche;
    }

    private synchronized void initMemoryLruCache() {
        if (memoryLruCache == null) {
            int maxSize = (int) (Runtime.getRuntime().maxMemory() / 8);
            memoryLruCache = new LruCache<String, CacheEntity>(maxSize) {
                @Override
                protected int sizeOf(String key, CacheEntity entity) {
                    return CacheUtils.calculateEntitySize(entity);
                }
            };
        }
    }

    public void put(String key, String value) {
        put(key, value, CacheEntity.NO_EXPIRATION);
    }

    public void put(String key, String value, long expiration) {
        put(key, value.getBytes(), expiration);
    }

    public void put(String key, Bitmap bitmap) {
        put(key, bitmap, CacheEntity.NO_EXPIRATION);
    }

    public void put(String key, Bitmap bitmap, long expiration) {
        put(key, CacheUtils.bitmapToBytes(bitmap), expiration);
    }

    public void put(String key, Serializable serializable) {
        put(key, serializable, CacheEntity.NO_EXPIRATION);
    }

    public void put(String key, Serializable serializable, long expiration) {
        put(key, CacheUtils.objectToBytes(serializable), expiration);
    }

    public void put(String key, byte[] bytes) {
        put(key, bytes, CacheEntity.NO_EXPIRATION);
    }

    public void put(String key, byte[] bytes, long expiration) {
        if (bytes == null) {
            return;
        }
        CacheEntity cacheEntity = new CacheEntity(bytes, expiration);
        put(key, cacheEntity);
    }

    private void put(String key, CacheEntity cacheEntity) {
        String hashedKey = CacheUtils.hashKey(key);
        memoryLruCache.put(hashedKey, cacheEntity);
    }

    public String getAsString(String key) {
        byte[] bytes = getAsBytes(key);
        return bytes != null ? new String(bytes) : null;
    }

    public Serializable getAsSerializable(String key) {
        byte[] bytes = getAsBytes(key);
        return bytes != null ? (Serializable) CacheUtils.bytesToObject(bytes) : null;
    }

    public Bitmap getAsBitmap(String key) {
        byte[] bytes = getAsBytes(key);
        return bytes != null ? CacheUtils.bytesToBitmap(bytes) : null;
    }

    public byte[] getAsBytes(String key) {
        CacheEntity cacheEntity = getAsCacheEntity(key);
        return cacheEntity != null ? cacheEntity.bytes : null;
    }

    private CacheEntity getAsCacheEntity(String key) {
        String hashedKey = CacheUtils.hashKey(key);
        CacheEntity cacheEntity = memoryLruCache.get(hashedKey);
        boolean notInMemory = cacheEntity == null;
        if (CacheUtils.isExpired(cacheEntity)) {
            memoryLruCache.remove(hashedKey);
            return null;
        }
        return cacheEntity;
    }

    public synchronized void remove(String key) {
        String hashedKey = CacheUtils.hashKey(key);
        memoryLruCache.remove(hashedKey);
    }

    public synchronized void clear() {
        memoryLruCache.evictAll();
    }

    public long size() {
        return memoryLruCache.size();
    }

    static class CacheEntity implements Serializable {
        private static final long serialVersionUID = -5137353261556372905L;
        private static final int NO_EXPIRATION = 0;

        byte[] bytes;
        long expiration;

        private CacheEntity(byte[] bytes, long expiration) {
            this.bytes = bytes;
            this.expiration = expiration;
        }
    }
}