package com.yodo1.sdk.kit;


import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.content.pm.PackageManager;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.os.Build;
import android.os.IBinder;
import android.os.IInterface;
import android.os.Looper;
import android.os.Parcel;
import android.os.RemoteException;
import android.provider.Settings;
import android.text.TextUtils;

import com.yodo1.sdk.base.android.Yodo1BaseUtils;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Locale;
import java.util.concurrent.LinkedBlockingQueue;

/**
 * @author yodo1
 */
public class Yodo1DeviceUtils {

    private static final String TAG = "Yodo1DeviceUtils";
    private static volatile String deviceIdCached = "", GoogleAdId = "";
    private static JSONObject deviceInfo;

    /**
     * 获取手机品牌
     *
     * @return 手机品牌
     */
    public static String getPhoneBrand() {
        return android.os.Build.BRAND;
    }

    /**
     * 获取手机型号
     *
     * @return 手机型号
     */
    public static String getPhoneVersion() {
        return android.os.Build.MODEL;
    }

    /**
     * 获取手机版本
     *
     * @return 系统版本
     */
    public static String getOsVersion() {
        return android.os.Build.VERSION.RELEASE;
    }

    /**
     * 获取 CountryCode
     *
     * @param context Context
     * @return Country Code
     */
    public static String getCountryCode(Context context) {
        if (context == null) {
            return "";
        }
        Locale locale;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            locale = context.getResources().getConfiguration().getLocales().get(0);
        } else {
            locale = context.getResources().getConfiguration().locale;
        }
        return locale.getCountry();
    }

    /**
     * 获取 LanguageCode
     *
     * @param context Context
     * @return Language Code
     */
    public static String getLanguageCode(Context context) {
        if (context == null) {
            return "";
        }
        Locale locale;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            locale = context.getResources().getConfiguration().getLocales().get(0);
        } else {
            locale = context.getResources().getConfiguration().locale;
        }
        return locale.getLanguage();
    }

    /**
     * 获取MAC地址
     *
     * @param context Context
     * @return mac address
     */
    public static String getMacAddress(Context context) {
        String macStr = "";
        WifiManager wifi = (WifiManager) context.getApplicationContext().getSystemService(Context.WIFI_SERVICE);
        try {
            WifiInfo info = wifi.getConnectionInfo();
            macStr = info.getMacAddress();
        } catch (Exception e) {
            YLog.d("getMacAddress", e);
        }
        return macStr;
    }


    /**
     * 这个方法是耗时的，不能在主线程调用
     *
     * @param context Context
     * @return google gaid
     * @throws Exception
     */
    public static String getGoogleAdId(Context context) throws Exception {
        if (!TextUtils.isEmpty(GoogleAdId)) {
            return GoogleAdId;
        } else {
            if (Looper.getMainLooper() == Looper.myLooper()) {
                throw new IllegalStateException("Cannot call in the main thread, You must call in the other thread");
            }
            try {
                PackageManager pm = context.getPackageManager();
                pm.getPackageInfo("com.android.vending", 0);
            } catch (Exception e) {
                throw e;
            }
            AdvertisingConnection connection = new AdvertisingConnection();
            Intent intent = new Intent("com.google.android.gms.ads.identifier.service.START");
            intent.setPackage("com.google.android.gms");
            if (context.bindService(intent, connection, Context.BIND_AUTO_CREATE)) {
                try {
                    AdvertisingInterface adInterface = new AdvertisingInterface(connection.getBinder());
                    GoogleAdId = adInterface.getId();
                    return GoogleAdId;
                } catch (Exception e) {
                    throw e;
                } finally {
                    context.unbindService(connection);
                }
            }
            throw new IOException("Google Play connection failed");
        }
    }

    /**
     * 获取设备ID。
     * <p>
     * 目前采取的策略是：第一次读取后在本地存储至本地，以后都从存储空间中读取。
     * <p>
     * 在6.0或者没有READ_PHONE_STATE的设备上，采取androidId + serial的方式。
     * 在非6.0并且有READ_PHONE_STATE权限的设备上，依旧采取老的imei方式
     *
     * @param ctx Context
     * @return 可以的话返回曾经的设备ID, 否则返回新策略生成的
     */
    public static String getDeviceId(Context ctx) {
        if (TextUtils.isEmpty(deviceIdCached)) {
            //1,首先sp然后sd.
            deviceIdCached = getDeviceIdForSave(ctx);
            if (TextUtils.isEmpty(deviceIdCached)) {
                deviceIdCached = YFileUtils.readFilesFromSDCard(YFileUtils.YODO1FILE);
            }
            //3,debug重置，hunter,重写
            if (!TextUtils.isEmpty(deviceIdCached) && deviceIdCached.contains("openYodo1Log")) {
                deviceIdCached = "";
            }
            //没有生成，则重新生成再存储。
            if (TextUtils.isEmpty(deviceIdCached)) {
                //4,imei
                if (TextUtils.isEmpty(deviceIdCached)) {
                    deviceIdCached = getIMEI(ctx);
                }
                //5,gaid
                if (TextUtils.isEmpty(deviceIdCached)) {
                    try {
                        deviceIdCached = getGoogleAdId(ctx);
                    } catch (Exception e) {
                        YLog.i(TAG, e);
                    }
                }
                //6,uuid生成并存储
                if (TextUtils.isEmpty(deviceIdCached)) {
                    deviceIdCached = java.util.UUID.randomUUID().toString();
                }
                //7，第一次生成，保存下来。以便共享。
                saveDeviceId(ctx, deviceIdCached);
                YFileUtils.writeFilesToSDCard(deviceIdCached, YFileUtils.YODO1FILE);
            }
            YLog.i(TAG, "deviceId:" + deviceIdCached);
        }
        return deviceIdCached;
    }

    /**
     * 获取 IMEI
     *
     * @param context Context
     * @return imei number
     */
    public static String getIMEI(Context context) {
        return Yodo1BaseUtils.getIMEI(context);
    }

    /**
     * 获取 Android ID
     *
     * @param context Context
     * @return android id
     */
    public static String getAndroidID(Context context) {
        if (context == null) {
            return "";
        }
        String androidId = Settings.Secure.getString(context.getContentResolver(), Settings.Secure.ANDROID_ID);
        return (TextUtils.isEmpty(androidId) ? "" : androidId);
    }

    private static void saveDeviceId(Context context, String value) {
        Yodo1SharedPreferences.put(context, "user_device_id", value);
    }

    private static String getDeviceIdForSave(Context context) {
        return Yodo1SharedPreferences.getString(context, "user_device_id");
    }

    private static final class AdvertisingConnection implements ServiceConnection {
        private final LinkedBlockingQueue<IBinder> queue = new LinkedBlockingQueue<>(1);
        boolean retrieved = false;

        @Override
        public void onServiceConnected(ComponentName name, IBinder service) {
            try {
                this.queue.put(service);
            } catch (InterruptedException localInterruptedException) {
            }
        }

        @Override
        public void onServiceDisconnected(ComponentName name) {
        }

        public IBinder getBinder() throws InterruptedException {
            if (this.retrieved) {
                throw new IllegalStateException();
            }
            this.retrieved = true;
            return this.queue.take();
        }
    }

    private static final class AdvertisingInterface implements IInterface {
        private IBinder binder;

        public AdvertisingInterface(IBinder pBinder) {
            binder = pBinder;
        }

        @Override
        public IBinder asBinder() {
            return binder;
        }

        public String getId() throws RemoteException {
            Parcel data = Parcel.obtain();
            Parcel reply = Parcel.obtain();
            String id;
            try {
                data.writeInterfaceToken("com.google.android.gms.ads.identifier.internal.IAdvertisingIdService");
                binder.transact(1, data, reply, 0);
                reply.readException();
                id = reply.readString();
            } finally {
                reply.recycle();
                data.recycle();
            }
            return id;
        }


        public boolean isLimitAdTrackingEnabled(boolean paramBoolean) throws RemoteException {
            Parcel data = Parcel.obtain();
            Parcel reply = Parcel.obtain();
            boolean limitAdTracking;
            try {
                data.writeInterfaceToken("com.google.android.gms.ads.identifier.internal.IAdvertisingIdService");
                data.writeInt(paramBoolean ? 1 : 0);
                binder.transact(2, data, reply, 0);
                reply.readException();
                limitAdTracking = 0 != reply.readInt();
            } finally {
                reply.recycle();
                data.recycle();
            }
            return limitAdTracking;
        }
    }


    /**
     * platform ：平台安卓
     * originalSystemVersion ：系统原生版本 可能是API版本，比如 16 17 28 29，可能是安卓系统版本：安卓7 8 9 10
     * osVersion ：系统自定义UI框架版本 比如华为EMUI ，小米MIUI，OPPO ColorOs等
     * deviceType ：设备型号
     * manufacturer ：设备品牌：使用硬件制造商属性，因为不同品牌有下辖子品牌，比如华为荣耀系列 vivo步步高系列
     * wifi ：是否连接了wifi
     * carrier ：手机卡运营商
     */
    public static synchronized JSONObject getDeviceInfo(Context context) {
        if (deviceInfo != null) {
            return deviceInfo;
        }
        deviceInfo = new JSONObject();
        try {
            deviceInfo.put("platform", "Android");
            deviceInfo.put("originalSystemVersion", Build.VERSION.SDK_INT + "");
            deviceInfo.put("osVersion", check((Build.BRAND + Build.MANUFACTURER).toUpperCase()));
            deviceInfo.put("deviceType", Build.MODEL + "");
            deviceInfo.put("manufacturer", Build.MANUFACTURER + "");
            deviceInfo.put("wifi", Yodo1NetWorkUtils.isWifiConnected(context) ? "true" : "false");
            deviceInfo.put("carrier", Yodo1NetWorkUtils.getCellularOperatorType(context) + "");
        } catch (JSONException e) {
            YLog.i("get device info exception ", e);
        }
        return deviceInfo;
    }

    /**
     * Build.BRAND 系统定制商 例如：荣耀 华为 oppo
     * Build.MODEL 设备型号
     * Build.MANUFACTURER 硬件制造商
     * Build.VERSION.SDK_INT 原生版本
     */
    private static final String ROM_MI = "XIAOMI", ROM_HW = "HUAWEI", ROM_MZ = "MEIZU", ROM_OPPO = "OPPO";
    private static final String ROM_SMARTISAN = "SMARTISAN", ROM_VIVO = "VIVO", ROM_QIKU = "QIKU", ROM_FLYME = "FLYME";
    private static final String KEY_VERSION_MIUI = "ro.miui.ui.version.name", KEY_VERSION_EMUI = "ro.build.version.emui";
    private static final String KEY_VERSION_OPPO = "ro.build.version.opporom", KEY_VERSION_SMARTISAN = "ro.smartisan.version";
    private static final String KEY_VERSION_VIVO = "ro.vivo.os.version";

    public static String osVersion;

    public static String check(String phoneType) {
        YLog.d("phoneType : " + phoneType);
        if (osVersion != null) {
            return osVersion;
        }
        if (phoneType.contains(ROM_MI)) {
            osVersion = getProp(KEY_VERSION_MIUI);
        } else if (phoneType.contains(ROM_HW)) {
            osVersion = getProp(KEY_VERSION_EMUI);
        } else if (phoneType.contains(ROM_OPPO)) {
            osVersion = getProp(KEY_VERSION_OPPO);
        } else if (phoneType.contains(ROM_VIVO)) {
            osVersion = getProp(KEY_VERSION_VIVO);
        } else if (phoneType.contains(ROM_SMARTISAN)) {
            osVersion = getProp(KEY_VERSION_SMARTISAN);
        } else {
            if (Build.DISPLAY.toUpperCase().contains(ROM_FLYME)) {
                osVersion = ROM_FLYME;
            } else {
                osVersion = Build.VERSION.INCREMENTAL;
            }
        }
        return osVersion;
    }

    public static String getProp(String name) {
        String line = null;
        BufferedReader input = null;
        try {
            Process p = Runtime.getRuntime().exec("getprop " + name);
            input = new BufferedReader(new InputStreamReader(p.getInputStream()), 1024);
            line = input.readLine();
            input.close();
        } catch (IOException ex) {
            return null;
        } finally {
            if (input != null) {
                try {
                    input.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return line;
    }
}
