/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.curator;

import ai.vespa.validation.Validation;
import com.yahoo.concurrent.UncheckedTimeoutException;
import com.yahoo.jdisc.Metric;
import com.yahoo.path.Path;
import com.yahoo.protect.Process;
import com.yahoo.vespa.curator.Curator;
import com.yahoo.vespa.curator.Lock;
import com.yahoo.vespa.curator.api.VespaCurator;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;

class SingletonManager {
    private static final Logger logger = Logger.getLogger(SingletonManager.class.getName());
    private static final String partPattern = "[a-zA-Z0-9$_]([a-zA-Z0-9$_-]+){0,63}";
    private static final Pattern idPattern = Pattern.compile("[a-zA-Z0-9$_]([a-zA-Z0-9$_-]+){0,63}(\\.[a-zA-Z0-9$_]([a-zA-Z0-9$_-]+){0,63})*");
    private final Curator curator;
    private final Clock clock;
    private final Duration tickTimeout;
    private final Map<String, Janitor> janitors = new HashMap<String, Janitor>();
    private final Map<String, Integer> count = new HashMap<String, Integer>();
    private final Map<VespaCurator.SingletonWorker, String> registrations = new IdentityHashMap<VespaCurator.SingletonWorker, String>();
    private final Metric metric;

    SingletonManager(Curator curator, Clock clock, Duration tickTimeout, Metric metric) {
        this.curator = curator;
        this.clock = clock;
        this.tickTimeout = tickTimeout;
        this.metric = metric;
    }

    synchronized CompletableFuture<?> register(String singletonId, VespaCurator.SingletonWorker singleton) {
        Validation.requireMatch((String)singletonId, (String)"Singleton ID", (Pattern)idPattern);
        Validation.requireLength((String)singletonId, (String)"Singleton ID", (int)1, (int)255);
        String old = this.registrations.putIfAbsent(singleton, singletonId);
        if (old != null) {
            throw new IllegalArgumentException(singleton + " already registered with ID " + old);
        }
        this.count.merge(singletonId, 1, Integer::sum);
        return this.janitors.computeIfAbsent(singletonId, x$0 -> new Janitor((String)x$0)).register(singleton);
    }

    synchronized CompletableFuture<?> unregister(VespaCurator.SingletonWorker singleton) {
        String id = this.registrations.remove(singleton);
        if (id == null) {
            throw new IllegalArgumentException(singleton + " is not registered");
        }
        return this.janitors.get(id).unregister(singleton).whenComplete((__, ___) -> this.unregistered(id, singleton));
    }

    synchronized void unregistered(String singletonId, VespaCurator.SingletonWorker singleton) {
        this.registrations.remove(singleton);
        if (this.count.merge(singletonId, -1, Integer::sum) > 0) {
            return;
        }
        this.count.remove(singletonId);
        this.janitors.remove(singletonId).shutdown();
    }

    synchronized Optional<Instant> activeUntil(String singletonId) {
        return Optional.ofNullable(this.janitors.get(singletonId)).map(janitor -> janitor.doom.get());
    }

    boolean isActive(String singletonId) {
        return this.activeUntil(singletonId).map(this.clock.instant()::isBefore).orElse(false);
    }

    synchronized void invalidate() {
        for (Janitor janitor : this.janitors.values()) {
            janitor.invalidate();
        }
    }

    public synchronized CompletableFuture<?> shutdown() {
        CompletableFuture[] futures = new CompletableFuture[this.registrations.size()];
        int i = 0;
        for (VespaCurator.SingletonWorker singleton : List.copyOf(this.registrations.keySet())) {
            String id = this.registrations.get(singleton);
            logger.log(Level.WARNING, singleton + " still registered with id '" + id + "' at shutdown");
            futures[i++] = this.unregister(singleton);
        }
        return CompletableFuture.allOf(futures).orTimeout(10L, TimeUnit.SECONDS);
    }

    private class Janitor {
        private static final Instant EMPTY = Instant.ofEpochMilli(-1L);
        private static final Instant INVALID = Instant.ofEpochMilli(-2L);
        final BlockingDeque<Task> tasks = new LinkedBlockingDeque<Task>();
        final Deque<VespaCurator.SingletonWorker> singletons = new ArrayDeque<VespaCurator.SingletonWorker>(2);
        final AtomicReference<Instant> doom = new AtomicReference<Instant>(EMPTY);
        final AtomicBoolean shutdown = new AtomicBoolean();
        final Thread worker;
        final String id;
        final Path path;
        final MetricHelper metrics;
        Lock lock = null;
        boolean active;

        Janitor(String id) {
            this.id = id;
            this.path = Path.fromString((String)("/vespa/singleton/v1/" + id));
            this.worker = new Thread(this::run, "singleton-janitor-" + id);
            this.metrics = new MetricHelper();
            this.worker.setDaemon(true);
            this.worker.start();
        }

        public void unlock() {
            if (this.lock != null) {
                try {
                    logger.log(Level.INFO, "Relinquishing lease for " + this.id);
                    this.lock.close();
                }
                catch (Exception e) {
                    logger.log(Level.WARNING, "Failed closing " + this.lock, e);
                }
            }
            this.doom.set(EMPTY);
            this.lock = null;
        }

        private void run() {
            try {
                while (!this.shutdown.get()) {
                    try {
                        this.renewLease();
                        this.updateStatus();
                        this.doTask();
                        this.metrics.ping();
                    }
                    catch (InterruptedException e) {
                        if (this.shutdown.get()) continue;
                        logger.log(Level.WARNING, this.worker + " interrupted, restarting event loop");
                    }
                }
                this.unlock();
            }
            catch (Throwable t) {
                Process.logAndDie((String)(this.worker + " can't continue, shutting down"), (Throwable)t);
            }
        }

        protected void doTask() throws InterruptedException {
            Task task = this.tasks.poll(SingletonManager.this.tickTimeout.toMillis(), TimeUnit.MILLISECONDS);
            try {
                if (task != null) {
                    switch (task.type) {
                        case register: {
                            this.doRegister(task.singleton);
                            task.future.complete(null);
                            break;
                        }
                        case unregister: {
                            this.doUnregister(task.singleton);
                            task.future.complete(null);
                        }
                    }
                }
            }
            catch (RuntimeException e) {
                logger.log(Level.WARNING, "Exception attempting to " + task.type + " " + task.singleton + " in " + this.worker, e);
                task.future.completeExceptionally(e);
            }
        }

        private void doRegister(VespaCurator.SingletonWorker singleton) {
            logger.log(Level.INFO, "Registering " + singleton + " with ID: " + this.id);
            VespaCurator.SingletonWorker current = this.singletons.peek();
            this.singletons.push(singleton);
            if (this.active) {
                RuntimeException e = null;
                if (current != null) {
                    try {
                        logger.log(Level.INFO, "Deactivating " + current);
                        this.metrics.deactivation(current::deactivate);
                    }
                    catch (RuntimeException f) {
                        e = f;
                    }
                }
                try {
                    logger.log(Level.INFO, "Activating " + singleton);
                    this.metrics.activation(singleton::activate);
                }
                catch (RuntimeException f) {
                    if (e == null) {
                        e = f;
                    }
                    e.addSuppressed(f);
                }
                if (this.singletons.isEmpty()) {
                    logger.log(Level.INFO, "No registered singletons, invalidating lease");
                    this.invalidate();
                }
                if (e != null) {
                    throw e;
                }
            }
        }

        private void doUnregister(VespaCurator.SingletonWorker singleton) {
            logger.log(Level.INFO, "Unregistering " + singleton + " with ID: " + this.id);
            RuntimeException e = null;
            VespaCurator.SingletonWorker current = this.singletons.peek();
            if (!this.singletons.remove(singleton)) {
                return;
            }
            if (this.active && current == singleton) {
                try {
                    logger.log(Level.INFO, "Deactivating " + singleton);
                    this.metrics.deactivation(singleton::deactivate);
                }
                catch (RuntimeException f) {
                    e = f;
                }
                if (!this.singletons.isEmpty()) {
                    try {
                        logger.log(Level.INFO, "Activating " + this.singletons.peek());
                        this.metrics.activation(this.singletons.peek()::activate);
                    }
                    catch (RuntimeException f) {
                        if (e == null) {
                            e = f;
                        }
                        e.addSuppressed(f);
                    }
                }
                if (this.singletons.isEmpty()) {
                    logger.log(Level.INFO, "No registered singletons, invalidating lease");
                    this.invalidate();
                }
            }
            if (e != null) {
                throw e;
            }
        }

        private void renewLease() {
            Instant ourDoom = this.doom.get();
            if (ourDoom == INVALID) {
                logger.log(Level.INFO, "Lease invalidated");
                return;
            }
            Instant start = SingletonManager.this.clock.instant();
            if (this.lock == null) {
                try {
                    this.lock = SingletonManager.this.curator.lock(this.path.append("lock"), SingletonManager.this.tickTimeout);
                    logger.log(Level.INFO, "Acquired lock for ID: " + this.id);
                }
                catch (UncheckedTimeoutException e) {
                    logger.log(Level.FINE, e, () -> "Timed out acquiring lock for '" + this.path + "' within " + SingletonManager.this.tickTimeout);
                    this.cleanOrphans();
                    return;
                }
                catch (RuntimeException e) {
                    logger.log(Level.WARNING, e, () -> "Failed acquiring lock for '" + this.path + "' within " + SingletonManager.this.tickTimeout);
                    this.cleanOrphans();
                    return;
                }
            }
            try {
                SingletonManager.this.curator.set(this.path.append("ping"), new byte[0]);
            }
            catch (RuntimeException e) {
                logger.log(Level.FINE, "Failed pinging ZK cluster", e);
                return;
            }
            if (!this.doom.compareAndSet(ourDoom, start.plus(SingletonManager.this.curator.sessionTimeout().multipliedBy(9L).dividedBy(10L)))) {
                logger.log(Level.INFO, "Deadline changed from " + ourDoom + " to " + this.doom.get() + "; current lease renewal is void");
            }
        }

        private void cleanOrphans() {
            List orphans = null;
            try {
                orphans = SingletonManager.this.curator.framework().getZookeeperClient().getZooKeeper().getEphemerals(this.path.getAbsolute());
                for (String orphan : orphans) {
                    SingletonManager.this.curator.delete(this.path.append(orphan));
                }
            }
            catch (Exception e) {
                logger.log(Level.WARNING, "Failed cleaning orphans: " + orphans, e);
            }
        }

        private void updateStatus() {
            boolean shouldBeActive = this.doom.get().isAfter(SingletonManager.this.clock.instant());
            if (!this.active && shouldBeActive) {
                try {
                    this.active = true;
                    if (!this.singletons.isEmpty()) {
                        this.metrics.activation(this.singletons.peek()::activate);
                    }
                }
                catch (RuntimeException e) {
                    logger.log(Level.WARNING, "Failed to activate " + this.singletons.peek() + ", deactivating again", e);
                    shouldBeActive = false;
                }
            }
            if (!shouldBeActive) {
                logger.log(Level.FINE, () -> "Doom value is " + this.doom.get());
                if (this.active) {
                    try {
                        if (!this.singletons.isEmpty()) {
                            this.metrics.deactivation(this.singletons.peek()::deactivate);
                        }
                        this.active = false;
                    }
                    catch (RuntimeException e) {
                        logger.log(Level.WARNING, "Failed to deactivate " + this.singletons.peek(), e);
                    }
                }
                this.unlock();
            }
        }

        CompletableFuture<?> register(VespaCurator.SingletonWorker singleton) {
            Task task = Task.register(singleton);
            this.tasks.offer(task);
            return task.future;
        }

        CompletableFuture<?> unregister(VespaCurator.SingletonWorker singleton) {
            Task task = Task.unregister(singleton);
            this.tasks.offer(task);
            return task.future;
        }

        void invalidate() {
            this.doom.set(INVALID);
        }

        void shutdown() {
            logger.log(Level.INFO, "Shutting down " + this);
            if (!this.shutdown.compareAndSet(false, true)) {
                logger.log(Level.WARNING, "Shutdown called more than once on " + this);
            }
            if (Thread.currentThread() != this.worker) {
                try {
                    this.worker.join();
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            } else {
                this.unlock();
            }
            if (!this.tasks.isEmpty()) {
                logger.log(Level.WARNING, "Non-empty task list after shutdown: " + this.tasks.size() + " remaining");
                for (Task task : this.tasks) {
                    task.future.cancel(true);
                }
            }
        }

        private class MetricHelper {
            static final String PREFIX = "jdisc.singleton.";
            static final String IS_ACTIVE = "jdisc.singleton.is_active";
            static final String ACTIVATION = "jdisc.singleton.activation.count";
            static final String ACTIVATION_MILLIS = "jdisc.singleton.activation.millis";
            static final String ACTIVATION_FAILURES = "jdisc.singleton.activation.failure.count";
            static final String DEACTIVATION = "jdisc.singleton.deactivation.count";
            static final String DEACTIVATION_MILLIS = "jdisc.singleton.deactivation.millis";
            static final String DEACTIVATION_FAILURES = "jdisc.singleton.deactivation.failure.count";
            final Metric.Context context;
            boolean isActive;

            MetricHelper() {
                this.context = SingletonManager.this.metric.createContext(Map.of("singletonId", Janitor.this.id));
            }

            void ping() {
                SingletonManager.this.metric.set(IS_ACTIVE, (Number)(this.isActive ? 1 : 0), this.context);
            }

            void activation(Runnable activation) {
                Instant start = SingletonManager.this.clock.instant();
                boolean failed = false;
                SingletonManager.this.metric.add(ACTIVATION, (Number)1, this.context);
                logger.log(Level.INFO, "Activating singleton for ID: " + Janitor.this.id);
                try {
                    activation.run();
                }
                catch (RuntimeException e) {
                    try {
                        failed = true;
                        throw e;
                    }
                    catch (Throwable throwable) {
                        long durationMillis = Duration.between(start, SingletonManager.this.clock.instant()).toMillis();
                        SingletonManager.this.metric.set(ACTIVATION_MILLIS, (Number)durationMillis, this.context);
                        logger.log(Level.INFO, "Activation completed " + (failed ? "un" : "") + "successfully in %.3f seconds".formatted((double)durationMillis * 0.001));
                        if (failed) {
                            SingletonManager.this.metric.add(ACTIVATION_FAILURES, (Number)1, this.context);
                        } else {
                            this.isActive = true;
                        }
                        this.ping();
                        throw throwable;
                    }
                }
                long durationMillis = Duration.between(start, SingletonManager.this.clock.instant()).toMillis();
                SingletonManager.this.metric.set(ACTIVATION_MILLIS, (Number)durationMillis, this.context);
                logger.log(Level.INFO, "Activation completed " + (failed ? "un" : "") + "successfully in %.3f seconds".formatted((double)durationMillis * 0.001));
                if (failed) {
                    SingletonManager.this.metric.add(ACTIVATION_FAILURES, (Number)1, this.context);
                } else {
                    this.isActive = true;
                }
                this.ping();
            }

            void deactivation(Runnable deactivation) {
                Instant start = SingletonManager.this.clock.instant();
                boolean failed = false;
                SingletonManager.this.metric.add(DEACTIVATION, (Number)1, this.context);
                logger.log(Level.INFO, "Deactivating singleton for ID: " + Janitor.this.id);
                try {
                    deactivation.run();
                }
                catch (RuntimeException e) {
                    try {
                        failed = true;
                        throw e;
                    }
                    catch (Throwable throwable) {
                        long durationMillis = Duration.between(start, SingletonManager.this.clock.instant()).toMillis();
                        SingletonManager.this.metric.set(DEACTIVATION_MILLIS, (Number)durationMillis, this.context);
                        logger.log(Level.INFO, "Deactivation completed " + (failed ? "un" : "") + "successfully in %.3f seconds".formatted((double)durationMillis * 0.001));
                        if (failed) {
                            SingletonManager.this.metric.add(DEACTIVATION_FAILURES, (Number)1, this.context);
                        }
                        this.isActive = false;
                        this.ping();
                        throw throwable;
                    }
                }
                long durationMillis = Duration.between(start, SingletonManager.this.clock.instant()).toMillis();
                SingletonManager.this.metric.set(DEACTIVATION_MILLIS, (Number)durationMillis, this.context);
                logger.log(Level.INFO, "Deactivation completed " + (failed ? "un" : "") + "successfully in %.3f seconds".formatted((double)durationMillis * 0.001));
                if (failed) {
                    SingletonManager.this.metric.add(DEACTIVATION_FAILURES, (Number)1, this.context);
                }
                this.isActive = false;
                this.ping();
            }
        }

        static class Task {
            final Type type;
            final VespaCurator.SingletonWorker singleton;
            final CompletableFuture<?> future = new CompletableFuture();

            private Task(Type type, VespaCurator.SingletonWorker singleton) {
                this.type = type;
                this.singleton = singleton;
            }

            static Task register(VespaCurator.SingletonWorker singleton) {
                return new Task(Type.register, singleton);
            }

            static Task unregister(VespaCurator.SingletonWorker singleton) {
                return new Task(Type.unregister, singleton);
            }

            static enum Type {
                register,
                unregister;

            }
        }
    }
}

