/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.curator.stats;

import com.yahoo.vespa.curator.stats.LockAttempt;
import com.yahoo.vespa.curator.stats.LockMetrics;
import com.yahoo.vespa.curator.stats.LockStats;
import com.yahoo.vespa.curator.stats.RecordedLockAttempts;
import java.time.Duration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.function.Consumer;
import java.util.logging.Logger;

public class ThreadLockStats {
    private static final Logger logger = Logger.getLogger(ThreadLockStats.class.getName());
    private final Thread thread;
    private final ConcurrentLinkedDeque<LockAttempt> lockAttemptsStack = new ConcurrentLinkedDeque();
    private volatile Optional<RecordedLockAttempts> ongoingRecording = Optional.empty();

    ThreadLockStats(Thread currentThread) {
        this.thread = currentThread;
    }

    public String getThreadName() {
        return this.thread.getName();
    }

    public String getStackTrace() {
        StringBuilder stackTrace = new StringBuilder();
        StackTraceElement[] elements = this.thread.getStackTrace();
        for (int i = 0; i < elements.length; ++i) {
            StackTraceElement element = elements[i];
            stackTrace.append(element.getClassName()).append('.').append(element.getMethodName()).append('(').append(element.getFileName()).append(':').append(element.getLineNumber()).append(")\n");
        }
        return stackTrace.toString();
    }

    public List<LockAttempt> getOngoingLockAttempts() {
        return List.copyOf(this.lockAttemptsStack);
    }

    public Optional<LockAttempt> getTopMostOngoingLockAttempt() {
        return Optional.ofNullable(this.lockAttemptsStack.peekFirst());
    }

    public Optional<LockAttempt> getBottomMostOngoingLockAttempt() {
        return Optional.ofNullable(this.lockAttemptsStack.peekLast());
    }

    public Optional<RecordedLockAttempts> getOngoingRecording() {
        return this.ongoingRecording;
    }

    public void invokingAcquire(String lockPath, Duration timeout) {
        boolean reentry = this.lockAttemptsStack.stream().anyMatch(lockAttempt -> lockAttempt.getLockPath().equals(lockPath));
        if (!reentry) {
            this.testForDeadlock(lockPath);
        }
        LockAttempt lockAttempt2 = LockAttempt.invokingAcquire(this, lockPath, timeout, this.getGlobalLockMetrics(lockPath), reentry);
        LockAttempt lastLockAttempt = this.lockAttemptsStack.peekLast();
        if (lastLockAttempt == null) {
            this.ongoingRecording.ifPresent(recording -> recording.addTopLevelLockAttempt(lockAttempt2));
        } else {
            lastLockAttempt.addNestedLockAttempt(lockAttempt2);
        }
        this.lockAttemptsStack.addLast(lockAttempt2);
    }

    public void acquireFailed() {
        this.removeLastLockAttempt(LockAttempt::acquireFailed);
    }

    public void acquireTimedOut() {
        this.removeLastLockAttempt(LockAttempt::timedOut);
    }

    public void lockAcquired() {
        this.withLastLockAttempt(lockAttempt -> {
            lockAttempt.lockAcquired();
            if (!lockAttempt.isReentry()) {
                LockStats.getGlobal().notifyOfThreadHoldingLock(this.thread, lockAttempt.getLockPath());
            }
        });
    }

    public void preRelease(String path) {
        this.withLastLockAttemptFor(path, lockAttempt -> {
            if (!lockAttempt.isReentry()) {
                LockStats.getGlobal().notifyOfThreadReleasingLock(this.thread, lockAttempt.getLockPath());
            }
            lockAttempt.preRelease();
        });
    }

    public void postRelease(String lockPath) {
        this.removeLastLockAttemptFor(lockPath, LockAttempt::postRelease);
    }

    public void releaseFailed(String lockPath) {
        this.removeLastLockAttemptFor(lockPath, LockAttempt::releaseFailed);
    }

    public void startRecording(String recordId) {
        this.ongoingRecording = Optional.of(RecordedLockAttempts.startRecording(recordId));
    }

    public void stopRecording() {
        if (this.ongoingRecording.isPresent()) {
            RecordedLockAttempts recording = this.ongoingRecording.get();
            this.ongoingRecording = Optional.empty();
            recording.stopRecording();
            LockStats.getGlobal().reportNewStoppedRecording(recording);
        }
    }

    private void testForDeadlock(String pathToAcquire) {
        LockStats globalLockStats = LockStats.getGlobal();
        StringBuilder errorMessage = new StringBuilder().append("Deadlock detected: Thread ").append(this.thread.getName());
        HashSet<Thread> threadsAcquiring = new HashSet<Thread>();
        Thread threadAcquiringLockPath = this.thread;
        String lockPath = pathToAcquire;
        Optional<ThreadLockStats> threadLockStats;
        while (!(threadLockStats = globalLockStats.getThreadLockStatsHolding(lockPath)).isEmpty()) {
            Thread threadHoldingLockPath = threadLockStats.get().thread;
            if (threadAcquiringLockPath == threadHoldingLockPath) {
                return;
            }
            errorMessage.append(", trying to acquire lock ").append(lockPath).append(" held by thread ").append(threadHoldingLockPath.getName());
            if (threadsAcquiring.contains(threadHoldingLockPath)) {
                this.getGlobalLockMetrics(pathToAcquire).incrementDeadlockCount();
                logger.warning(errorMessage.toString());
                return;
            }
            Optional<String> nextLockPath = threadLockStats.get().acquiringLockPath();
            if (nextLockPath.isEmpty()) {
                return;
            }
            threadsAcquiring.add(threadAcquiringLockPath);
            lockPath = nextLockPath.get();
            threadAcquiringLockPath = threadHoldingLockPath;
        }
        return;
    }

    private LockMetrics getGlobalLockMetrics(String lockPath) {
        return LockStats.getGlobal().getLockMetrics(lockPath);
    }

    private Optional<String> acquiringLockPath() {
        return Optional.ofNullable(this.lockAttemptsStack.peekLast()).filter(LockAttempt::isAcquiring).map(LockAttempt::getLockPath);
    }

    private void withLastLockAttempt(Consumer<LockAttempt> lockAttemptConsumer) {
        LockAttempt lockAttempt = this.lockAttemptsStack.peekLast();
        if (lockAttempt == null) {
            logger.warning("Unable to get last lock attempt as the lock attempt stack is empty");
            return;
        }
        lockAttemptConsumer.accept(lockAttempt);
    }

    private void removeLastLockAttempt(Consumer<LockAttempt> completeLockAttempt) {
        LockAttempt lockAttempt = this.lockAttemptsStack.pollLast();
        if (lockAttempt == null) {
            logger.warning("Unable to remove last lock attempt as the lock attempt stack is empty");
            return;
        }
        completeLockAttempt.accept(lockAttempt);
        LockStats.getGlobal().maybeSample(lockAttempt);
    }

    private void withLastLockAttemptFor(String lockPath, Consumer<LockAttempt> consumer) {
        Iterator<LockAttempt> lockAttemptIterator = this.lockAttemptsStack.descendingIterator();
        while (lockAttemptIterator.hasNext()) {
            LockAttempt lockAttempt = lockAttemptIterator.next();
            if (!lockAttempt.getLockPath().equals(lockPath)) continue;
            consumer.accept(lockAttempt);
            return;
        }
        logger.warning("Unable to find any lock attempts for " + lockPath);
    }

    private void removeLastLockAttemptFor(String lockPath, Consumer<LockAttempt> consumer) {
        Iterator<LockAttempt> lockAttemptIterator = this.lockAttemptsStack.descendingIterator();
        while (lockAttemptIterator.hasNext()) {
            LockAttempt lockAttempt = lockAttemptIterator.next();
            if (!lockAttempt.getLockPath().equals(lockPath)) continue;
            lockAttemptIterator.remove();
            consumer.accept(lockAttempt);
            LockStats.getGlobal().maybeSample(lockAttempt);
            return;
        }
        logger.warning("Unable to remove last lock attempt as no locks were found for " + lockPath);
    }
}

