/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.curator;

import com.google.inject.Inject;
import com.yahoo.cloud.config.ConfigserverConfig;
import com.yahoo.io.IOUtils;
import com.yahoo.net.HostName;
import com.yahoo.path.Path;
import com.yahoo.text.Utf8;
import com.yahoo.vespa.curator.CuratorCompletionWaiter;
import com.yahoo.vespa.curator.Lock;
import com.yahoo.vespa.curator.NodeCacheWrapper;
import com.yahoo.vespa.curator.PathChildrenCacheWrapper;
import com.yahoo.vespa.curator.VespaZooKeeperFactory;
import com.yahoo.vespa.defaults.Defaults;
import com.yahoo.vespa.zookeeper.VespaZooKeeperServer;
import java.io.File;
import java.time.Duration;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.function.Function;
import java.util.logging.Logger;
import org.apache.curator.RetryPolicy;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.framework.api.transaction.CuratorTransaction;
import org.apache.curator.framework.api.transaction.CuratorTransactionBridge;
import org.apache.curator.framework.api.transaction.CuratorTransactionFinal;
import org.apache.curator.framework.recipes.atomic.DistributedAtomicLong;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.NodeCacheListener;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheListener;
import org.apache.curator.framework.recipes.locks.InterProcessLock;
import org.apache.curator.framework.recipes.locks.InterProcessMutex;
import org.apache.curator.retry.ExponentialBackoffRetry;
import org.apache.curator.utils.ZookeeperFactory;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.client.ZKClientConfig;
import org.apache.zookeeper.data.Stat;
import org.apache.zookeeper.server.quorum.QuorumPeerConfig;

public class Curator
implements AutoCloseable {
    private static final Logger LOG = Logger.getLogger(Curator.class.getName());
    private static final File ZK_CLIENT_CONFIG_FILE = new File(Defaults.getDefaults().underVespaHome("conf/zookeeper/zookeeper-client.cfg"));
    private static final Duration ZK_SESSION_TIMEOUT = Duration.ofSeconds(30L);
    private static final Duration ZK_CONNECTION_TIMEOUT = Duration.ofSeconds(30L);
    private static final Duration BASE_SLEEP_TIME = Duration.ofSeconds(1L);
    private static final int MAX_RETRIES = 10;
    protected final RetryPolicy retryPolicy;
    private final CuratorFramework curatorFramework;
    private final String connectionSpec;
    private final String zooKeeperEnsembleConnectionSpec;
    private final int zooKeeperEnsembleCount;
    private final ConcurrentHashMap<Path, Lock> locks = new ConcurrentHashMap();

    public static Curator create(String connectionSpec) {
        return new Curator(connectionSpec, connectionSpec, Optional.of(ZK_CLIENT_CONFIG_FILE));
    }

    public static Curator create(String connectionSpec, Optional<File> clientConfigFile) {
        return new Curator(connectionSpec, connectionSpec, clientConfigFile);
    }

    @Inject
    public Curator(ConfigserverConfig configserverConfig, VespaZooKeeperServer server) {
        this(configserverConfig, Optional.of(ZK_CLIENT_CONFIG_FILE));
    }

    Curator(ConfigserverConfig configserverConfig, Optional<File> clientConfigFile) {
        this(Curator.createConnectionSpec(configserverConfig), Curator.createEnsembleConnectionSpec(configserverConfig), clientConfigFile);
    }

    private Curator(String connectionSpec, String zooKeeperEnsembleConnectionSpec, Optional<File> clientConfigFile) {
        this(connectionSpec, zooKeeperEnsembleConnectionSpec, (RetryPolicy retryPolicy) -> CuratorFrameworkFactory.builder().retryPolicy(retryPolicy).sessionTimeoutMs((int)ZK_SESSION_TIMEOUT.toMillis()).connectionTimeoutMs((int)ZK_CONNECTION_TIMEOUT.toMillis()).connectString(connectionSpec).zookeeperFactory((ZookeeperFactory)new VespaZooKeeperFactory(Curator.createClientConfig(clientConfigFile))).dontUseContainerParents().build());
    }

    protected Curator(String connectionSpec, String zooKeeperEnsembleConnectionSpec, Function<RetryPolicy, CuratorFramework> curatorFactory) {
        this(connectionSpec, zooKeeperEnsembleConnectionSpec, curatorFactory, (RetryPolicy)new ExponentialBackoffRetry((int)BASE_SLEEP_TIME.toMillis(), 10));
    }

    private Curator(String connectionSpec, String zooKeeperEnsembleConnectionSpec, Function<RetryPolicy, CuratorFramework> curatorFactory, RetryPolicy retryPolicy) {
        this.connectionSpec = connectionSpec;
        this.retryPolicy = retryPolicy;
        this.curatorFramework = curatorFactory.apply(retryPolicy);
        if (this.curatorFramework != null) {
            Curator.validateConnectionSpec(connectionSpec);
            Curator.validateConnectionSpec(zooKeeperEnsembleConnectionSpec);
            this.addLoggingListener();
            this.curatorFramework.start();
        }
        this.zooKeeperEnsembleConnectionSpec = zooKeeperEnsembleConnectionSpec;
        this.zooKeeperEnsembleCount = zooKeeperEnsembleConnectionSpec.split(",").length;
    }

    private static String createConnectionSpec(ConfigserverConfig configserverConfig) {
        return configserverConfig.zookeeperLocalhostAffinity() ? Curator.createConnectionSpecForLocalhost(configserverConfig) : Curator.createEnsembleConnectionSpec(configserverConfig);
    }

    private static ZKClientConfig createClientConfig(Optional<File> clientConfigFile) {
        if (clientConfigFile.isPresent()) {
            boolean useSecureClient = Boolean.parseBoolean(Curator.getEnvironmentVariable("VESPA_USE_TLS_FOR_ZOOKEEPER_CLIENT").orElse("false"));
            String config = "zookeeper.client.secure=" + useSecureClient + "\n";
            clientConfigFile.get().getParentFile().mkdirs();
            IOUtils.writeFile((File)clientConfigFile.get(), (byte[])Utf8.toBytes((String)config));
            try {
                return new ZKClientConfig(clientConfigFile.get());
            }
            catch (QuorumPeerConfig.ConfigException e) {
                throw new RuntimeException("Unable to create ZooKeeper client config file " + clientConfigFile.get());
            }
        }
        return new ZKClientConfig();
    }

    private static String createEnsembleConnectionSpec(ConfigserverConfig config) {
        StringBuilder connectionSpec = new StringBuilder();
        for (int i = 0; i < config.zookeeperserver().size(); ++i) {
            if (connectionSpec.length() > 0) {
                connectionSpec.append(',');
            }
            ConfigserverConfig.Zookeeperserver server = config.zookeeperserver(i);
            connectionSpec.append(server.hostname());
            connectionSpec.append(':');
            connectionSpec.append(server.port());
        }
        return connectionSpec.toString();
    }

    static String createConnectionSpecForLocalhost(ConfigserverConfig config) {
        String thisServer = HostName.getLocalhost();
        for (int i = 0; i < config.zookeeperserver().size(); ++i) {
            ConfigserverConfig.Zookeeperserver server = config.zookeeperserver(i);
            if (!thisServer.equals(server.hostname())) continue;
            return String.format("%s:%d", server.hostname(), server.port());
        }
        throw new IllegalArgumentException("Unable to create connect string to localhost: There is no localhost server specified in config: " + config);
    }

    private static void validateConnectionSpec(String connectionSpec) {
        if (connectionSpec == null || connectionSpec.isEmpty()) {
            throw new IllegalArgumentException(String.format("Connections spec '%s' is not valid", connectionSpec));
        }
    }

    public String connectionSpec() {
        return this.connectionSpec;
    }

    public DistributedAtomicLong createAtomicCounter(String path) {
        return new DistributedAtomicLong(this.curatorFramework, path, (RetryPolicy)new ExponentialBackoffRetry((int)BASE_SLEEP_TIME.toMillis(), 10));
    }

    public InterProcessLock createMutex(String lockPath) {
        return new InterProcessMutex(this.curatorFramework, lockPath);
    }

    private void addLoggingListener() {
        this.curatorFramework.getConnectionStateListenable().addListener((curatorFramework, connectionState) -> {
            switch (connectionState) {
                case SUSPENDED: {
                    LOG.info("ZK connection state change: SUSPENDED");
                    break;
                }
                case RECONNECTED: {
                    LOG.info("ZK connection state change: RECONNECTED");
                    break;
                }
                case LOST: {
                    LOG.warning("ZK connection state change: LOST");
                }
            }
        });
    }

    public CompletionWaiter getCompletionWaiter(Path waiterPath, int numMembers, String id) {
        return CuratorCompletionWaiter.create(this, waiterPath, id);
    }

    public CompletionWaiter createCompletionWaiter(Path parentPath, String waiterNode, int numMembers, String id) {
        return CuratorCompletionWaiter.createAndInitialize(this, parentPath, waiterNode, id);
    }

    public DirectoryCache createDirectoryCache(String path, boolean cacheData, boolean dataIsCompressed, ExecutorService executorService) {
        return new PathChildrenCacheWrapper(this.framework(), path, cacheData, dataIsCompressed, executorService);
    }

    public FileCache createFileCache(String path, boolean dataIsCompressed) {
        return new NodeCacheWrapper(this.framework(), path, dataIsCompressed);
    }

    public boolean exists(Path path) {
        try {
            return this.framework().checkExists().forPath(path.getAbsolute()) != null;
        }
        catch (Exception e) {
            throw new RuntimeException("Could not check existence of " + path.getAbsolute(), e);
        }
    }

    public void set(Path path, byte[] data) {
        String absolutePath = path.getAbsolute();
        try {
            if (!this.exists(path)) {
                this.framework().create().creatingParentsIfNeeded().forPath(absolutePath, data);
            } else {
                this.framework().setData().forPath(absolutePath, data);
            }
        }
        catch (Exception e) {
            throw new RuntimeException("Could not set data at " + absolutePath, e);
        }
    }

    public boolean create(Path path) {
        if (this.exists(path)) {
            return false;
        }
        String absolutePath = path.getAbsolute();
        try {
            this.framework().create().creatingParentsIfNeeded().forPath(absolutePath, new byte[0]);
        }
        catch (KeeperException.NodeExistsException nodeExistsException) {
        }
        catch (Exception e) {
            throw new RuntimeException("Could not create " + absolutePath, e);
        }
        return true;
    }

    public void createAtomically(Path ... paths) {
        try {
            CuratorTransaction transaction = this.framework().inTransaction();
            for (Path path : paths) {
                if (this.exists(path)) continue;
                transaction = ((CuratorTransactionBridge)transaction.create().forPath(path.getAbsolute(), new byte[0])).and();
            }
            ((CuratorTransactionFinal)transaction).commit();
        }
        catch (Exception e) {
            throw new RuntimeException("Could not create " + Arrays.toString(paths), e);
        }
    }

    public void delete(Path path) {
        try {
            this.framework().delete().guaranteed().deletingChildrenIfNeeded().forPath(path.getAbsolute());
        }
        catch (KeeperException.NoNodeException noNodeException) {
        }
        catch (Exception e) {
            throw new RuntimeException("Could not delete " + path.getAbsolute(), e);
        }
    }

    public List<String> getChildren(Path path) {
        try {
            return (List)this.framework().getChildren().forPath(path.getAbsolute());
        }
        catch (KeeperException.NoNodeException e) {
            return List.of();
        }
        catch (Exception e) {
            throw new RuntimeException("Could not get children of " + path.getAbsolute(), e);
        }
    }

    public Optional<byte[]> getData(Path path) {
        try {
            return Optional.of((byte[])this.framework().getData().forPath(path.getAbsolute()));
        }
        catch (KeeperException.NoNodeException e) {
            return Optional.empty();
        }
        catch (Exception e) {
            throw new RuntimeException("Could not get data at " + path.getAbsolute(), e);
        }
    }

    public Optional<Stat> getStat(Path path) {
        try {
            return Optional.ofNullable((Stat)this.framework().checkExists().forPath(path.getAbsolute()));
        }
        catch (KeeperException.NoNodeException e) {
            return Optional.empty();
        }
        catch (Exception e) {
            throw new RuntimeException("Could not get data at " + path.getAbsolute(), e);
        }
    }

    public Lock lock(Path path, Duration timeout) {
        this.create(path);
        Lock lock = this.locks.computeIfAbsent(path, pathArg -> new Lock(pathArg.getAbsolute(), this));
        lock.acquire(timeout);
        return lock;
    }

    public CuratorFramework framework() {
        return this.curatorFramework;
    }

    @Override
    public void close() {
        this.curatorFramework.close();
    }

    public String zooKeeperEnsembleConnectionSpec() {
        return this.zooKeeperEnsembleConnectionSpec;
    }

    public int zooKeeperEnsembleCount() {
        return this.zooKeeperEnsembleCount;
    }

    private static Optional<String> getEnvironmentVariable(String variableName) {
        return Optional.ofNullable(System.getenv().get(variableName)).filter(var -> !var.isEmpty());
    }

    public static interface FileCache {
        public void start();

        public void addListener(NodeCacheListener var1);

        public ChildData getCurrentData();

        public void close();
    }

    public static interface DirectoryCache {
        public void start();

        public void addListener(PathChildrenCacheListener var1);

        public List<ChildData> getCurrentData();

        public ChildData getCurrentData(Path var1);

        public void close();
    }

    public static interface CompletionWaiter {
        public void awaitCompletion(Duration var1);

        public void notifyCompletion();
    }
}

