/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.compress;

import com.yahoo.compress.CompressionType;
import com.yahoo.compress.ZstdCompressor;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Optional;
import java.util.Random;
import net.jpountz.lz4.LZ4Compressor;
import net.jpountz.lz4.LZ4Factory;
import net.jpountz.lz4.LZ4FastDecompressor;
import net.jpountz.lz4.LZ4SafeDecompressor;

public class Compressor {
    private final ZstdCompressor zstdCompressor = new ZstdCompressor();
    private final CompressionType type;
    private final int level;
    private final double compressionThresholdFactor;
    private final int compressMinSizeBytes;
    private static final LZ4Factory factory = LZ4Factory.fastestInstance();

    public Compressor() {
        this(CompressionType.LZ4);
    }

    public Compressor(CompressionType type) {
        this(type, 9, 0.95, 0);
    }

    public Compressor(CompressionType type, int level) {
        this(type, level, 0.95, 0);
    }

    public Compressor(CompressionType type, int level, double compressionThresholdFactor, int compressMinSizeBytes) {
        this.type = type;
        this.level = level;
        this.compressionThresholdFactor = compressionThresholdFactor;
        this.compressMinSizeBytes = compressMinSizeBytes;
    }

    public CompressionType type() {
        return this.type;
    }

    public int level() {
        return this.level;
    }

    public double compressionThresholdFactor() {
        return this.compressionThresholdFactor;
    }

    public int compressMinSizeBytes() {
        return this.compressMinSizeBytes;
    }

    public Compression compress(CompressionType requestedCompression, byte[] data, Optional<Integer> uncompressedSize) {
        switch (requestedCompression) {
            case NONE: {
                data = uncompressedSize.isPresent() ? Arrays.copyOf(data, (int)uncompressedSize.get()) : data;
                return new Compression(CompressionType.NONE, data.length, data);
            }
            case LZ4: {
                int dataSize;
                int n = dataSize = uncompressedSize.isPresent() ? uncompressedSize.get() : data.length;
                if (dataSize < this.compressMinSizeBytes) {
                    return new Compression(CompressionType.INCOMPRESSIBLE, dataSize, data);
                }
                byte[] compressedData = this.getCompressor().compress(data, 0, dataSize);
                if ((double)(compressedData.length + 8) >= (double)dataSize * this.compressionThresholdFactor) {
                    return new Compression(CompressionType.INCOMPRESSIBLE, dataSize, data);
                }
                return new Compression(CompressionType.LZ4, dataSize, compressedData);
            }
            case ZSTD: {
                int dataLength = uncompressedSize.orElse(data.length);
                if (dataLength < this.compressMinSizeBytes) {
                    return new Compression(CompressionType.INCOMPRESSIBLE, dataLength, data);
                }
                byte[] compressed = this.zstdCompressor.compress(data, 0, dataLength);
                return new Compression(CompressionType.ZSTD, dataLength, compressed);
            }
        }
        throw new IllegalArgumentException(requestedCompression + " is not supported");
    }

    private LZ4Compressor getCompressor() {
        return this.level < 7 ? factory.fastCompressor() : factory.highCompressor();
    }

    public Compression compress(CompressionType requestedCompression, byte[] data) {
        return this.compress(requestedCompression, data, Optional.empty());
    }

    public Compression compress(byte[] data, int uncompressedSize) {
        return this.compress(this.type, data, Optional.of(uncompressedSize));
    }

    public Compression compress(byte[] data) {
        return this.compress(this.type, data, Optional.empty());
    }

    public byte[] decompress(CompressionType compression, byte[] compressedData, int compressedDataOffset, int expectedUncompressedSize, Optional<Integer> expectedCompressedSize) {
        switch (compression) {
            case NONE: 
            case INCOMPRESSIBLE: {
                int endPosition = expectedCompressedSize.isPresent() ? compressedDataOffset + expectedCompressedSize.get() : compressedData.length;
                return Arrays.copyOfRange(compressedData, compressedDataOffset, endPosition);
            }
            case LZ4: {
                byte[] uncompressedLZ4Data = new byte[expectedUncompressedSize];
                int compressedSize = factory.fastDecompressor().decompress(compressedData, compressedDataOffset, uncompressedLZ4Data, 0, expectedUncompressedSize);
                if (expectedCompressedSize.isPresent() && compressedSize != expectedCompressedSize.get()) {
                    throw new IllegalStateException("Compressed size mismatch. Expected " + compressedSize + ". Got " + expectedCompressedSize.get());
                }
                return uncompressedLZ4Data;
            }
            case ZSTD: {
                int compressedLength = expectedCompressedSize.orElseThrow(() -> new IllegalArgumentException("Zstd decompressor requires input size"));
                byte[] decompressedData = this.zstdCompressor.decompress(compressedData, compressedDataOffset, compressedLength);
                expectedCompressedSize.ifPresent(expectedSize -> {
                    if (compressedData.length != expectedSize) {
                        throw new IllegalStateException("Compressed size mismatch. Expected " + expectedSize + ". Got " + decompressedData.length);
                    }
                });
                return decompressedData;
            }
        }
        throw new IllegalArgumentException(compression + " is not supported");
    }

    public byte[] decompress(byte[] compressedData, CompressionType compressionType, int uncompressedSize) {
        return this.decompress(compressionType, compressedData, 0, uncompressedSize, Optional.empty());
    }

    public byte[] decompress(Compression compression) {
        return this.decompress(compression.type(), compression.data(), 0, compression.uncompressedSize(), Optional.empty());
    }

    public byte[] compressUnconditionally(byte[] input) {
        return this.getCompressor().compress(input, 0, input.length);
    }

    public byte[] compressUnconditionally(ByteBuffer input) {
        return this.getCompressor().compress(input.array(), input.arrayOffset() + input.position(), input.remaining());
    }

    public void decompressUnconditionally(ByteBuffer input, ByteBuffer output) {
        if (input.remaining() > 0) {
            factory.fastDecompressor().decompress(input, output);
        }
    }

    public byte[] decompressUnconditionally(byte[] input, int srcOffset, int uncompressedLen) {
        if (input.length > 0) {
            return factory.fastDecompressor().decompress(input, srcOffset, uncompressedLen);
        }
        return new byte[0];
    }

    public long warmup(double seconds) {
        byte[] input = new byte[16384];
        new Random().nextBytes(input);
        long timeDone = System.nanoTime() + (long)(seconds * 1.0E9);
        long compressedBytes = 0L;
        byte[] decompressed = new byte[input.length];
        LZ4FastDecompressor fastDecompressor = factory.fastDecompressor();
        LZ4SafeDecompressor safeDecompressor = factory.safeDecompressor();
        LZ4Compressor fastCompressor = factory.fastCompressor();
        LZ4Compressor highCompressor = factory.highCompressor();
        while (System.nanoTime() < timeDone) {
            byte[] compressedFast = fastCompressor.compress(input);
            byte[] compressedHigh = highCompressor.compress(input);
            fastDecompressor.decompress(compressedFast, decompressed);
            fastDecompressor.decompress(compressedHigh, decompressed);
            safeDecompressor.decompress(compressedFast, decompressed);
            safeDecompressor.decompress(compressedHigh, decompressed);
            compressedBytes += (long)(compressedFast.length + compressedHigh.length);
        }
        return compressedBytes;
    }

    public static class Compression {
        private final CompressionType compressionType;
        private final int uncompressedSize;
        private final byte[] data;

        public Compression(CompressionType compressionType, int uncompressedSize, byte[] data) {
            this.compressionType = compressionType;
            this.uncompressedSize = uncompressedSize;
            this.data = data;
        }

        public CompressionType type() {
            return this.compressionType;
        }

        public int uncompressedSize() {
            return this.uncompressedSize;
        }

        public byte[] data() {
            return this.data;
        }
    }
}

