/*
 * Decompiled with CFR 0.152.
 */
package ai.vespa.metricsproxy.service;

import ai.vespa.metricsproxy.metric.Metric;
import ai.vespa.metricsproxy.metric.Metrics;
import ai.vespa.metricsproxy.metric.model.MetricId;
import ai.vespa.metricsproxy.service.CpuJiffies;
import ai.vespa.metricsproxy.service.VespaService;
import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

public class SystemPoller {
    private static final Logger log = Logger.getLogger(SystemPoller.class.getName());
    private final int pollingIntervalSecs;
    private final List<VespaService> services;
    private final int memoryTypeVirtual = 0;
    private final int memoryTypeResident = 1;
    private final Map<VespaService, Long> lastCpuJiffiesMetrics = new ConcurrentHashMap<VespaService, Long>();
    private final Timer systemPollTimer;
    private long lastTotalCpuJiffies = -1L;

    public SystemPoller(List<VespaService> services, int pollingIntervalSecs) {
        this.services = services;
        this.pollingIntervalSecs = pollingIntervalSecs;
        this.systemPollTimer = new Timer("systemPollTimer", true);
    }

    void stop() {
        this.systemPollTimer.cancel();
    }

    long[] getMemoryUsage(VespaService service) {
        BufferedReader br;
        long[] size = new long[2];
        int pid = service.getPid();
        try {
            br = new BufferedReader(new FileReader("/proc/" + pid + "/smaps"));
        }
        catch (FileNotFoundException ex) {
            service.setAlive(false);
            return size;
        }
        try {
            String line;
            while ((line = br.readLine()) != null) {
                String[] elems = line.split("\\s+");
                if (line.startsWith("Rss:")) {
                    size[1] = size[1] + Long.parseLong(elems[1]) * 1024L;
                    continue;
                }
                if (!line.startsWith("Size:")) continue;
                size[0] = size[0] + Long.parseLong(elems[1]) * 1024L;
            }
            br.close();
        }
        catch (IOException ex) {
            log.log(Level.FINE, "Unable to read line from smaps file", ex);
            return size;
        }
        return size;
    }

    void poll() {
        long startTime = System.currentTimeMillis();
        boolean someAlive = false;
        if (this.services.isEmpty()) {
            this.schedule();
            return;
        }
        log.log(Level.FINE, () -> "Monitoring system metrics for " + this.services.size() + " services");
        long sysJiffies = this.getNormalizedSystemJiffies();
        for (VespaService s : this.services) {
            long last;
            long diff;
            if (s.isAlive()) {
                someAlive = true;
            }
            Metrics metrics = new Metrics();
            log.log(Level.FINE, () -> "Current size of system metrics for service  " + s + " is " + metrics.size());
            long[] size = this.getMemoryUsage(s);
            log.log(Level.FINE, () -> "Updating memory metric for service " + s);
            metrics.add(new Metric(MetricId.toMetricId("memory_virt"), size[0], startTime / 1000L));
            metrics.add(new Metric(MetricId.toMetricId("memory_rss"), size[1], startTime / 1000L));
            long procJiffies = this.getPidJiffies(s);
            if (this.lastTotalCpuJiffies >= 0L && this.lastCpuJiffiesMetrics.containsKey(s) && (diff = procJiffies - (last = this.lastCpuJiffiesMetrics.get(s).longValue())) >= 0L) {
                metrics.add(new Metric(MetricId.toMetricId("cpu"), 100.0 * (double)diff / (double)(sysJiffies - this.lastTotalCpuJiffies), startTime / 1000L));
            }
            this.lastCpuJiffiesMetrics.put(s, procJiffies);
            s.setSystemMetrics(metrics);
        }
        this.lastTotalCpuJiffies = sysJiffies;
        if (!someAlive) {
            this.reschedule(System.currentTimeMillis() - startTime);
        } else {
            this.schedule();
        }
    }

    long getPidJiffies(VespaService service) {
        String line;
        BufferedReader in;
        int pid = service.getPid();
        try {
            in = new BufferedReader(new FileReader("/proc/" + pid + "/stat"));
        }
        catch (FileNotFoundException ex) {
            log.log(Level.FINE, () -> "Unable to find pid " + pid + " in proc directory, for service " + service.getInstanceName());
            service.setAlive(false);
            return 0L;
        }
        try {
            line = in.readLine();
            in.close();
        }
        catch (IOException ex) {
            log.log(Level.FINE, "Unable to read line from process stat file", ex);
            return 0L;
        }
        String[] elems = line.split(" ");
        return Long.parseLong(elems[13]) + Long.parseLong(elems[14]);
    }

    long getNormalizedSystemJiffies() {
        BufferedReader in;
        ArrayList<CpuJiffies> jiffies = new ArrayList<CpuJiffies>();
        CpuJiffies total = null;
        try {
            in = new BufferedReader(new FileReader("/proc/stat"));
        }
        catch (FileNotFoundException ex) {
            log.log(Level.SEVERE, "Unable to open stat file", ex);
            return 0L;
        }
        try {
            String line;
            while ((line = in.readLine()) != null) {
                if (line.startsWith("cpu ")) {
                    total = new CpuJiffies(line);
                    continue;
                }
                if (!line.startsWith("cpu")) continue;
                jiffies.add(new CpuJiffies(line));
            }
            in.close();
        }
        catch (IOException ex) {
            log.log(Level.SEVERE, "Unable to read line from stat file", ex);
            return 0L;
        }
        if (total != null) {
            return total.getTotalJiffies() / (long)jiffies.size();
        }
        return 0L;
    }

    private void schedule(long time) {
        try {
            this.systemPollTimer.schedule((TimerTask)new PollTask(this), time);
        }
        catch (IllegalStateException e) {
            log.info("Tried to schedule task, but timer was already shut down.");
        }
    }

    public void schedule() {
        this.schedule(this.pollingIntervalSecs * 1000);
    }

    private void reschedule(long skew) {
        long sleep = (long)(this.pollingIntervalSecs * 1000) - skew;
        sleep = Math.max(60000L, sleep);
        this.schedule(sleep);
    }

    private static class PollTask
    extends TimerTask {
        private final SystemPoller poller;

        PollTask(SystemPoller poller) {
            this.poller = poller;
        }

        @Override
        public void run() {
            this.poller.poll();
        }
    }
}

