/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.filedistribution;

import com.yahoo.config.FileReference;
import com.yahoo.io.IOUtils;
import com.yahoo.jrt.Int32Value;
import com.yahoo.jrt.Method;
import com.yahoo.jrt.Request;
import com.yahoo.jrt.Supervisor;
import com.yahoo.jrt.Value;
import com.yahoo.security.tls.Capability;
import com.yahoo.vespa.filedistribution.Downloads;
import com.yahoo.vespa.filedistribution.FileReferenceCompressor;
import com.yahoo.vespa.filedistribution.FileReferenceData;
import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.jpountz.xxhash.StreamingXXHash64;
import net.jpountz.xxhash.XXHashFactory;

public class FileReceiver {
    private static final Logger log = Logger.getLogger(FileReceiver.class.getName());
    public static final String RECEIVE_META_METHOD = "filedistribution.receiveFileMeta";
    public static final String RECEIVE_PART_METHOD = "filedistribution.receiveFilePart";
    public static final String RECEIVE_EOF_METHOD = "filedistribution.receiveFileEof";
    private final Supervisor supervisor;
    private final Downloads downloads;
    private final File downloadDirectory;
    private final AtomicInteger nextSessionId = new AtomicInteger(1);
    private final Map<Integer, Session> sessions = new HashMap<Integer, Session>();

    FileReceiver(Supervisor supervisor, Downloads downloads, File downloadDirectory) {
        this.supervisor = supervisor;
        this.downloads = downloads;
        this.downloadDirectory = downloadDirectory;
        this.registerMethods();
    }

    private void registerMethods() {
        this.receiveFileMethod().forEach(arg_0 -> ((Supervisor)this.supervisor).addMethod(arg_0));
    }

    private List<Method> receiveFileMethod() {
        ArrayList<Method> methods = new ArrayList<Method>();
        methods.add(new Method(RECEIVE_META_METHOD, "sssl*", "ii", this::receiveFileMeta).requireCapabilities(new Capability[]{Capability.CLIENT__FILERECEIVER_API}).paramDesc(0, "filereference", "file reference to download").paramDesc(1, "filename", "filename").paramDesc(2, "type", "'file' or 'compressed'").paramDesc(3, "filelength", "length in bytes of file").paramDesc(3, "compressionType", "compression type: gzip, lz4, zstd").returnDesc(0, "ret", "0 if success, 1 otherwise").returnDesc(1, "session-id", "Session id to be used for this transfer"));
        methods.add(new Method(RECEIVE_PART_METHOD, "siix", "i", this::receiveFilePart).requireCapabilities(new Capability[]{Capability.CLIENT__FILERECEIVER_API}).paramDesc(0, "filereference", "file reference to download").paramDesc(1, "session-id", "Session id to be used for this transfer").paramDesc(2, "partid", "relative part number starting at zero").paramDesc(3, "data", "bytes in this part").returnDesc(0, "ret", "0 if success, 1 otherwise"));
        methods.add(new Method(RECEIVE_EOF_METHOD, "silis", "i", this::receiveFileEof).requireCapabilities(new Capability[]{Capability.CLIENT__FILERECEIVER_API}).paramDesc(0, "filereference", "file reference to download").paramDesc(1, "session-id", "Session id to be used for this transfer").paramDesc(2, "crc-code", "crc code (xxhash64)").paramDesc(3, "error-code", "Error code. 0 if none").paramDesc(4, "error-description", "Error description.").returnDesc(0, "ret", "0 if success, 1 if crc mismatch, 2 otherwise"));
        return methods;
    }

    private static void moveFileToDestination(File tempFile, File destination) {
        try {
            Files.move(tempFile.toPath(), destination.toPath(), new CopyOption[0]);
            log.log(Level.FINEST, () -> "File moved from " + tempFile.getAbsolutePath() + " to " + destination.getAbsolutePath());
        }
        catch (FileAlreadyExistsException e) {
            log.log(Level.FINE, () -> "Failed moving file '" + tempFile.getAbsolutePath() + "' to '" + destination.getAbsolutePath() + "', it already exists");
        }
        catch (IOException e) {
            String message = "Failed moving file '" + tempFile.getAbsolutePath() + "' to '" + destination.getAbsolutePath() + "'";
            log.log(Level.SEVERE, message, e);
            throw new RuntimeException(message, e);
        }
        finally {
            FileReceiver.deletePath(tempFile);
        }
    }

    private static void deletePath(File path) {
        if (path == null || !path.exists()) {
            return;
        }
        try {
            if (path.isDirectory()) {
                IOUtils.recursiveDeleteDir((File)path);
            } else {
                Files.delete(path.toPath());
            }
        }
        catch (IOException ioe) {
            log.log(Level.WARNING, "Failed deleting file/dir " + path);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void receiveFileMeta(Request req) {
        log.log(Level.FINEST, () -> "Received method call '" + req.methodName() + "' with parameters : " + req.parameters());
        FileReference reference = new FileReference(req.parameters().get(0).asString());
        String fileName = req.parameters().get(1).asString();
        FileReferenceData.Type type = FileReferenceData.Type.valueOf(req.parameters().get(2).asString());
        long fileSize = req.parameters().get(3).asInt64();
        FileReferenceData.CompressionType compressionType = req.parameters().size() > 4 ? FileReferenceData.CompressionType.valueOf(req.parameters().get(4).asString()) : FileReferenceData.CompressionType.gzip;
        int sessionId = this.nextSessionId.getAndIncrement();
        int retval = 0;
        Map<Integer, Session> map = this.sessions;
        synchronized (map) {
            if (this.sessions.containsKey(sessionId)) {
                retval = 1;
                log.severe("Session id " + sessionId + " already exist, impossible. Request from " + req.target());
            } else {
                try {
                    this.sessions.put(sessionId, new Session(this.downloadDirectory, sessionId, reference, type, compressionType, fileName, fileSize));
                }
                catch (Exception e) {
                    retval = 1;
                }
            }
        }
        req.returnValues().add((Value)new Int32Value(retval));
        req.returnValues().add((Value)new Int32Value(sessionId));
    }

    private void receiveFilePart(Request req) {
        log.log(Level.FINEST, () -> "Received method call '" + req.methodName() + "' with parameters : " + req.parameters());
        FileReference reference = new FileReference(req.parameters().get(0).asString());
        int sessionId = req.parameters().get(1).asInt32();
        int partId = req.parameters().get(2).asInt32();
        byte[] part = req.parameters().get(3).asData();
        Session session = this.getSession(sessionId);
        int retval = FileReceiver.verifySession(session, sessionId, reference);
        if (retval == 0) {
            try {
                session.addPart(partId, part);
            }
            catch (Exception e) {
                log.severe("Got exception " + e);
                retval = 1;
            }
            double completeness = (double)session.currentFileSize / (double)session.fileSize;
            log.log(Level.FINEST, () -> String.format("%.1f percent of '%s' downloaded", completeness * 100.0, reference.value()));
            this.downloads.setDownloadStatus(reference, completeness);
        }
        req.returnValues().add((Value)new Int32Value(retval));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void receiveFileEof(Request req) {
        log.log(Level.FINEST, () -> "Received method call '" + req.methodName() + "' with parameters : " + req.parameters());
        FileReference reference = new FileReference(req.parameters().get(0).asString());
        int sessionId = req.parameters().get(1).asInt32();
        long xxhash = req.parameters().get(2).asInt64();
        Session session = this.getSession(sessionId);
        int retval = FileReceiver.verifySession(session, sessionId, reference);
        File file = session.close(xxhash);
        this.downloads.completedDownloading(reference, file);
        Map<Integer, Session> map = this.sessions;
        synchronized (map) {
            this.sessions.remove(sessionId);
        }
        req.returnValues().add((Value)new Int32Value(retval));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Session getSession(Integer sessionId) {
        Map<Integer, Session> map = this.sessions;
        synchronized (map) {
            return this.sessions.get(sessionId);
        }
    }

    private static int verifySession(Session session, int sessionId, FileReference reference) {
        if (session == null) {
            log.severe("session-id " + sessionId + " does not exist.");
            return 1;
        }
        if (!session.reference.equals((Object)reference)) {
            log.severe("Session " + session.sessionId + " expects reference " + reference.value() + ", but was " + session.reference.value());
            return 1;
        }
        return 0;
    }

    static final class Session {
        private final StreamingXXHash64 hasher = XXHashFactory.fastestInstance().newStreamingHash64(0L);
        private final int sessionId;
        private final FileReference reference;
        private final FileReferenceData.Type fileType;
        private final FileReferenceData.CompressionType compressionType;
        private final String fileName;
        private final long fileSize;
        private long currentFileSize;
        private long currentPartId;
        private final long currentHash;
        private final File fileReferenceDir;
        private final File tmpDir;
        private final File inProgressDir;
        private final File file;

        Session(File downloadDirectory, int sessionId, FileReference reference, FileReferenceData.Type fileType, FileReferenceData.CompressionType compressionType, String fileName, long fileSize) {
            this.sessionId = sessionId;
            this.reference = reference;
            this.fileType = fileType;
            this.compressionType = compressionType;
            this.fileName = fileName;
            this.fileSize = fileSize;
            this.currentFileSize = 0L;
            this.currentPartId = 0L;
            this.currentHash = 0L;
            this.fileReferenceDir = new File(downloadDirectory, reference.value());
            this.tmpDir = downloadDirectory;
            try {
                this.inProgressDir = Files.createTempDirectory(this.tmpDir.toPath(), "inprogress", new FileAttribute[0]).toFile();
                this.file = new File(this.inProgressDir, fileName);
            }
            catch (IOException e) {
                String msg = "Failed creating temp file for inprogress file for " + fileName + " in '" + this.tmpDir.toPath() + "': ";
                log.log(Level.SEVERE, msg + e.getMessage(), e);
                throw new RuntimeException(msg, e);
            }
        }

        void addPart(int partId, byte[] part) {
            if ((long)partId != this.currentPartId) {
                throw new IllegalStateException("Received partid " + partId + " while expecting " + this.currentPartId);
            }
            if (this.fileSize < this.currentFileSize + (long)part.length) {
                throw new IllegalStateException("Received part would extend the file from " + this.currentFileSize + " to " + (this.currentFileSize + (long)part.length) + ", but " + this.fileSize + " is max.");
            }
            try {
                Files.write(this.file.toPath(), part, StandardOpenOption.WRITE, this.file.exists() ? StandardOpenOption.APPEND : StandardOpenOption.CREATE);
            }
            catch (IOException e) {
                String message = "Failed writing to file (" + this.inProgressDir.toPath() + "): ";
                log.log(Level.SEVERE, message + e.getMessage(), e);
                boolean successfulDelete = this.inProgressDir.delete();
                if (!successfulDelete) {
                    log.log(Level.INFO, "Unable to delete " + this.inProgressDir.toPath());
                }
                throw new RuntimeException(message, e);
            }
            this.currentFileSize += (long)part.length;
            ++this.currentPartId;
            this.hasher.update(part, 0, part.length);
        }

        File close(long hash) {
            File decompressedDir;
            block5: {
                this.verifyHash(hash);
                decompressedDir = null;
                try {
                    if (this.fileType == FileReferenceData.Type.file) {
                        log.log(Level.FINE, () -> "Uncompressed file, moving to " + this.file.getAbsolutePath());
                        FileReceiver.moveFileToDestination(this.inProgressDir, this.fileReferenceDir);
                        break block5;
                    }
                    decompressedDir = Files.createTempDirectory(this.tmpDir.toPath(), "archive", new FileAttribute[0]).toFile();
                    log.log(Level.FINEST, () -> "compression type to use=" + this.compressionType);
                    new FileReferenceCompressor(this.fileType, this.compressionType).decompress(this.file, decompressedDir);
                    FileReceiver.moveFileToDestination(decompressedDir, this.fileReferenceDir);
                }
                catch (IOException e) {
                    try {
                        log.log(Level.SEVERE, "Failed writing file: " + e.getMessage(), e);
                        throw new RuntimeException("Failed writing file: ", e);
                    }
                    catch (Throwable throwable) {
                        FileReceiver.deletePath(this.inProgressDir);
                        FileReceiver.deletePath(decompressedDir);
                        throw throwable;
                    }
                }
            }
            FileReceiver.deletePath(this.inProgressDir);
            FileReceiver.deletePath(decompressedDir);
            return new File(this.fileReferenceDir, this.fileName);
        }

        double percentageReceived() {
            return (double)this.currentFileSize / (double)this.fileSize;
        }

        void verifyHash(long hash) {
            if (this.hasher.getValue() != hash) {
                throw new RuntimeException("xxhash from content (" + this.currentHash + ") is not equal to xxhash in request (" + hash + ")");
            }
        }
    }
}

