/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.documentapi;

import com.yahoo.document.BucketId;
import com.yahoo.document.BucketIdFactory;
import com.yahoo.document.select.BucketSelector;
import com.yahoo.document.select.BucketSet;
import com.yahoo.document.select.parser.ParseException;
import com.yahoo.documentapi.ProgressToken;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;

public class VisitorIterator {
    private final ProgressToken progressToken;
    private final BucketSource bucketSource;
    private int distributionBitCount;
    private static final Logger log = Logger.getLogger(VisitorIterator.class.getName());

    private VisitorIterator(ProgressToken progressToken, BucketSource bucketSource) {
        assert (progressToken.getDistributionBitCount() == bucketSource.getDistributionBitCount()) : "inconsistent distribution bit counts";
        this.distributionBitCount = progressToken.getDistributionBitCount();
        this.progressToken = progressToken;
        this.bucketSource = bucketSource;
    }

    public BucketProgress getNext() {
        assert (this.progressToken.getDistributionBitCount() == this.bucketSource.getDistributionBitCount()) : "inconsistent distribution bit counts for progress and source";
        assert (this.hasNext());
        if (this.progressToken.hasPending()) {
            TreeMap<ProgressToken.BucketKeyWrapper, ProgressToken.BucketEntry> buckets = this.progressToken.getBuckets();
            ProgressToken.BucketEntry pending = null;
            BucketId superbucket = null;
            for (Map.Entry<ProgressToken.BucketKeyWrapper, ProgressToken.BucketEntry> entry : buckets.entrySet()) {
                if (entry.getValue().getState() != ProgressToken.BucketState.BUCKET_PENDING) continue;
                pending = entry.getValue();
                superbucket = new BucketId(ProgressToken.keyToBucketId(entry.getKey().getKey()));
                break;
            }
            assert (pending != null) : "getNext() called with inconsistent state";
            pending.setState(ProgressToken.BucketState.BUCKET_ACTIVE);
            this.progressToken.setActiveBucketCount(this.progressToken.getActiveBucketCount() + 1L);
            this.progressToken.setPendingBucketCount(this.progressToken.getPendingBucketCount() - 1L);
            return new BucketProgress(superbucket, pending.getProgress());
        }
        BucketProgress ret = this.bucketSource.getNext();
        this.progressToken.addBucket(ret.getSuperbucket(), ret.getProgress(), ProgressToken.BucketState.BUCKET_ACTIVE);
        return ret;
    }

    public boolean hasNext() {
        return (this.progressToken.hasPending() || this.bucketSource.hasNext()) && !this.bucketSource.shouldYield();
    }

    public boolean isDone() {
        return !this.hasNext() && !this.progressToken.hasActive();
    }

    public void update(BucketId superbucket, BucketId progress) {
        this.bucketSource.update(superbucket, progress, this.progressToken);
    }

    public long getRemainingBucketCount() {
        return this.progressToken.getTotalBucketCount() - this.progressToken.getFinishedBucketCount();
    }

    protected BucketSource getBucketSource() {
        return this.bucketSource;
    }

    public ProgressToken getProgressToken() {
        return this.progressToken;
    }

    public int getDistributionBitCount() {
        return this.distributionBitCount;
    }

    public void setDistributionBitCount(int distBits) {
        if (this.distributionBitCount != distBits) {
            this.bucketSource.setDistributionBitCount(distBits, this.progressToken);
            this.distributionBitCount = distBits;
            if (log.isLoggable(Level.FINE)) {
                log.log(Level.FINE, "Set visitor iterator distribution bit count to " + distBits);
            }
        }
    }

    public boolean visitsAllBuckets() {
        return this.bucketSource.visitsAllBuckets();
    }

    public static VisitorIterator createFromDocumentSelection(String documentSelection, BucketIdFactory idFactory, int distributionBitCount, ProgressToken progress) throws ParseException {
        return VisitorIterator.createFromDocumentSelection(documentSelection, idFactory, distributionBitCount, progress, 1, 0);
    }

    public static VisitorIterator createFromDocumentSelection(String documentSelection, BucketIdFactory idFactory, int distributionBitCount, ProgressToken progress, int slices, int sliceId) throws ParseException {
        BucketSelector bucketSel = new BucketSelector(idFactory);
        BucketSet rawBuckets = bucketSel.getBucketList(documentSelection);
        BucketSource src = rawBuckets == null ? new DistributionRangeBucketSource(distributionBitCount, progress, slices, sliceId) : new ExplicitBucketSource((Set<BucketId>)rawBuckets, distributionBitCount, progress);
        return new VisitorIterator(progress, src);
    }

    public static VisitorIterator createFromExplicitBucketSet(Set<BucketId> bucketsToVisit, int distributionBitCount, ProgressToken progress) {
        ExplicitBucketSource src = new ExplicitBucketSource(bucketsToVisit, distributionBitCount, progress);
        return new VisitorIterator(progress, src);
    }

    protected static class ExplicitBucketSource
    implements BucketSource {
        private int distributionBitCount;
        private long totalBucketCount = 0L;

        public ExplicitBucketSource(Set<BucketId> superbuckets, int distributionBitCount, ProgressToken progress) {
            this.distributionBitCount = progress.getDistributionBitCount();
            this.totalBucketCount = superbuckets.size();
            if (progress.getTotalBucketCount() == 0L) {
                progress.setTotalBucketCount(this.totalBucketCount);
                progress.setDistributionBitCount(distributionBitCount);
                this.distributionBitCount = distributionBitCount;
            } else {
                if (progress.getTotalBucketCount() != this.totalBucketCount || progress.getFinishedBucketCount() + progress.getPendingBucketCount() + progress.getActiveBucketCount() != this.totalBucketCount) {
                    throw new IllegalArgumentException("Total bucket count in existing progress is not consistent with that of the current document selection");
                }
                if (progress.getBucketCursor() != 0L) {
                    throw new IllegalArgumentException("Cannot use given progress file with the current document selection");
                }
                this.distributionBitCount = progress.getDistributionBitCount();
            }
            if (progress.isFinished() || !progress.isEmpty()) {
                return;
            }
            for (BucketId id : superbuckets) {
                progress.addBucket(id, new BucketId(), ProgressToken.BucketState.BUCKET_PENDING);
            }
        }

        @Override
        public boolean hasNext() {
            return false;
        }

        @Override
        public boolean shouldYield() {
            return false;
        }

        @Override
        public boolean visitsAllBuckets() {
            return false;
        }

        @Override
        public long getTotalBucketCount() {
            return this.totalBucketCount;
        }

        @Override
        public BucketProgress getNext() {
            throw new IllegalStateException("getNext() called on ExplicitBucketSource");
        }

        @Override
        public int getDistributionBitCount() {
            return this.distributionBitCount;
        }

        @Override
        public void setDistributionBitCount(int distributionBitCount, ProgressToken progress) {
            progress.setDistributionBitCount(distributionBitCount);
            this.distributionBitCount = distributionBitCount;
            if (log.isLoggable(Level.FINE)) {
                log.log(Level.FINE, "Set distribution bit count to " + distributionBitCount + " for explicit bucket source (no-op)");
            }
        }

        @Override
        public void update(BucketId superbucket, BucketId progress, ProgressToken token) {
            token.updateProgress(superbucket, progress);
        }
    }

    protected static class DistributionRangeBucketSource
    implements BucketSource {
        private boolean flushActive = false;
        private int distributionBitCount;
        private final int slices;
        private final int sliceId;
        private ProgressToken progressToken;

        public DistributionRangeBucketSource(int distributionBitCount, ProgressToken progress, int slices, int sliceId) {
            if (slices < 1) {
                throw new IllegalArgumentException("slices must be positive, but was " + slices);
            }
            if (sliceId < 0 || sliceId >= slices) {
                throw new IllegalArgumentException("sliceId must be in [0, " + slices + "), but was " + sliceId);
            }
            this.slices = slices;
            this.sliceId = sliceId;
            this.progressToken = progress;
            if (this.progressToken.getTotalBucketCount() == 0L) {
                assert (this.progressToken.isEmpty()) : "inconsistent progress state";
                this.progressToken.setTotalBucketCount(1L << distributionBitCount);
                this.progressToken.setDistributionBitCount(distributionBitCount);
                this.progressToken.setBucketCursor(0L);
                this.progressToken.setFinishedBucketCount(0L);
                this.distributionBitCount = distributionBitCount;
            } else {
                this.distributionBitCount = this.progressToken.getDistributionBitCount();
                if (this.progressToken.getTotalBucketCount() != 1L << this.progressToken.getDistributionBitCount()) {
                    throw new IllegalArgumentException("Total bucket count in existing progress is not consistent with that of the current document selection");
                }
            }
            if (!progress.isFinished()) {
                if (log.isLoggable(Level.FINE)) {
                    log.log(Level.FINE, "Importing unfinished progress token with bits: " + this.progressToken.getDistributionBitCount() + ", active: " + this.progressToken.getActiveBucketCount() + ", pending: " + this.progressToken.getPendingBucketCount() + ", cursor: " + this.progressToken.getBucketCursor() + ", finished: " + this.progressToken.getFinishedBucketCount() + ", total: " + this.progressToken.getTotalBucketCount());
                }
                if (!progress.isEmpty()) {
                    if (this.progressToken.getActiveBucketCount() > 0L) {
                        if (log.isLoggable(Level.FINE)) {
                            log.log(Level.FINE, "Progress token had active buckets upon range construction. Setting these as pending");
                        }
                        this.progressToken.setAllBucketsToState(ProgressToken.BucketState.BUCKET_PENDING);
                    }
                    this.correctInconsistentPending(this.progressToken.getDistributionBitCount());
                    this.correctTruncatedBucketCursor();
                    if (log.isLoggable(Level.FINE)) {
                        log.log(Level.FINE, "Partial bucket space progress; continuing from position " + this.progressToken.getBucketCursor());
                    }
                }
                this.progressToken.setFinishedBucketCount(this.progressToken.getBucketCursor() - this.progressToken.getPendingBucketCount());
            } else assert (this.progressToken.getBucketCursor() == this.progressToken.getTotalBucketCount());
            this.progressToken.setInconsistentState(false);
            this.skipToSlice();
        }

        protected boolean isLosslessResetPossible() {
            if (this.progressToken.getPendingBucketCount() != this.progressToken.getBucketCursor()) {
                return false;
            }
            for (Map.Entry<ProgressToken.BucketKeyWrapper, ProgressToken.BucketEntry> entry : this.progressToken.getBuckets().entrySet()) {
                if (entry.getValue().getState() != ProgressToken.BucketState.BUCKET_PENDING) {
                    return false;
                }
                if (entry.getValue().getProgress().getId() == 0L) continue;
                return false;
            }
            return true;
        }

        private void correctInconsistentPending(int targetDistBits) {
            boolean maybeInconsistent = true;
            long bucketsSplit = 0L;
            long bucketsMerged = 0L;
            long pendingBefore = this.progressToken.getPendingBucketCount();
            ProgressToken p = this.progressToken;
            if (this.isLosslessResetPossible()) {
                if (log.isLoggable(Level.FINE)) {
                    log.log(Level.FINE, "At start of bucket space and all buckets have no progress; doing a lossless reset instead of splitting/merging");
                }
                assert (p.getActiveBucketCount() == 0L);
                p.clearAllBuckets();
                p.setBucketCursor(0L);
                this.skipToSlice();
                return;
            }
            while (maybeInconsistent) {
                BucketId pending;
                BucketId lastMergedBucket = null;
                maybeInconsistent = false;
                TreeMap<ProgressToken.BucketKeyWrapper, ProgressToken.BucketEntry> buckets = new TreeMap<ProgressToken.BucketKeyWrapper, ProgressToken.BucketEntry>((SortedMap<ProgressToken.BucketKeyWrapper, ProgressToken.BucketEntry>)p.getBuckets());
                for (Map.Entry<ProgressToken.BucketKeyWrapper, ProgressToken.BucketEntry> entry : buckets.entrySet()) {
                    assert (entry.getValue().getState() == ProgressToken.BucketState.BUCKET_PENDING);
                    pending = new BucketId(ProgressToken.keyToBucketId(entry.getKey().getKey()));
                    if (pending.getUsedBits() >= targetDistBits) continue;
                    if (pending.getUsedBits() + 1 < targetDistBits) {
                        maybeInconsistent = true;
                    }
                    p.splitPendingBucket(pending);
                    ++bucketsSplit;
                }
                buckets = new TreeMap<ProgressToken.BucketKeyWrapper, ProgressToken.BucketEntry>((SortedMap<ProgressToken.BucketKeyWrapper, ProgressToken.BucketEntry>)p.getBuckets());
                for (Map.Entry<ProgressToken.BucketKeyWrapper, ProgressToken.BucketEntry> entry : buckets.entrySet()) {
                    BucketId rightCheck;
                    assert (entry.getValue().getState() == ProgressToken.BucketState.BUCKET_PENDING);
                    pending = new BucketId(ProgressToken.keyToBucketId(entry.getKey().getKey()));
                    if (pending.getUsedBits() <= targetDistBits) continue;
                    if (lastMergedBucket != null && pending.equals((Object)(rightCheck = new BucketId(lastMergedBucket.getUsedBits(), lastMergedBucket.getId() | 1L << lastMergedBucket.getUsedBits() - 1)))) {
                        if (!log.isLoggable(Level.FINEST)) continue;
                        log.log(Level.FINEST, "Skipped " + pending + ", as it was right sibling of " + lastMergedBucket);
                        continue;
                    }
                    if (pending.getUsedBits() - 1 > targetDistBits) {
                        maybeInconsistent = true;
                    }
                    p.mergePendingBucket(pending);
                    ++bucketsMerged;
                    lastMergedBucket = pending;
                }
            }
            if ((bucketsSplit > 0L || bucketsMerged > 0L) && log.isLoggable(Level.FINE)) {
                log.log(Level.FINE, "Existing progress' pending buckets had inconsistent distribution bits; performed " + bucketsSplit + " split ops and " + bucketsMerged + " merge ops. Pending: " + pendingBefore + " -> " + p.getPendingBucketCount());
            }
        }

        private void correctTruncatedBucketCursor() {
            for (ProgressToken.BucketKeyWrapper bucketKey : this.progressToken.getBuckets().keySet()) {
                BucketId bid = bucketKey.toBucketId();
                long idx = bucketKey.getKey() >>> 64 - bid.getUsedBits();
                if (bid.getUsedBits() != this.distributionBitCount || idx < this.progressToken.getBucketCursor()) continue;
                this.progressToken.setBucketCursor(idx + 1L);
            }
            if (log.isLoggable(Level.FINEST)) {
                log.log(Level.FINEST, "New range bucket cursor is " + this.progressToken.getBucketCursor());
            }
        }

        @Override
        public boolean hasNext() {
            long nextBucket = this.progressToken.getBucketCursor();
            if (this.distributionBitCount != 1) {
                nextBucket += (long)Math.floorMod((long)this.sliceId - nextBucket, this.slices);
            }
            return nextBucket < 1L << this.distributionBitCount;
        }

        @Override
        public boolean shouldYield() {
            return this.flushActive;
        }

        @Override
        public boolean visitsAllBuckets() {
            return true;
        }

        @Override
        public long getTotalBucketCount() {
            return 1L << this.distributionBitCount;
        }

        @Override
        public BucketProgress getNext() {
            assert (this.hasNext()) : "getNext() called with hasNext() == false";
            BucketProgress progress = new BucketProgress(this.progressToken.getCurrentBucketId(), new BucketId());
            this.progressToken.setBucketCursor(this.progressToken.getBucketCursor() + 1L);
            this.skipToSlice();
            return progress;
        }

        private void skipToSlice() {
            if (this.distributionBitCount == 1) {
                return;
            }
            while (this.progressToken.getBucketCursor() < this.getTotalBucketCount() && this.progressToken.getBucketCursor() % (long)this.slices != (long)this.sliceId) {
                this.progressToken.skipCurrentBucket();
            }
        }

        @Override
        public int getDistributionBitCount() {
            return this.distributionBitCount;
        }

        @Override
        public void setDistributionBitCount(int distributionBitCount, ProgressToken progress) {
            this.distributionBitCount = distributionBitCount;
            if (this.progressToken.getActiveBucketCount() > 0L) {
                this.flushActive = true;
                if (log.isLoggable(Level.FINE)) {
                    log.log(Level.FINE, "Holding off new/pending buckets and consistency correction until all " + progress.getActiveBucketCount() + " active buckets have been updated");
                }
                this.progressToken.setInconsistentState(true);
            } else {
                int delta = distributionBitCount - this.progressToken.getDistributionBitCount();
                this.correctInconsistentPending(distributionBitCount);
                if (delta > 0) {
                    if (log.isLoggable(Level.FINE)) {
                        log.log(Level.FINE, "Increasing distribution bits for full bucket space range source from " + this.progressToken.getDistributionBitCount() + " to " + distributionBitCount);
                    }
                    this.progressToken.setFinishedBucketCount(this.progressToken.getFinishedBucketCount() << delta);
                    this.progressToken.setBucketCursor(this.progressToken.getBucketCursor() << delta);
                } else if (delta < 0) {
                    if (log.isLoggable(Level.FINE)) {
                        log.log(Level.FINE, "Decreasing distribution bits for full bucket space range source from " + this.progressToken.getDistributionBitCount() + " to " + distributionBitCount + " bits");
                    }
                    this.progressToken.setBucketCursor(this.progressToken.getBucketCursor() >>> -delta);
                    this.progressToken.setFinishedBucketCount(this.progressToken.getFinishedBucketCount() >>> -delta);
                }
                this.progressToken.setTotalBucketCount(1L << distributionBitCount);
                this.progressToken.setDistributionBitCount(distributionBitCount);
                this.correctTruncatedBucketCursor();
                this.progressToken.setInconsistentState(false);
            }
        }

        @Override
        public void update(BucketId superbucket, BucketId progress, ProgressToken token) {
            this.progressToken.updateProgress(superbucket, progress);
            if (superbucket.getUsedBits() != this.distributionBitCount) {
                if (!progress.equals((Object)ProgressToken.FINISHED_BUCKET)) {
                    assert (this.flushActive);
                    if (log.isLoggable(Level.FINE)) {
                        log.log(Level.FINE, "Received non-finished bucket " + superbucket + " with wrong distribution bit count (" + superbucket.getUsedBits() + "). Waiting to correct until all active are done");
                    }
                } else if (log.isLoggable(Level.FINE)) {
                    log.log(Level.FINE, "Received finished bucket " + superbucket + " with wrong distribution bit count (" + superbucket.getUsedBits() + "). Waiting to correct until all active are done");
                }
            }
            if (this.progressToken.getActiveBucketCount() == 0L) {
                if (this.flushActive) {
                    if (log.isLoggable(Level.FINE)) {
                        log.log(Level.FINE, "All active buckets flushed, correcting progress token and continuing normal operation");
                    }
                    this.setDistributionBitCount(this.distributionBitCount, this.progressToken);
                    assert (this.progressToken.getDistributionBitCount() == this.distributionBitCount);
                }
                this.flushActive = false;
                if (this.progressToken.getPendingBucketCount() <= this.progressToken.getBucketCursor()) {
                    this.progressToken.setFinishedBucketCount(this.progressToken.getBucketCursor() - this.progressToken.getPendingBucketCount());
                }
            }
        }
    }

    protected static interface BucketSource {
        public boolean hasNext();

        public boolean shouldYield();

        public boolean visitsAllBuckets();

        public BucketProgress getNext();

        public long getTotalBucketCount();

        public int getDistributionBitCount();

        public void setDistributionBitCount(int var1, ProgressToken var2);

        public void update(BucketId var1, BucketId var2, ProgressToken var3);
    }

    public static class BucketProgress {
        private BucketId superbucket;
        private BucketId progress;

        public BucketProgress(BucketId superbucket, BucketId progress) {
            this.superbucket = superbucket;
            this.progress = progress;
        }

        public BucketId getProgress() {
            return this.progress;
        }

        public BucketId getSuperbucket() {
            return this.superbucket;
        }
    }
}

