/*
 * Copyright (c) 2016 riontech-xten
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.xtensolution.support.ui.adapter;

import android.content.Context;
import android.view.LayoutInflater;
import android.widget.BaseAdapter;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by MIT on 10/21/2016.
 */
public abstract class BaseListAdapter<T> extends BaseAdapter {
    private static final String TAG = BaseListAdapter.class.getSimpleName();
    protected LayoutInflater inflater;
    protected Context context;
    protected List<T> objectsList;
    protected int displayItemCount;
    private PaginationListener paginationListener;
    private boolean isPageLoading;

    public BaseListAdapter(Context context) {
        this(context, new ArrayList<T>());
    }

    public BaseListAdapter(Context context, List<T> objectsList) {
        this.context = context;
        this.objectsList = objectsList;
        this.inflater = LayoutInflater.from(context);
        displayItemCount = objectsList.size();
    }

    public void setPageLoading(boolean pageLoading) {
        isPageLoading = pageLoading;
    }

    public boolean isPageLoading() {
        return isPageLoading;
    }

    public void setDisplayItemCount(int itemCount) {
        this.displayItemCount = itemCount;
    }

    public void setPaginationListener(PaginationListener paginationListener) {
        this.paginationListener = paginationListener;
    }

    @Override
    public int getItemViewType(int position) {
        return super.getItemViewType(position);
    }

    @Override
    public int getCount() {
        if (objectsList.size() < displayItemCount)
            displayItemCount = objectsList.size();
        return displayItemCount;
    }

    public T getItem(int position) {
        return objectsList.get(position);
    }

    @Override
    public long getItemId(int position) {
        return position;
    }

    public void updateList(List<T> newData) {
//        objectsList.clear();
        objectsList = newData;
        displayItemCount = objectsList.size();
        notifyDataSetChanged();
    }

    public void add(T item) {
        objectsList.add(item);
        displayItemCount = objectsList.size();
        notifyDataSetChanged();
    }

    public void addAt(int position, T item) {
        objectsList.add(position, item);
        displayItemCount = objectsList.size();
        notifyDataSetChanged();
    }

    public void addList(List<T> items) {
        objectsList.addAll(0, items);
        displayItemCount = objectsList.size();
        notifyDataSetChanged();
    }

    public void addAtLast(List<T> items) {
        if (isPageLoading())
            objectsList.addAll(objectsList.size() - 1, items);
        else
            objectsList.addAll(items);
        displayItemCount = objectsList.size();
        notifyDataSetChanged();
    }


    public List<T> getList() {
        return objectsList;
    }

    public void remove(T item) {
        objectsList.remove(item);
        displayItemCount = objectsList.size();
        notifyDataSetChanged();
    }

    public void remove(int position) {
        objectsList.remove(position);
        displayItemCount = objectsList.size();
    }

    public interface PaginationListener {
        void onPageLoad(int offset);
    }

    protected void loadNextPage(int position) {
        if (paginationListener != null && position == (objectsList.size() - 1)) {
            if (isPageLoading())
                paginationListener.onPageLoad(objectsList.size() - 1);
            else
                paginationListener.onPageLoad(objectsList.size());
        }
    }
}
