/*
 * Copyright (c) 2016 riontech-xten
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.xtensolution.support.ui.activity;

import android.os.Bundle;
import android.view.MenuItem;
import android.view.View;
import android.widget.ProgressBar;

import androidx.annotation.StringRes;
import androidx.appcompat.app.AppCompatActivity;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentTransaction;

import com.google.android.material.floatingactionbutton.FloatingActionButton;
import com.xtensolution.support.R;
import com.xtensolution.support.exceptions.FragmentContainerResourceNotFoundException;
import com.xtensolution.support.ui.fragment.CoreBaseFragment;
import com.xtensolution.support.utils.AppLog;
import com.xtensolution.support.utils.ErrorUtils;
import com.xtensolution.support.utils.ResourceUtils;

/**
 * A abstract {@link AppCompatActivity} subclass.
 * To handle and initialize the basic resource like
 * actionbar, progressbar, fab
 */
public abstract class CoreBaseActivity extends AppCompatActivity {
    private static final String TAG = CoreBaseActivity.class.getSimpleName();

    protected ProgressBar progressBar;
    protected FloatingActionButton fab;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(getLayout());

        // to initialize the progressbar
        if (getProgressBarId() != 0)
            progressBar = _findViewById(getProgressBarId());

        // to initialize the fab button
        if (getFabId() != 0) {
            fab = _findViewById(getFabId());
            fab.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    onFabClick(view);
                }
            });
        }

        // to inform that content view bind with activity
        onContentViewBind();
    }

    /**
     * A factory method to handle the fab button
     * click event
     *
     * @param view
     */
    protected void onFabClick(View view) {
        showSnackbar(R.string.fab_clicked, new View.OnClickListener() {
            @Override
            public void onClick(View view) {

            }
        });
    }

    /**
     * Override into child class to get fab button resource id.
     * default id is 0.
     *
     * @return fab resource id
     */
    protected int getFabId() {
        return 0;
    }

    @SuppressWarnings("unchecked")
    /**
     * A generic factory method use to initialize the view
     * @return view
     */
    public <T extends View> T _findViewById(int viewId) {
        return (T) findViewById(viewId);
    }


    /**
     * A factory method use to show the snackbar on UI mode
     * to display error or exception
     *
     * @param resId
     * @param e
     * @param clickListener
     */
    protected void showSnackbarError(@StringRes int resId, Exception e, View.OnClickListener clickListener) {
        ErrorUtils.showSnackbar(getRootView(), resId, e, R.string.dlg_retry, clickListener);
    }

    /**
     * A abstract method to get the current layout
     * from child class
     *
     * @return string
     */
    protected abstract int getLayout();

    /**
     * A abstract method to get the current root view
     * from child class
     *
     * @return string
     */
    public abstract View getRootView();

    /**
     * A abstract method call from child class
     */
    protected abstract void onContentViewBind();


    /**
     * To get the progress bar resource id from child class
     *
     * @return int progressbar resource id
     */
    public int getProgressBarId() {
        return 0;
    }


    /**
     * To replace the container fragment
     *
     * @param fragment
     */
    public void replaceFragment(Fragment fragment) {
        try {

            if (getFragmentContainer() != 0) {
//                changeSelectedMenu();
                FragmentTransaction ft = getSupportFragmentManager().beginTransaction();
                ft.replace(getFragmentContainer(), fragment, fragment.getClass().getSimpleName());
                ft.setTransition(FragmentTransaction.TRANSIT_FRAGMENT_FADE);
                ft.addToBackStack(fragment.getClass().getSimpleName());
                ft.commit();

            } else {
                throw new FragmentContainerResourceNotFoundException(ResourceUtils.getString(R.string.no_fragment_container_found));
            }
        } catch (Exception e) {
            AppLog.e(TAG, e.getMessage(), e);
        }
    }

    /**
     * To get the container resource id from child class
     *
     * @return int container resource id
     */
    protected int getFragmentContainer() {
        return 0;
    }

    /**
     * To handle the current selected menu item
     */
    protected void changeSelectedMenu() {

    }

    /**
     * To show the snackbar on UI mode
     *
     * @param resId
     * @param listener
     */
    public void showSnackbar(@StringRes int resId, View.OnClickListener listener) {
        ErrorUtils.showSnackbar(getRootView(), resId, R.string.dlg_ok, listener);
    }

    public void showSnackbar(String message){
        ErrorUtils.showSnackbar(getRootView(), message, R.string.dlg_ok, new View.OnClickListener() {
            @Override
            public void onClick(View v) {

            }
        });
    }

}
