/*
 * Copyright (c) 2016 riontech-xten
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.xtensolutions.support.ui.activity;

import android.os.Bundle;
import android.support.annotation.StringRes;
import android.support.design.widget.FloatingActionButton;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentTransaction;
import android.support.v7.app.ActionBar;
import android.support.v7.app.AppCompatActivity;
import android.view.MenuItem;
import android.view.View;
import android.widget.ProgressBar;

import com.xtensolutions.support.R;
import com.xtensolutions.support.exceptions.FragmentContainerResourceNotFoundException;
import com.xtensolutions.support.ui.fragment.CoreBaseFragment;
import com.xtensolutions.support.utils.AppLog;
import com.xtensolutions.support.utils.ErrorUtils;
import com.xtensolutions.support.utils.ResourceUtils;


/**
 * A abstract subclass.
 * To handle and initialize the basic resource like
 * actionbar, progressbar, fab
 */
public abstract class CoreBaseActivity extends AppCompatActivity {
    private static final String TAG = CoreBaseActivity.class.getSimpleName();
    protected ActionBar mActionBar;
    protected ProgressBar progressBar;
    protected FloatingActionButton fab;
    protected Fragment currentLoadedFragment;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(getLayout());

        // to initialize the progressbar
        if (getProgressBarId() != 0)
            progressBar = _findViewById(getProgressBarId());

        // to initialize the fab button
        if (getFabId() != 0) {
            fab = _findViewById(getFabId());
            fab.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    onFabClick(view);
                }
            });
        }

        // to inform that content view bind with activity
        onContentViewBind();

//        if (getSupportActionBar() != null)
//            mActionBar = getSupportActionBar();
//
//        // Hack. Forcing overflow button on actionbar on devices with hardware menu button
//        try {
//            ViewConfiguration config = ViewConfiguration.get(this);
//            Field menuKeyField = ViewConfiguration.class.getDeclaredField("sHasPermanentMenuKey");
//            if (menuKeyField != null) {
//                menuKeyField.setAccessible(true);
//                menuKeyField.setBoolean(config, false);
//            }
//        } catch (Exception ex) {
//            // Ignore
//        }
    }

    /**
     * A factory method to handle the fab button
     * click event
     * @param view
     */
    protected void onFabClick(View view) {
        showSnackbar(R.string.fab_clicked, new View.OnClickListener() {
            @Override
            public void onClick(View view) {

            }
        });
    }

    /**
     * Override into child class to get fab button resource id.
     * default id is 0.
     * @return fab resource id
     */
    protected int getFabId() {
        return 0;
    }

    @Override
    protected void onResume() {
        super.onResume();
        setActionBarTitle(getScreenTitle());
    }

    @SuppressWarnings("unchecked")
    /**
     * A generic factory method use to initialize the view
     * @return view
     */
    public <T extends View> T _findViewById(int viewId) {
        return (T) findViewById(viewId);
    }

    /**
     * To handle the action bar title
     * @param title
     */
    public void setActionBarTitle(int title) {
        if (mActionBar != null) {
            mActionBar.setTitle(title);
        }
    }

    /**
     * To handle the action bar title
     * @param title
     */
    public void setActionBarTitle(CharSequence title) {
        if (mActionBar != null) {
            mActionBar.setTitle(title);
        }
    }

    /**
     * A factory method use to show the snackbar on UI mode
     * to display error or exception
     * @param resId
     * @param e
     * @param clickListener
     */
    protected void showSnackbarError(@StringRes int resId, Exception e, View.OnClickListener clickListener) {
        ErrorUtils.showSnackbar(getRootView(), resId, e, R.string.dlg_retry, clickListener);
    }

    /**
     * A abstract method to get the current screen title
     * from child class
     * @return string
     */
    protected abstract String getScreenTitle();

    /**
     * A abstract method to get the current layout
     * from child class
     * @return string
     */
    protected abstract int getLayout();

    /**
     * A abstract method to get the current root view
     * from child class
     * @return string
     */
    public abstract View getRootView();

    /**
     * A abstract method call from child class
     */
    protected abstract void onContentViewBind();

    /**
     * To handle the home button action enable or disable
     * @param enable
     */
    protected void setHomeButtonEnable(boolean enable) {
        if (mActionBar != null) {
            mActionBar.setHomeButtonEnabled(enable);
            mActionBar.setDisplayHomeAsUpEnabled(enable);
        }
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        if (item.getItemId() == android.R.id.home)
            onBackPressed();

        return super.onOptionsItemSelected(item);
    }

    /**
     * To get the progress bar resource id from child class
     * @return int progressbar resource id
     */
    public int getProgressBarId() {
        return 0;
    }


    /**
     * To replace the container fragment
     * @param fragment
     */
    public void replaceFragment(Fragment fragment) {
        try {

            if (fragment instanceof CoreBaseFragment) {
                CoreBaseFragment baseFragment = (CoreBaseFragment) fragment;
                if (baseFragment.isHomePage()) {
                    onBackPressed();
                    return;
                }
            }

            if (getFragmentContainer() != 0) {
                currentLoadedFragment = getSupportFragmentManager().findFragmentById(getFragmentContainer());

                if (currentLoadedFragment != null && currentLoadedFragment.getTag() != null) {
                    String tag = fragment.getClass().getSimpleName();
                    if (currentLoadedFragment.getTag().equalsIgnoreCase(tag)) {
                        return;
                    }
                }

//                changeSelectedMenu();
                FragmentTransaction ft = getSupportFragmentManager().beginTransaction();
                ft.replace(getFragmentContainer(), fragment, fragment.getClass().getSimpleName());
                ft.setTransition(FragmentTransaction.TRANSIT_ENTER_MASK);
                ft.addToBackStack(fragment.getClass().getSimpleName());
                ft.commit();

            } else {
                throw new FragmentContainerResourceNotFoundException(ResourceUtils.getString(R.string.no_fragment_container_found));
            }
        } catch (Exception e) {
            AppLog.e(TAG, e.getMessage(), e);
        }
    }

    /**
     * To get the container resource id from child class
     * @return int container resource id
     */
    protected int getFragmentContainer() {
        return 0;
    }

    /**
     * To handle the current selected menu item
     */
    protected void changeSelectedMenu(){

    }

    /**
     * To show the snackbar on UI mode
     * @param resId
     * @param listener
     */
    public void showSnackbar(@StringRes int resId, View.OnClickListener listener) {
        ErrorUtils.showSnackbar(getRootView(), resId, R.string.dlg_ok, listener);
    }

}
