package com.wms.network.common;

import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;

import com.google.gson.JsonParseException;
import com.wms.network.R;
import com.wms.network.RetrofitManager;
import com.wms.network.callback.NetCallback;
import com.wms.network.exception.ApiException;
import com.wms.network.exception.ResultEmptyException;

import org.json.JSONException;

import java.net.ConnectException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.text.ParseException;

import io.reactivex.Observer;
import io.reactivex.disposables.Disposable;
import retrofit2.adapter.rxjava2.HttpException;

/**
 * Created by 王梦思 on 2018/9/6.
 * <p/>
 */
public class NetSubscriber<T> implements Observer<T> {

    private NetCallback<T> mCallback;

    public NetSubscriber(NetCallback<T> callback) {
        this.mCallback = callback;
    }

    private static boolean isConnected() {
        ConnectivityManager connectivity = (ConnectivityManager) RetrofitManager.getContext().getSystemService(Context.CONNECTIVITY_SERVICE);
        if (null != connectivity) {
            NetworkInfo info = connectivity.getActiveNetworkInfo();
            if (null != info && info.isConnected()) {
                return info.getState() == NetworkInfo.State.CONNECTED;
            }
        }
        return false;
    }

    @Override
    public void onSubscribe(Disposable d) {
        if (!isConnected()) {
            if (mCallback != null) {
                mCallback.onNoNet();
                mCallback.onComplete();
            }
            d.dispose();
            return;
        }
        if (mCallback != null) {
            mCallback.onStart();
        }
    }

    @Override
    public void onNext(T result) {
        if (mCallback != null) {
            mCallback.onSucceed(result);
            mCallback.onComplete();
        }
    }

    @Override
    public void onError(Throwable e) {
        String errorMsg;
        if (e instanceof HttpException) {
            errorMsg = RetrofitManager.getString(R.string.wms_network_server_error);
        } else if (e instanceof JsonParseException
                || e instanceof JSONException
                || e instanceof ParseException) {
            errorMsg = RetrofitManager.getString(R.string.wms_network_data_error);
        } else if (e instanceof ConnectException
                || e instanceof SocketException
                || e instanceof SocketTimeoutException) {
            errorMsg = RetrofitManager.getString(R.string.wms_network_network_error);
        } else if (e instanceof ApiException) {
            errorMsg = e.getMessage();
        } else if (e instanceof ResultEmptyException) {
            onNext(null);
            return;
        } else {
            errorMsg = RetrofitManager.getString(R.string.wms_network_unknow_error);
        }

        if (mCallback != null) {
            mCallback.onFailed(errorMsg);
            mCallback.onComplete();
        }
    }

    @Override
    public void onComplete() {
    }
}
