package com.wms.network;

import android.annotation.SuppressLint;
import android.content.Context;
import android.text.TextUtils;

import com.google.gson.Gson;
import com.trello.rxlifecycle3.LifecycleTransformer;
import com.trello.rxlifecycle3.RxLifecycle;
import com.trello.rxlifecycle3.android.ActivityEvent;
import com.wms.network.baseserver.BaseHttpResult;
import com.wms.network.callback.NetCallback;
import com.wms.network.common.IHttpRequestExecutor;
import com.wms.network.common.NetSubscriber;
import com.wms.network.exception.ServerException;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import io.reactivex.Observable;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.annotations.NonNull;
import io.reactivex.schedulers.Schedulers;
import io.reactivex.subjects.BehaviorSubject;
import okhttp3.HttpUrl;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.logging.HttpLoggingInterceptor;
import retrofit2.Retrofit;
import retrofit2.adapter.rxjava2.RxJava2CallAdapterFactory;
import retrofit2.converter.gson.GsonConverterFactory;
import retrofit2.converter.scalars.ScalarsConverterFactory;

/**
 * Created by 王梦思 on 2018/9/6.
 * <p/>
 */
@SuppressLint("StaticFieldLeak")
public class RetrofitManager implements IHttpRequestExecutor {
    private static RetrofitManager mInstance = new RetrofitManager();
    private static Retrofit mRetrofit;
    private static RetrofitBuilder retrofitBuilder = new RetrofitBuilder();
    private static Map<String, Object> mCacheService = new HashMap<>();
    private static Context mContext;

    private RetrofitManager() {
        try {
            getClass().getClassLoader().loadClass("retrofit2.Retrofit");
            getClass().getClassLoader().loadClass("retrofit2.adapter.rxjava2.RxJava2CallAdapterFactory");
            getClass().getClassLoader().loadClass("io.reactivex.android.schedulers.AndroidSchedulers");
        } catch (ClassNotFoundException e) {
            throw new IllegalArgumentException("you should add library on your module,please add `implementation 'io.reactivex.rxjava2:rxjava:2.2.8'` " +
                    "and `annotationProcessor 'io.reactivex.rxjava2:rxandroid:2.1.1'` " +
                    "and `implementation com.squareup.retrofit2:retrofit:2.5.0'`" +
                    "and `com.squareup.retrofit2:adapter-rxjava2:2.5.0`");
        }
    }

    public static void init(Context context, RetrofitBuilder builder) {
        mContext = context.getApplicationContext();
        if (builder != null) {
            retrofitBuilder = builder;
        }
    }

    public static RetrofitManager getInstance() {
        if (mRetrofit == null) {
            checkUrl();
            OkHttpClient.Builder okhttpBuilder = new OkHttpClient.Builder()
                    .connectTimeout(retrofitBuilder.getConnectTimeout(), TimeUnit.MILLISECONDS)
                    .connectTimeout(retrofitBuilder.getReadTimeout(), TimeUnit.MILLISECONDS)
                    .writeTimeout(retrofitBuilder.getWriteTimeout(), TimeUnit.MILLISECONDS)
                    .addInterceptor(new Interceptor() {
                        @Override
                        public Response intercept(Chain chain) throws IOException {
                            return addHeaderInterceptor(chain);
                        }
                    });
            okhttpBuilder.addInterceptor(new BaseUrlInterceptor());
            if (retrofitBuilder.isEnableHttpLogger()) {
                try {
                    retrofitBuilder.getClass().getClassLoader().loadClass("okhttp3.logging.HttpLoggingInterceptor");
                } catch (ClassNotFoundException e) {
                    throw new IllegalArgumentException("you should add library on your module" +
                            ",please add `implementation 'com.squareup.okhttp3:logging-interceptor:3.10.0'`");
                }
                okhttpBuilder.addNetworkInterceptor(new HttpLoggingInterceptor().setLevel(HttpLoggingInterceptor.Level.BODY));
            }
            Retrofit.Builder builder = new Retrofit.Builder()
                    .addCallAdapterFactory(RxJava2CallAdapterFactory.create())
                    .baseUrl(retrofitBuilder.getBaseUrls()[0])
                    .client(okhttpBuilder.build());
            if (retrofitBuilder.isJson()) {
                try {
                    builder.getClass().getClassLoader().loadClass("retrofit2.converter.gson.GsonConverterFactory");
                } catch (ClassNotFoundException e) {
                    throw new IllegalArgumentException("you should add library on your module" +
                            ",please add `implementation 'com.squareup.retrofit2:converter-gson:2.5.0'`");
                }
                if (retrofitBuilder.getGson() != null) {
                    builder.addConverterFactory(GsonConverterFactory.create(retrofitBuilder.getGson()));
                } else {
                    builder.addConverterFactory(GsonConverterFactory.create());
                }
            } else {
                try {
                    builder.getClass().getClassLoader().loadClass("retrofit2.converter.scalars.ScalarsConverterFactory");
                } catch (ClassNotFoundException e) {
                    throw new IllegalArgumentException("you should add library on your module" +
                            ",please add `implementation 'com.squareup.retrofit2:converter-scalars:2.5.0'`");
                }
                builder.addConverterFactory(ScalarsConverterFactory.create());
            }
            mRetrofit = builder.build();
        }
        return mInstance;
    }

    private static void checkUrl() {
        if (retrofitBuilder.getBaseUrls() == null || retrofitBuilder.getBaseUrls().length <= 0) {
            throw new IllegalArgumentException("base url can not be null");
        }

        for (String url : retrofitBuilder.getBaseUrls()) {
            if (TextUtils.isEmpty(url)) {
                throw new IllegalArgumentException("you pass a empty base url");
            }

            if (!url.startsWith("http://") && !url.startsWith("https")) {
                throw new IllegalArgumentException("this is not a valid url.[" + url + "]");
            }
        }
    }

    private static Response addHeaderInterceptor(Interceptor.Chain chain) throws IOException {
        Request.Builder builder = chain.request()
                .newBuilder();
        if (retrofitBuilder.getHeader() == null || retrofitBuilder.getHeader().size() <= 0) {
            return chain.proceed(builder.build());
        }

        for (String key : retrofitBuilder.getHeader().keySet()) {
            String value = retrofitBuilder.getHeader().get(key);
            if (!TextUtils.isEmpty(value)) {
                builder.addHeader(key, value);
            }
        }

        return chain.proceed(builder.build());
    }

    public static Retrofit getRetrofit() {
        return mRetrofit;
    }

    private static Response parseBaseUrl(@NonNull Interceptor.Chain chain, Request originalRequest, String newBaseUrl) throws IOException {
        HttpUrl oldUrl = originalRequest.url();
        Request.Builder builder = originalRequest.newBuilder();
        builder.removeHeader("baseUrls");
        builder.removeHeader("baseUrlIndex");
        HttpUrl newUrl = HttpUrl.parse(newBaseUrl);
        if (newUrl == null) return chain.proceed(originalRequest);

        HttpUrl.Builder oldBuilder = oldUrl.newBuilder();
        List<String> oldPathSegments = oldUrl.pathSegments();
        List<String> validateSegments = new ArrayList<>();
        if (oldPathSegments != null && oldPathSegments.size() > 0) {
            validateSegments.addAll(oldPathSegments);
            HttpUrl baseUrl = HttpUrl.parse(retrofitBuilder.getBaseUrls()[0]);
            if (baseUrl != null) {
                List<String> baseUrlSegments = baseUrl.pathSegments();
                Iterator<String> iterator = validateSegments.iterator();
                String segment;
                while (iterator.hasNext()) {
                    segment = iterator.next();
                    boolean needRemove = false;
                    for (String baseSegment : baseUrlSegments) {
                        if (segment.equals(baseSegment)) {
                            needRemove = true;
                            break;
                        }
                    }
                    if (needRemove) {
                        iterator.remove();
                    }
                }
            }

            int size = oldPathSegments.size();
            for (int i = 0; i < size; i++) {
                oldBuilder.removePathSegment(0);
            }
        }

        List<String> newPathSegments = newUrl.pathSegments();
        if (newPathSegments != null) {
            for (String segment : newPathSegments) {
                oldBuilder.addPathSegments(segment);
            }
        }
        for (String segment : validateSegments) {
            oldBuilder.addPathSegments(segment);
        }

        HttpUrl newHttpUrl = oldBuilder
                .scheme(newUrl.scheme())
                .host(newUrl.host())
                .port(newUrl.port())
                .build();
        Request newRequest = builder.url(newHttpUrl).build();
        return chain.proceed(newRequest);
    }

    public static Context getContext() {
        return mContext;
    }

    public static String getString(int res) {
        return getContext().getString(res);
    }

    @SuppressWarnings("unchecked")
    public <T> T create(Class<T> service) {
        if (mContext == null) {
            throw new IllegalArgumentException("You should initialize RetrofitManager before using,You can initialize in your Application class");
        }
        if (service == null) {
            throw new RuntimeException("Api service is null!");
        }
        if (mCacheService.containsKey(service.getName())) {
            return (T) mCacheService.get(service.getName());
        }
        T realService = mRetrofit.create(service);
        mCacheService.put(service.getName(), realService);
        return realService;
    }

    /**
     * 清除所有的服务
     */
    public void clearService() {
        mCacheService.clear();
    }

    /**
     * 清除指定的服务
     */
    public void removeSevice(Class clz) {
        if (clz != null) {
            mCacheService.remove(clz.getName());
        }
    }

    @SuppressLint("CheckResult")
    @Override
    public <T> void execute(Observable<? extends BaseHttpResult<T>> observable, BehaviorSubject<ActivityEvent> lifecycle, NetCallback<T> callback) {
        if (lifecycle != null) {
            LifecycleTransformer<BaseHttpResult<T>> compose = RxLifecycle.bindUntilEvent(lifecycle, ActivityEvent.DESTROY);
            observable = observable.compose(compose);
        }

        execute(observable, callback);
    }

    /**
     * 执行retrofit请求
     *
     * @param observable 具体的请求
     * @param callback   回调
     * @param <T>        返回的数据类型
     */
    @SuppressLint("CheckResult")
    @Override
    public <T> void execute(Observable<? extends BaseHttpResult<T>> observable, NetCallback<T> callback) {
        observable.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .map(new ServerException<T>())
                .subscribe(new NetSubscriber<>(callback));
    }

    @Override
    public <T> void executeRawData(Observable<T> observable, BehaviorSubject<ActivityEvent> lifecycle, NetCallback<T> callback) {
        if (lifecycle != null) {
            LifecycleTransformer<T> compose = RxLifecycle.bindUntilEvent(lifecycle, ActivityEvent.DESTROY);
            observable = observable.compose(compose);
        }

        executeRawData(observable, callback);
    }

    /**
     * 执行retrofit请求获取原始数据
     *
     * @param observable 具体的请求
     * @param callback   回调
     * @param <T>        返回的数据类型
     */
    @SuppressLint("CheckResult")
    @Override
    public <T> void executeRawData(Observable<T> observable, NetCallback<T> callback) {
        observable
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new NetSubscriber<>(callback));
    }

    private static class BaseUrlInterceptor implements Interceptor {
        @Override
        public Response intercept(@NonNull Chain chain) throws IOException {
            Request originalRequest = chain.request();
            List<String> baseUrlList = originalRequest.headers("baseUrls");
            List<String> baseUrlIndexList = originalRequest.headers("baseUrlIndex");
            if (baseUrlList.size() > 0) {
                return parseBaseUrl(chain, originalRequest, baseUrlList.get(0));
            } else if (baseUrlIndexList.size() > 0) {
                try {
                    int index = Integer.parseInt(baseUrlIndexList.get(0));
                    if (index < retrofitBuilder.getBaseUrls().length) {
                        return parseBaseUrl(chain, originalRequest, retrofitBuilder.getBaseUrls()[index]);
                    } else {
                        return chain.proceed(originalRequest);
                    }
                } catch (Exception e) {
                    return chain.proceed(originalRequest);
                }
            } else {
                return chain.proceed(originalRequest);
            }
        }
    }

    public final static class RetrofitBuilder {
        /**
         * 支持多baseurl
         */
        private String[] baseUrls;
        /**
         * 连接超时时间
         */
        private int connectTimeout = 10000;
        private int readTimeout = 10000;
        private int writeTimeout = 10000;
        private boolean isJson = true;
        /**
         * 请求头
         */
        private Map<String, String> header;
        /**
         * 是否启用http日志
         */
        private boolean enableHttpLogger = true;
        private Gson gson;

        @NonNull
        public static RetrofitBuilder newBuilder() {
            return new RetrofitBuilder();
        }

        public RetrofitBuilder baseUrls(String... baseUrls) {
            this.baseUrls = baseUrls;
            return this;
        }

        public RetrofitBuilder connectTimeout(int time) {
            this.connectTimeout = time;
            return this;
        }

        public RetrofitBuilder readTimeout(int time) {
            this.readTimeout = time;
            return this;
        }

        public RetrofitBuilder writeTimeout(int time) {
            this.writeTimeout = time;
            return this;
        }

        public RetrofitBuilder isJson(boolean json) {
            this.isJson = json;
            return this;
        }

        public RetrofitBuilder enableInterceptor(boolean enableInterceptor) {
            this.enableHttpLogger = enableInterceptor;
            return this;
        }

        public RetrofitBuilder header(Map<String, String> headers) {
            this.header = headers;
            return this;
        }

        public RetrofitBuilder gson(Gson gson) {
            this.gson = gson;
            return this;
        }

        String[] getBaseUrls() {
            return baseUrls;
        }

        int getConnectTimeout() {
            return connectTimeout;
        }

        int getReadTimeout() {
            return readTimeout;
        }

        int getWriteTimeout() {
            return writeTimeout;
        }

        boolean isJson() {
            return isJson;
        }

        Map<String, String> getHeader() {
            return header;
        }

        boolean isEnableHttpLogger() {
            return enableHttpLogger;
        }

        public Gson getGson() {
            return gson;
        }
    }
}
