package com.webkul.mobikul.mobikuldialoglibrary;

import android.app.Dialog;
import android.content.Context;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.Bundle;
import android.support.annotation.ColorInt;
import android.support.annotation.DrawableRes;
import android.support.annotation.NonNull;
import android.support.annotation.StringRes;
import android.support.v4.content.ContextCompat;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.Window;
import android.widget.ImageView;
import android.widget.LinearLayout;

import com.webkul.mobikul.mobikuldialoglibrary.databinding.DialogCustomBinding;

/**
 * Created by anchit.makkar.
 * A Dialog Class to create customized dialogs.
 */

public class CustomDialog extends Dialog implements View.OnClickListener {

    private DialogCustomBinding binding;

    private String mTitleHeading = "";
    private String mSubTitle = "";
    private String mPositiveButtontext = "", mNegativeButtontext = "";
    private Drawable mIcon;
    private Context mContext;
    private int mTitleTextSize = 20;
    private int mSubTitleTextSize = 16;
    private boolean mButtonEnabled = false, mNegativeButtonEnabled = false, dismissOnBackPress = true;
    private Drawable mPositiveButtonBackgroundDrawable, mNegativeButtonBackgroundDrawable;
    private
    @ColorInt
    int mPositiveButtonBackgroundColor, mButtonTextColor, mNegativeButtonBackgroundColor;
    private int mButtonsOrientation = LinearLayout.HORIZONTAL;
    private int mButtonsGravity = Gravity.END;
    private String mIconUrl;
    private CustomDialogButtonClickListener positiveButtonClickListener = null;
    private CustomDialogButtonClickListener negativeButtonClickListener = null;

    public CustomDialog(@NonNull Context context) {
        this(context, Type.NORMAL_TYPE_DIALOG);
    }

    public CustomDialog(@NonNull Context context, CustomDialog.Type type) {
        super(context);
        mContext = context;
        if (this.getWindow() != null) {
            requestWindowFeature(Window.FEATURE_NO_TITLE);
            this.getWindow().setBackgroundDrawable(new ColorDrawable(android.graphics.Color.TRANSPARENT));
        }
        binding = DialogCustomBinding.inflate(LayoutInflater.from(context), null, false);
        this.setContentView(binding.getRoot());
        this.setType(type);

    }

    public static CustomDialog getInstantDialog(@NonNull Context context, @NonNull Type type) {
        CustomDialog customDialog = new CustomDialog(context, type);
        customDialog.setPositiveButtonText(android.R.string.ok)
                .setPositiveButtonEnabled(true)
                .setTitleTextSize(22)
                .setSubTitleTextSize(16)
                .setDismissOnBackPress(false)
                .setCanceledOnTouchOutside(false);
        switch (type) {
            case SUCCESS_TYPE_DIALOG:
                customDialog.setIcon(R.drawable.ic_vector_custom_dialog_success_icon)
                        .setTitleHeading(R.string.success)
                        .setSubTitle(R.string.process_is_successful)
                        .setPositiveButtonEnabled(false)
                        .setButtonsGravity(Gravity.CENTER)
                        .setDismissOnBackPress(true)
                        .setCanceledOnTouchOutside(true);
                break;
            case ERROR_TYPE_DIALOG:
                customDialog.setIcon(R.drawable.ic_vector_custom_dialog_error_icon)
                        .setTitleHeading(R.string.error)
                        .setButtonsGravity(Gravity.CENTER)
                        .setSubTitle(R.string.error_occured_please_try_again);
                break;
            case WARNING_TYPE_DIALOG:
                customDialog.setIcon(R.drawable.ic_vector_custom_dialog_warning_icon)
                        .setTitleHeading(R.string.warning)
                        .setButtonsGravity(Gravity.CENTER)
                        .setSubTitle(R.string.warning_mesaage);
                break;
            case CUSTOM_TYPE_DIALOG:
            case NORMAL_TYPE_DIALOG:
            default:
                break;
        }
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            customDialog.setButtonTextColor(context.getResources().getColor(android.R.color.white, null));
        } else {
            customDialog.setButtonTextColor(context.getResources().getColor(android.R.color.white));
        }
        return customDialog;
    }

    public CustomDialog setType(CustomDialog.Type type) {
        switch (type) {
            case SUCCESS_TYPE_DIALOG:
                setIcon(R.drawable.ic_vector_custom_dialog_success_icon);
                break;
            case ERROR_TYPE_DIALOG:
                setIcon(R.drawable.ic_vector_custom_dialog_error_icon);
                break;
            case WARNING_TYPE_DIALOG:
                setIcon(R.drawable.ic_vector_custom_dialog_warning_icon);
                break;
            case CUSTOM_TYPE_DIALOG:
            case NORMAL_TYPE_DIALOG:
            default:
                break;
        }

        return this;
    }

    public CustomDialog setTitleHeading(@StringRes int text) {
        mTitleHeading = mContext.getString(text);
        return this;
    }

    public CustomDialog setTitleHeading(String text) {
        mTitleHeading = text;
        return this;
    }

    public CustomDialog setSubTitle(@StringRes int text) {
        mSubTitle = mContext.getString(text);
        return this;
    }

    public CustomDialog setSubTitle(String text) {
        mSubTitle = text;
        return this;
    }

    public CustomDialog setTitleTextSize(int testSize) {
        mTitleTextSize = testSize;
        return this;
    }

    public CustomDialog setSubTitleTextSize(int testSize) {
        mSubTitleTextSize = testSize;
        return this;
    }

    public CustomDialog setIcon(@DrawableRes int resource) {
        Drawable drawable = ContextCompat.getDrawable(mContext, resource);
        return setIcon(drawable);
    }

    public CustomDialog setIcon(Drawable drawable) {
        mIcon = drawable;
        return this;
    }

    public CustomDialog setPositiveButtonText(@StringRes int text) {
        mPositiveButtontext = mContext.getString(text);
        return this;
    }

    public CustomDialog setPositiveButtonText(String text) {
        mPositiveButtontext = text;
        return this;
    }

    public CustomDialog setNegativeButtonText(@StringRes int text) {
        mNegativeButtontext = mContext.getString(text);
        return this;
    }

    public CustomDialog setNegativeButtonText(String text) {
        mNegativeButtontext = text;
        return this;

    }

    public CustomDialog setPositiveButtonBackground(@DrawableRes int resource) {
        Drawable drawable = ContextCompat.getDrawable(mContext, resource);
        return setPositiveButtonBackground(drawable);
    }

    public CustomDialog setNegativeButtonBackground(@DrawableRes int resource) {
        Drawable drawable = ContextCompat.getDrawable(mContext, resource);
        return setNegativeButtonBackground(drawable);
    }

    public CustomDialog setPositiveButtonBackground(Drawable drawable) {
        mPositiveButtonBackgroundDrawable = drawable;
        return this;
    }

    public CustomDialog setNegativeButtonBackground(Drawable drawable) {
        mNegativeButtonBackgroundDrawable = drawable;
        return this;

    }

    public CustomDialog setPositiveButtonBackgroundColor(@ColorInt int color) {
        mPositiveButtonBackgroundColor = color;
        return this;
    }

    public CustomDialog setNegativeButtonBackgroundColor(@ColorInt int color) {
        mNegativeButtonBackgroundColor = color;
        return this;
    }

    public CustomDialog setPositiveButtonEnabled(boolean enabled) {
        mButtonEnabled = enabled;
        return this;
    }

    public CustomDialog setNegativeButtonEnabled(boolean enabled) {
        mNegativeButtonEnabled = enabled;
        return this;
    }

    /*
    * A function for setting Orientation of the Buttons in Dialog
    * @param orientation - The desired orientation of the object, as defined by the
    *                LinearLayout class.
    */
    public CustomDialog setButtonsOrientation(int orientation) {
        mButtonsOrientation = orientation;
        return this;
    }

    /*
    * A function for setting Gravity of the Buttons in Dialog
    * @param gravity - The desired placement of the object, as defined by the
    *                constants in this class.
    */
    public CustomDialog setButtonsGravity(int gravity) {
        mButtonsGravity = gravity;
        return this;
    }

    public CustomDialog setButtonTextColor(@ColorInt int color) {
        mButtonTextColor = color;
        return this;

    }

    public CustomDialog setDismissOnBackPress(boolean dismissOnBackPress) {
        this.dismissOnBackPress = dismissOnBackPress;
        return this;
    }

    @Override
    public void onBackPressed() {
        if (dismissOnBackPress) {
            super.onBackPressed();
        }
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        if (!mTitleHeading.isEmpty()) {
            binding.heading.setVisibility(View.VISIBLE);
            binding.heading.setText(mTitleHeading);
        } else {
            binding.heading.setVisibility(View.GONE);
        }

        if (!mSubTitle.isEmpty()) {
            binding.subheading.setVisibility(View.VISIBLE);
            binding.subheading.setText(mSubTitle);
        } else {
            binding.subheading.setVisibility(View.GONE);
        }


        binding.heading.setTextSize(mTitleTextSize);
        binding.subheading.setTextSize(mSubTitleTextSize);

        if (mIcon != null) {
            binding.icon.setVisibility(View.VISIBLE);
            binding.icon.setImageDrawable(mIcon);
        } else if (binding.icon.getDrawable() != null) {
            binding.icon.setVisibility(View.VISIBLE);

        } else {
            binding.icon.setVisibility(View.GONE);
        }
        binding.positiveButton.setVisibility(mButtonEnabled ? View.VISIBLE : View.GONE);
        binding.negativeButton.setVisibility(mNegativeButtonEnabled ? View.VISIBLE : View.GONE);
        binding.positiveButton.setText(mPositiveButtontext);
        binding.negativeButton.setText(mNegativeButtontext);
        if (mButtonTextColor != 0) {
            binding.positiveButton.setTextColor(mButtonTextColor);
            binding.negativeButton.setTextColor(mButtonTextColor);
        }
        if (mPositiveButtonBackgroundColor != 0) {
            binding.positiveButton.setBackgroundColor(mPositiveButtonBackgroundColor);
        }

        if (mNegativeButtonBackgroundColor != 0) {
            binding.negativeButton.setBackgroundColor(mNegativeButtonBackgroundColor);
        }

        if (mPositiveButtonBackgroundDrawable != null) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                binding.positiveButton.setBackground(mPositiveButtonBackgroundDrawable);
            } else {
                binding.positiveButton.setBackgroundDrawable(mPositiveButtonBackgroundDrawable);
            }
        }

        if (mNegativeButtonBackgroundDrawable != null) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                binding.negativeButton.setBackground(mNegativeButtonBackgroundDrawable);
            } else {
                binding.negativeButton.setBackgroundDrawable(mNegativeButtonBackgroundDrawable);
            }
        }

        binding.positiveButton.setOnClickListener(this);
        binding.negativeButton.setOnClickListener(this);

        binding.buttonLayout.setOrientation(mButtonsOrientation);
        binding.buttonLayout.setGravity(mButtonsGravity);
    }

    public CustomDialog setPositiveButtonClickListener(CustomDialogButtonClickListener positiveButtonClickListener) {
        this.positiveButtonClickListener = positiveButtonClickListener;
        return this;
    }

    public CustomDialog setNegativeButtonClickListener(CustomDialogButtonClickListener negativeButtonClickListener) {
        this.negativeButtonClickListener = negativeButtonClickListener;
        return this;
    }

    @Override
    public void onClick(View v) {
        int viewId = v.getId();
        if (viewId == R.id.positiveButton) {
            if (positiveButtonClickListener != null) {
                positiveButtonClickListener.onClick(CustomDialog.this);
            } else {
                CustomDialog.this.dismiss();
            }
        } else if (viewId == R.id.negativeButton) {
            if (negativeButtonClickListener != null) {
                negativeButtonClickListener.onClick(CustomDialog.this);
            } else {
                CustomDialog.this.dismiss();
            }

        }
    }

//    public CustomDialog setIconFromUrl(String iconUrl) {
//        mIconUrl = iconUrl;
//        return this;
//    }

//    @Override
//    public void onAttachedToWindow() {
//        super.onAttachedToWindow();
//        if (mIconUrl != null && !mIconUrl.isEmpty()) {
//            binding.icon.setVisibility(View.VISIBLE);
//            Glide.with(binding.icon.getContext()).load(mIconUrl).into(binding.icon);
//        }
//
//    }

    public ImageView getIconview() {
        return binding.icon;
    }

    public CustomDialog addMoreViews(View view) {
        binding.extraViewContainer.addView(view);
        return this;
    }

    public enum Type {
        SUCCESS_TYPE_DIALOG,
        ERROR_TYPE_DIALOG,
        WARNING_TYPE_DIALOG,
        NORMAL_TYPE_DIALOG,
        CUSTOM_TYPE_DIALOG


    }


    public interface CustomDialogButtonClickListener {
        void onClick(CustomDialog customDialog);
    }

}

