/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.webauthn4j.data;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;
import com.webauthn4j.util.AssertUtil;
import org.checkerframework.checker.nullness.qual.NonNull;

import java.io.Serializable;
import java.util.Objects;

/**
 * WebAuthn Relying Parties may use AttestationConveyancePreference to specify their preference
 * regarding attestation conveyance during credential generation.
 *
 * @see <a href="https://www.w3.org/TR/webauthn-1/#enumdef-attestationconveyancepreference">
 * §5.4.6. Attestation Conveyance Preference Enumeration (enum AttestationConveyancePreference)</a>
 */
public class AttestationConveyancePreference implements Serializable {

    /**
     * This value indicates that the Relying Party is not interested in authenticator attestation.
     * For example, in order to potentially avoid having to obtain user consent to relay identifying
     * information to the Relying Party, or to save a roundtrip to an Attestation CA.
     */
    public static final AttestationConveyancePreference NONE = new AttestationConveyancePreference("none");

    /**
     * This value indicates that the Relying Party prefers an attestation conveyance yielding
     * verifiable attestation statements, but allows the client to decide how to obtain such
     * attestation statements. The client MAY replace the authenticator-generated attestation
     * statements with attestation statements generated by an Anonymization CA, in order to protect
     * the user’s privacy, or to assist Relying Parties with attestation verification in a
     * heterogeneous ecosystem.
     */
    public static final AttestationConveyancePreference INDIRECT = new AttestationConveyancePreference("indirect");

    /**
     * This value indicates that the Relying Party wants to receive the attestation statement as
     * generated by the authenticator.
     */
    public static final AttestationConveyancePreference DIRECT = new AttestationConveyancePreference("direct");

    /**
     * This value indicates that the Relying Party wants to receive an attestation statement
     * that may include uniquely identifying information.
     * This is intended for controlled deployments within an enterprise where the organization wishes
     * to tie registrations to specific authenticators.
     * User agents MUST NOT provide such an attestation unless the user agent or authenticator configuration permits it
     * for the requested RP ID.
     * <p>
     * If permitted, the user agent SHOULD signal to the authenticator (at invocation time) that enterprise attestation
     * is requested, and convey the resulting AAGUID and attestation statement, unaltered, to the Relying Party.
     */
    public static final AttestationConveyancePreference ENTERPRISE = new AttestationConveyancePreference("enterprise");

    private final String value;

    private AttestationConveyancePreference(@NonNull String value) {
        this.value = value;
    }

    @JsonCreator
    public static @NonNull AttestationConveyancePreference create(@NonNull String value) {
        AssertUtil.notNull(value, "value must not be null.");
        switch (value) {
            case "none":
                return NONE;
            case "indirect":
                return INDIRECT;
            case "direct":
                return DIRECT;
            case "enterprise":
                return ENTERPRISE;
            default:
                return new AttestationConveyancePreference(value);
        }
    }

    @JsonValue
    public @NonNull String getValue() {
        return value;
    }

    @Override
    public String toString() {
        return value;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        AttestationConveyancePreference that = (AttestationConveyancePreference) o;
        return value.equals(that.value);
    }

    @Override
    public int hashCode() {
        return Objects.hash(value);
    }
}
