package com.vungle.ads.internal.util

import android.app.Activity
import android.app.Application
import android.content.Context
import android.content.Intent
import android.os.Bundle
import androidx.annotation.VisibleForTesting
import com.vungle.ads.internal.ui.PresenterAdOpenCallback
import java.lang.ref.WeakReference
import java.util.concurrent.CopyOnWriteArraySet
import java.util.concurrent.atomic.AtomicBoolean

internal class ActivityManager private constructor() : Application.ActivityLifecycleCallbacks {
    open class LifeCycleCallback {

        open fun onForeground() {
            //no-op
        }

        open fun onBackground() {
            //no-op
        }
    }

    private val isInitialized = AtomicBoolean(false)

    @Volatile
    private var foregroundActivityCount = 0

    @Volatile
    private var isAppInForeground = false

    @Volatile
    private var targetActivityInfo: TargetActivityInfo? = null
    private val callbacks = CopyOnWriteArraySet<LifeCycleCallback>()

    private fun init(context: Context) {
        if (isInitialized.getAndSet(true)) return
        runCatching {
            val app = context.applicationContext as Application
            app.registerActivityLifecycleCallbacks(this)
        }.onFailure { e ->
            Logger.e(TAG, "Error initializing ActivityManager", e)
            isInitialized.set(false)
        }
    }

    private fun deInit(context: Context) {
        val app = context.applicationContext as Application
        app.unregisterActivityLifecycleCallbacks(this)
        isInitialized.set(false)
        targetActivityInfo = null
        foregroundActivityCount = 0
        isAppInForeground = false
        callbacks.clear()
    }

    // Check current app foreground state
    private fun isAppInForeground(): Boolean = !isInitialized.get() || isAppInForeground

    private fun addListener(callback: LifeCycleCallback) {
        callbacks.add(callback)
    }

    fun removeListener(callback: LifeCycleCallback) {
        callbacks.remove(callback)
    }

    override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {
        //no-op
    }

    override fun onActivityStarted(activity: Activity) {
        foregroundActivityCount++
        if (!isAppInForeground && foregroundActivityCount == 1) {
            isAppInForeground = true

            targetActivityInfo?.run {
                context.get()?.let {
                    startWhenForeground(it, deepLinkOverrideIntent, defaultIntent, adOpenCallback)
                }
                targetActivityInfo = null
            }

            callbacks.forEach {
                it.onForeground()
            }
        }
    }

    override fun onActivityStopped(activity: Activity) {
        foregroundActivityCount--
        if (isAppInForeground && foregroundActivityCount == 0) {
            isAppInForeground = false

            callbacks.forEach {
                it.onBackground()
            }
        }
    }

    override fun onActivityResumed(activity: Activity) {
        //no-op
    }

    override fun onActivityPaused(activity: Activity) {
        //no-op
    }

    override fun onActivitySaveInstanceState(activity: Activity, outState: Bundle) {
        //no-op
    }

    override fun onActivityDestroyed(activity: Activity) {
        //no-op
    }

    private fun startActivitySafely(
        context: Context,
        deepLinkOverrideIntent: Intent?,
        defaultIntent: Intent?,
        adOpenCallback: PresenterAdOpenCallback?
    ): Boolean {
        try {
            when {
                deepLinkOverrideIntent != null -> {
                    context.startActivity(deepLinkOverrideIntent)
                    adOpenCallback?.onDeeplinkClick(true)
                }

                defaultIntent != null -> {
                    context.startActivity(defaultIntent)
                }

                else -> return false
            }
            return true
        } catch (e: Exception) {
            Logger.e(TAG, "Failed to start activity: $e")
            try {
                if (deepLinkOverrideIntent != null) {
                    adOpenCallback?.onDeeplinkClick(false)
                }

                if (deepLinkOverrideIntent == null || defaultIntent == null) {
                    return false
                }
                context.startActivity(defaultIntent)
            } catch (exception: Exception) {
                return false
            }
        }

        return true
    }

    private data class TargetActivityInfo(
        val context: WeakReference<Context>,
        val deepLinkOverrideIntent: Intent?,
        val defaultIntent: Intent?,
        val adOpenCallback: PresenterAdOpenCallback?
    )

    companion object {
        private const val TAG = "ActivityManager"

        @VisibleForTesting
        internal val instance = ActivityManager()

        fun startWhenForeground(
            context: Context,
            deepLinkOverrideIntent: Intent?,
            defaultIntent: Intent?,
            adOpenCallback: PresenterAdOpenCallback?
        ): Boolean {
            return if (isForeground()) {
                instance.startActivitySafely(context, deepLinkOverrideIntent, defaultIntent, adOpenCallback)
            } else {
                // API 29 and 30 are not work as expected as what google API said, so we need to start
                // activity in onActivityStarted manually
                instance.targetActivityInfo = TargetActivityInfo(
                    WeakReference(context),
                    deepLinkOverrideIntent,
                    defaultIntent,
                    adOpenCallback
                )
                false
            }
        }

        fun init(context: Context) = instance.init(context)

        fun isForeground(): Boolean = instance.isAppInForeground()

        fun addLifecycleListener(listener: LifeCycleCallback) {
            instance.addListener(listener)
        }

        fun removeLifecycleListener(listener: LifeCycleCallback) {
            instance.removeListener(listener)
        }

        internal fun deInit(context: Context) = instance.deInit(context)
    }
}
