package com.vungle.ads.internal.ui.view

import android.annotation.SuppressLint
import android.content.Context
import android.content.ContextWrapper
import android.graphics.Color
import android.os.Build
import android.os.Handler
import android.os.Looper
import android.view.MotionEvent
import android.view.ViewGroup
import android.webkit.WebView
import android.webkit.WebViewClient
import android.widget.RelativeLayout
import androidx.annotation.VisibleForTesting
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.platform.WebViewUtil
import com.vungle.ads.internal.presenter.WebViewManager
import com.vungle.ads.internal.util.HandlerScheduler
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.ViewUtility

class MRAIDAdWidget @Throws(InstantiationException::class) @JvmOverloads constructor(context: Context, val eventId: String? = null) :
    RelativeLayout(context) {
    private var webView: WebView? = null

    @VisibleForTesting internal var onViewTouchListener: OnViewTouchListener? = null
    @VisibleForTesting internal var closeDelegate: CloseDelegate? = null
    @VisibleForTesting internal var orientationDelegate: OrientationDelegate? = null

    @SuppressLint("ClickableViewAccessibility")
    private fun bindListeners() {
        webView?.setOnTouchListener { _, event ->
            onViewTouchListener?.onTouch(event) ?: false
        }
    }

    private fun prepare() {
        webView?.let {
            it.setLayerType(LAYER_TYPE_HARDWARE, null)
            it.setBackgroundColor(Color.TRANSPARENT)
            it.visibility = GONE
        }
    }

    interface CloseDelegate {
        fun close()
    }

    interface OrientationDelegate {
        fun setOrientation(orientation: Int)
    }

    interface OnViewTouchListener {
        fun onTouch(event: MotionEvent?): Boolean
    }

    fun setCloseDelegate(closeDelegate: CloseDelegate) {
        this.closeDelegate = closeDelegate
    }

    fun setOnViewTouchListener(onViewTouchListener: OnViewTouchListener?) {
        this.onViewTouchListener = onViewTouchListener
    }

    fun setOrientationDelegate(orientationDelegate: OrientationDelegate?) {
        this.orientationDelegate = orientationDelegate
    }

    fun close() {
        closeDelegate?.close()
    }

    fun setOrientation(requestedOrientation: Int) {
        orientationDelegate?.setOrientation(requestedOrientation)
    }

    fun linkWebView(vngWebViewClient: WebViewClient, webViewSettings: AdPayload.WebViewSettings?) {
        webView?.let {
            WebViewUtil.applyWebSettings(it, webViewSettings)
            it.webViewClient = vngWebViewClient
        }
    }

    fun showWebsite(url: String) {
        Logger.d(TAG, "loadUrl: $url")
        webView?.loadUrl(url)
    }

    fun pauseWeb() {
        webView?.onPause()
    }

    fun resumeWeb() {
        webView?.visibility = VISIBLE
        webView?.onResume()
    }

    val url: String?
        get() = webView?.url

    fun destroyWebView(webViewDestroyDelay: Long, isMaliBuggy: Boolean = false) {
        runCatching {
            (parent as? ViewGroup)?.removeView(this)
        }
        runCatching {
            removeAllViews()
        }

        if (webViewDestroyDelay <= 0) {
            DestroyRunnable(this, isMaliBuggy).run()
        } else {
            HandlerScheduler().schedule(DestroyRunnable(this, isMaliBuggy), webViewDestroyDelay)
        }
    }

    private class DestroyRunnable(private val widget: MRAIDAdWidget, val isMaliBuggy: Boolean) : Runnable {

        override fun run() {
            val wv = widget.webView ?: return

            if (Looper.myLooper() != Looper.getMainLooper()) {
                Handler(Looper.getMainLooper()).post { run() }
                return
            }

            try {
                runCatching { wv.onPause() }
                runCatching { wv.stopLoading() }

                runCatching { wv.webChromeClient = null }
                runCatching { wv.webViewClient = WebViewClient() }
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                    runCatching { wv.webViewRenderProcessClient = null }
                }

                wv.visibility = GONE

                if (isMaliBuggy) {
                    wv.setLayerType(LAYER_TYPE_SOFTWARE, null)
                }

                runCatching { wv.loadUrl("about:blank") }
                runCatching { wv.clearHistory() }
                runCatching { wv.removeAllViews() }
                val extraDelay = if (isMaliBuggy) 300L else 100L
                HandlerScheduler().schedule( {
                    try {
                        wv.destroy()
                        widget.eventId?.let { WebViewManager.destroyWebView(it) }
                        widget.webView = null
                    } catch (t: Throwable) {
                        Logger.e(TAG, "Destroy webview ${t.message}")
                    }
                }, extraDelay)

            } catch (t: Throwable) {
                Logger.e(TAG, "Destroy webview: ${t.message}")
            }
        }
    }

    override fun onAttachedToWindow() {
        super.onAttachedToWindow()
        layoutParams?.let {
            it.height = ViewGroup.LayoutParams.MATCH_PARENT
            it.width = ViewGroup.LayoutParams.MATCH_PARENT
        }
        webView?.layoutParams?.let {
            it.height = ViewGroup.LayoutParams.MATCH_PARENT
            it.width = ViewGroup.LayoutParams.MATCH_PARENT
        }
    }

    class AudioContextWrapper(base: Context?) : ContextWrapper(base) {
        override fun getSystemService(name: String): Any {
            return if (AUDIO_SERVICE == name) {
                applicationContext.getSystemService(name)
            } else {
                super.getSystemService(name)
            }
        }
    }

    companion object {
        private const val TAG = "MRAIDAdWidget"
    }

    init {
        val matchParentLayoutParams = LayoutParams(
            LayoutParams.MATCH_PARENT,
            LayoutParams.MATCH_PARENT
        )
        layoutParams = matchParentLayoutParams
        webView = ViewUtility.getWebView(context, eventId)
        webView?.layoutParams = matchParentLayoutParams
        webView?.tag = "VungleWebView"
        addView(webView, matchParentLayoutParams)
        bindListeners()
        prepare()
    }

    annotation class AdStopReason {
        companion object {
            const val IS_CHANGING_CONFIGURATION = 1 //1 << 0
            const val IS_AD_FINISHING = 1 shl 1
            const val IS_AD_FINISHED_BY_API = 1 shl 2
        }
    }

}
