package com.vungle.ads.internal.signals

import android.content.Context
import android.content.res.Configuration
import androidx.annotation.VisibleForTesting
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.executor.Executors
import com.vungle.ads.internal.model.UnclosedAd
import com.vungle.ads.internal.persistence.FilePreferences
import com.vungle.ads.internal.platform.DeviceCheckUtils
import com.vungle.ads.internal.session.UnclosedAdDetector
import com.vungle.ads.internal.util.ActivityManager
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.PathProvider
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import java.util.concurrent.ConcurrentHashMap

class SignalManager(val context: Context) {
    companion object {
        private const val TAG = "SignalManager"
        private const val SESSION_COUNT_NOT_SET = -1
        const val SIGNAL_VERSION = 2
        const val TWENTY_FOUR_HOURS_MILLIS: Long = 24 * 60 * 60 * 1000
        const val SESSION_COUNT_KEY = "vungle_signal_session_count"
        const val SESSION_TIME_KEY = "vungle_signal_session_creation_time"
    }

    private val json = Json {
        ignoreUnknownKeys = true
        encodeDefaults = true
        explicitNulls = false
    }

    var enterBackgroundTime = 0L
    var enterForegroundTime = System.currentTimeMillis()

    /** Session duration in millis *except* since last foreground if there was no background.
     * In other words, the amount of time spent in foregrounds before last coming from background. */
    var sessionDuration = 0L
    var sessionCount: Int = SESSION_COUNT_NOT_SET
    var sessionSeriesCreatedTime: Long

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal var currentSession: SessionData
    var mapOfLastLoadTimes = ConcurrentHashMap<String, Long>()
    val filePreferences: FilePreferences by inject(context)
    val uuid: String
        get() = currentSession.sessionId
    private var unclosedAdDetector: UnclosedAdDetector

    init {
        registerNotifications()
        sessionSeriesCreatedTime = filePreferences.getLong(SESSION_TIME_KEY, -1)
        updateSessionCount()
        currentSession = SessionData(sessionCount)

        val executors: Executors by inject(context)
        val pathProvider: PathProvider by inject(context)

        unclosedAdDetector =
            UnclosedAdDetector(context, currentSession.sessionId, executors, pathProvider)

        currentSession.unclosedAd = unclosedAdDetector.retrieveUnclosedAd()
        Logger.w(TAG) { "unclosedad: ${json.encodeToString(currentSession.unclosedAd)}" }

        try {
            currentSession.isDevice = if (DeviceCheckUtils.isEmulator()) 0 else 1
            currentSession.isVPNConnected = if (DeviceCheckUtils.isVpnConnected(context)) 1 else 0
            currentSession.overlayGranted = if (DeviceCheckUtils.hasSystemAlertWindowPermission(context)) 1 else 0
            currentSession.sensorCount = DeviceCheckUtils.getSensorCount(context)
            currentSession.httpProxyEnabled = if (DeviceCheckUtils.isProxyEnabled(context)) 1 else 0
        } catch (ex: Exception) {
            Logger.e(TAG, "Failed to collect device signals: ${ex.localizedMessage}")
        }

    }

    private fun updateSessionCount() {
        // SessionCount isn't set yet so pull from filePreferences
        if (sessionCount == SESSION_COUNT_NOT_SET) {
            sessionCount = filePreferences.getInt(SESSION_COUNT_KEY, 0)
        }
        val currentTime = System.currentTimeMillis()
        val timeDifference = currentTime - sessionSeriesCreatedTime
        // If the session created time isn't set (new install)
        // or if the time difference is >= 24 hours
        // Then set the session count to be 1 and store the current new time
        if (sessionSeriesCreatedTime < 0 || timeDifference >= TWENTY_FOUR_HOURS_MILLIS) {
            sessionCount = 1
            filePreferences.put(SESSION_TIME_KEY, currentTime)
            sessionSeriesCreatedTime = currentTime
        } else {
            sessionCount++
        }
        filePreferences.put(SESSION_COUNT_KEY, sessionCount)
        filePreferences.apply()
    }

    private fun registerNotifications() {
        ActivityManager.addLifecycleListener(object : ActivityManager.LifeCycleCallback() {
            override fun onForeground() {
                Logger.d(TAG, "SignalManager#onForeground()")
                val now = System.currentTimeMillis()
                val millisSinceLastBackground = now - enterBackgroundTime
                if (millisSinceLastBackground > ConfigManager.getSignalsSessionTimeout()) {
                    createNewSessionData()
                }
                enterForegroundTime = now
                enterBackgroundTime = 0L
            }

            override fun onBackground() {
                Logger.d(TAG, "SignalManager#onBackground()")
                enterBackgroundTime = System.currentTimeMillis()
                sessionDuration += enterBackgroundTime - enterForegroundTime
            }
        })
    }

    fun createNewSessionData() {
        updateSessionCount()
        val newSessionData = SessionData(sessionCount)
        currentSession = newSessionData
    }

    // Call when loadAd is called to give the ad object a signal ad
    // The multiple ad instances with the same placementId may call loadAd in multiple thread
    @Synchronized
    fun getSignaledAd(placementId: String): SignaledAd {
        val now = System.currentTimeMillis()
        val lastAdLoadTime =
            if (mapOfLastLoadTimes.containsKey(placementId)) mapOfLastLoadTimes[placementId] else null
        mapOfLastLoadTimes[placementId] = now
        return SignaledAd(lastAdLoadTime, now)
    }

    @Synchronized
    fun increaseSessionDepthCounter() {
        currentSession.sessionDepthCounter++
    }

    fun recordUnclosedAd(unclosedAd: UnclosedAd) {
        if (ConfigManager.signalsDisabled()) {
            return
        }
        unclosedAdDetector.addUnclosedAd(unclosedAd)
    }

    fun removeUnclosedAd(unclosedAd: UnclosedAd) {
        if (ConfigManager.signalsDisabled()) {
            return
        }
        unclosedAdDetector.removeUnclosedAd(unclosedAd)
    }

    // Call when playAd is called to set the signaled ad
    fun registerSignaledAd(context: Context?, signaledAd: SignaledAd) {
        currentSession.signaledAd.clear()
        currentSession.signaledAd.add(signaledAd)
        currentSession.signaledAd[0].screenOrientation = screenOrientation(context)
    }

    fun updateTemplateSignals(signals: String?) {
        if (!signals.isNullOrEmpty() && currentSession.signaledAd.isNotEmpty()) {
            currentSession.signaledAd[0].templateSignals = signals
        }
    }

    private fun updateSessionDuration() {
        currentSession.sessionDuration =
            (sessionDuration + System.currentTimeMillis() - enterForegroundTime)
    }

    fun generateSignals(): String? {
        updateSessionDuration()
        return try {
            "${SIGNAL_VERSION}:${json.encodeToString(currentSession)}"
        } catch (_: Throwable) {
            null
        }
    }

    fun screenOrientation(ctx: Context?): Int {
        val contextImpl = ctx ?: context
        return when (contextImpl.resources?.configuration?.orientation) {
            Configuration.ORIENTATION_LANDSCAPE -> 2
            Configuration.ORIENTATION_PORTRAIT -> 1
            Configuration.ORIENTATION_UNDEFINED -> 0
            else -> -1
        }
    }

}
