package com.vungle.ads.internal.session

import android.content.Context
import com.vungle.ads.internal.executor.Executors
import com.vungle.ads.internal.executor.FutureResult
import com.vungle.ads.internal.model.UnclosedAd
import com.vungle.ads.internal.util.FileUtility
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.PathProvider
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import java.io.File
import java.util.concurrent.Callable
import java.util.concurrent.CopyOnWriteArrayList
import java.util.concurrent.TimeUnit

class UnclosedAdDetector(
    val context: Context,
    private val sessionId: String,
    val executors: Executors,
    val pathProvider: PathProvider
) {

    companion object {
        private const val FILENAME = "unclosed_ad"
        private val json = Json {
            ignoreUnknownKeys = true
            encodeDefaults = true
            explicitNulls = false
            allowStructuredMapKeys = true
        }
    }

    private var file: File = pathProvider.getUnclosedAdFile(FILENAME)
    private val unclosedAdList = CopyOnWriteArrayList<UnclosedAd>()
    private var ready = false
    init {
        ready = if (!file.exists()) {
            runCatching { file.createNewFile() }.onFailure {
                Logger.e("UnclosedAdDetector", "Fail to create unclosed ad file: ${it.message}")
            }.isSuccess
        } else {
            true
        }
    }

    fun addUnclosedAd(ad: UnclosedAd) {
        if (!ready) return
        ad.sessionId = sessionId
        unclosedAdList.add(ad)
        writeUnclosedAdToFile(unclosedAdList)
    }

    fun removeUnclosedAd(ad: UnclosedAd) {
        if (!ready) return
        if (unclosedAdList.contains(ad)) {
            unclosedAdList.remove(ad)
            writeUnclosedAdToFile(unclosedAdList)
        }
    }

    fun retrieveUnclosedAd(): List<UnclosedAd> {
        val unclosedAdList = mutableListOf<UnclosedAd>()
        if (!ready) {
            return unclosedAdList
        }
        readUnclosedAdFromFile()?.let { unclosedAdList.addAll(it) }
        // Remove the local file after reading. Make sure unclosed ad only retrieved once.
        executors.ioExecutor.execute {
            try {
                FileUtility.deleteAndLogIfFailed(file)
            } catch (ex: Exception) {
                Logger.e("UnclosedAdDetector", "Fail to delete file ${ex.message}")
            }
        }

        return unclosedAdList
    }
    private inline fun <reified T> decodeJson(jsonString: String): T = json.decodeFromString(jsonString)

    private fun readUnclosedAdFromFile(): List<UnclosedAd>? {
        if (!ready) {
            return emptyList()
        }
        val futureResult: FutureResult<List<UnclosedAd>> = FutureResult(
            executors.ioExecutor
                .submit(Callable<List<UnclosedAd>> {
                    try {
                        val jsonString = FileUtility.readString(file)
                        if (!jsonString.isNullOrEmpty()) {
                            decodeJson<List<UnclosedAd>>(jsonString)
                        } else {
                            mutableListOf()
                        }
                    } catch (ex: Exception) {
                        Logger.e("UnclosedAdDetector", "Fail to read unclosed ad file ${ex.message}")
                        mutableListOf()
                    }
                })
        )
        return futureResult.get(1000, TimeUnit.MILLISECONDS)
    }

    private fun writeUnclosedAdToFile(unclosedAdList: List<UnclosedAd>) {
        if (!ready) {
            return
        }
        try {
            val jsonContent = json.encodeToString(unclosedAdList)
            executors.ioExecutor.execute {
                FileUtility.writeString(file, jsonContent)
            }
        } catch (tr: Throwable) {
            Logger.e("UnclosedAdDetector", "Fail to write unclosed ad file ${tr.message}")
        }
    }
}