package com.vungle.ads.internal.presenter

import android.content.Context
import android.webkit.WebView
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.executor.Executors
import com.vungle.ads.internal.executor.VungleThreadPoolExecutor
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.model.Placement
import com.vungle.ads.internal.platform.Platform
import com.vungle.ads.internal.platform.WebViewUtil
import com.vungle.ads.internal.privacy.PrivacyManager
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.ui.VungleWebClient
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.ThreadUtil
import java.util.concurrent.locks.ReentrantLock
import kotlin.concurrent.withLock

object WebViewManager {

    private const val TAG = "WebViewManager"
    private val lock = ReentrantLock()
    private val webViewCache = LinkedHashMap<String, WebViewEntry>()

    internal fun preloadWebView(
        context: Context,
        adv: AdPayload,
        placement: Placement,
        templatePath: String,
        webSettings: AdPayload.WebViewSettings?,
        delegate: PreloadDelegate,
        loadDuration: Long? = null
    ) = lock.withLock {
        Logger.d(TAG, "Preload webview start. Cache size: ${webViewCache.size}")

        val key = adv.eventId()
        if (webViewCache.containsKey(key)) return@withLock
        ThreadUtil.runOnUiThread {
            runCatching {
                // Use application context to avoid memory leak.
                val webView = WebView(context.applicationContext).apply {
                    WebViewUtil.applyWebSettings(this, webSettings)
                }

                val executors: Executors by inject(context)
                val platform: Platform by inject(context)
                val offloadExecutor = executors.offloadExecutor

                val webClient =
                    VungleWebClient(
                        adv,
                        placement,
                        offloadExecutor,
                        platform,
                        delegate,
                        loadDuration
                    )
                val collectedConsent =
                    ConfigManager.getGDPRIsCountryDataProtected() && "unknown" == PrivacyManager.getConsentStatus()
                webClient.setConsentStatus(
                    collectedConsent,
                    ConfigManager.getGDPRConsentTitle(),
                    ConfigManager.getGDPRConsentMessage(),
                    ConfigManager.getGDPRButtonAccept(),
                    ConfigManager.getGDPRButtonDeny()
                )
                webView.webViewClient = webClient
                webView.loadUrl(templatePath)

                webViewCache[key.toString()] = WebViewEntry(webView, webClient)
                AnalyticsClient.logMetric(
                    Sdk.SDKMetric.SDKMetricType.CONCURRENT_CACHED_WEBVIEW_COUNT,
                    webViewCache.size.toLong(),
                    adv.logEntry
                )
                Logger.d(TAG, "Preload complete. Cache size: ${webViewCache.size}")
            }.onFailure {
                Logger.e(TAG, "Preload webview failed", it)
                delegate.onAdReadyToPlay()
            }

        }
    }

    fun getOrCreateWebView(context: Context, key: String?): WebView = lock.withLock {
        val entry = webViewCache[key]
        return if (entry != null) {
            Logger.d(TAG, "Reusing cached webview. Cache size: ${webViewCache.size}")
            entry.webView
        } else {
            Logger.d(TAG, "Creating new webview. Cache size: ${webViewCache.size}")
            WebView(context)
        }
    }

    fun getOrCreateWebViewClient(
        advertisement: AdPayload,
        placement: Placement,
        offloadExecutor: VungleThreadPoolExecutor,
        platform: Platform
    ): VungleWebClient = lock.withLock {

        val webClient = webViewCache[advertisement.eventId()]?.webViewClient
        return webClient ?: VungleWebClient(advertisement, placement, offloadExecutor, platform)
    }

    fun destroyWebView(key: String) = lock.withLock {
        destroyWebViewInternal(key)
    }

    private fun destroyWebViewInternal(key: String) {
        webViewCache.remove(key)
    }

    private data class WebViewEntry(
        val webView: WebView,
        val webViewClient: VungleWebClient
    )
}
