package com.vungle.ads.internal.presenter

import android.content.Intent
import android.content.pm.ActivityInfo
import android.view.MotionEvent
import android.view.View
import android.webkit.WebView
import android.webkit.WebViewRenderProcess
import androidx.annotation.VisibleForTesting
import com.vungle.ads.AdConfig
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.HeartbeatMissingError
import com.vungle.ads.IndexHtmlError
import com.vungle.ads.InvalidCTAUrl
import com.vungle.ads.LinkError
import com.vungle.ads.MraidTemplateError
import com.vungle.ads.NetworkUnreachable
import com.vungle.ads.PrivacyUrlError
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.SingleValueMetric
import com.vungle.ads.TpatError
import com.vungle.ads.VungleError
import com.vungle.ads.WebViewError
import com.vungle.ads.WebViewRenderProcessUnresponsive
import com.vungle.ads.WebViewRenderingProcessGone
import com.vungle.ads.internal.ClickCoordinateTracker
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.Constants
import com.vungle.ads.internal.Constants.AD_CLOSE
import com.vungle.ads.internal.Constants.CHECKPOINT_0
import com.vungle.ads.internal.Constants.CLICK_URL
import com.vungle.ads.internal.Constants.CP_0_NOT_FIRED
import com.vungle.ads.internal.Constants.DEEPLINK_CLICK
import com.vungle.ads.internal.Constants.GOOGLE_PLAY_PACKAGE
import com.vungle.ads.internal.Constants.VIDEO_LENGTH_TPAT
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.model.CommonRequestBody
import com.vungle.ads.internal.model.Placement
import com.vungle.ads.internal.network.Call
import com.vungle.ads.internal.network.Callback
import com.vungle.ads.internal.network.Response
import com.vungle.ads.internal.network.TpatRequest
import com.vungle.ads.internal.network.TpatSender
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.omsdk.OMTracker
import com.vungle.ads.internal.platform.Platform
import com.vungle.ads.internal.privacy.PrivacyConsent
import com.vungle.ads.internal.privacy.PrivacyManager
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.protos.Sdk.SDKError
import com.vungle.ads.internal.signals.SignalManager
import com.vungle.ads.internal.ui.PresenterAdOpenCallback
import com.vungle.ads.internal.ui.VungleWebClient
import com.vungle.ads.internal.ui.view.MRAIDAdWidget
import com.vungle.ads.internal.ui.view.WebViewAPI
import com.vungle.ads.internal.util.ExternalRouter
import com.vungle.ads.internal.util.FileUtility
import com.vungle.ads.internal.util.HandlerScheduler
import com.vungle.ads.internal.util.JsonUtil
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.PathProvider
import com.vungle.ads.internal.util.SuspendableTimer
import com.vungle.ads.internal.util.ThreadUtil
import com.vungle.ads.internal.util.Utils
import com.vungle.ads.internal.util.Utils.getWebViewDataSize
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import java.io.File
import java.util.Locale
import java.util.concurrent.Executor
import java.util.concurrent.atomic.AtomicBoolean

class MRAIDPresenter(
    private val adWidget: MRAIDAdWidget,
    private val advertisement: AdPayload,
    private val placement: Placement,
    private val vungleWebClient: VungleWebClient,
    private var executor: Executor,
    private val omTracker: OMTracker,
    private val platform: Platform,
) :
    WebViewAPI.MraidDelegate,
    WebViewAPI.WebClientErrorHandler {

    @VisibleForTesting
    internal var lastUserInteractionTimestamp: Long = 0

    @VisibleForTesting
    var bus: AdEventListener? = null
    private var cp0Fired = false

    @VisibleForTesting
    internal val isDestroying = AtomicBoolean(false)
    private val sendReportIncentivized = AtomicBoolean(false)

    @VisibleForTesting
    internal var adStartTime: Long? = null

    @VisibleForTesting
    internal var userId: String? = null

    private val vungleApiClient: VungleApiClient by inject(adWidget.context)
    private val pathProvider: PathProvider by inject(adWidget.context)
    private val signalManager: SignalManager by inject(adWidget.context)
    private val tpatSender: TpatSender by inject(adWidget.context)

    private var presenterDelegate: PresenterDelegate? = null

    private var appStoreDelegate: OpenActivityDelegate? = null

    private val scheduler by lazy { HandlerScheduler() }

    private val logEntry by lazy { advertisement.logEntry }

    @VisibleForTesting
    internal var heartbeatEnabled = false

    @VisibleForTesting
    internal val suspendableTimer by lazy {
        SuspendableTimer(HEARTBEAT_INTERVAL, true, onFinish = {
            reportErrorAndCloseAd(HeartbeatMissingError())
        })
    }

    /**
     * If true, the back button will close the advertisement.
     */
    @VisibleForTesting
    internal var backEnabled = false

    @VisibleForTesting
    internal val clickCoordinateTracker: ClickCoordinateTracker by lazy {
        ClickCoordinateTracker(adWidget.context, advertisement)
    }

    @VisibleForTesting
    internal var videoLength = 0L

    companion object {
        private const val TAG = "MRAIDPresenter"

        @VisibleForTesting
        internal const val CLOSE = "close"

        @VisibleForTesting
        internal const val CONSENT_ACTION = "consentAction"

        @VisibleForTesting
        internal const val ACTION_WITH_VALUE = "actionWithValue"

        @VisibleForTesting
        internal const val VIDEO_LENGTH = "videoLength"

        @VisibleForTesting
        internal const val TPAT = "tpat"
        private const val ACTION = "action"

        @VisibleForTesting
        internal const val PING_URL = "pingUrl"

        @VisibleForTesting
        internal const val OPEN = "open"
        private const val OPEN_NON_MRAID = "openNonMraid"
        private const val OPEN_APP_STORE = "openAppStore"
        private const val USE_CUSTOM_CLOSE = "useCustomClose"
        private const val USE_CUSTOM_PRIVACY = "useCustomPrivacy"

        @VisibleForTesting
        internal const val OPEN_PRIVACY = "openPrivacy"

        @VisibleForTesting
        internal const val SUCCESSFUL_VIEW = "successfulView"

        @VisibleForTesting
        internal const val SET_ORIENTATION_PROPERTIES = "setOrientationProperties"

        @VisibleForTesting
        internal const val CREATIVE_HEARTBEAT = "creativeHeartbeat"

        @VisibleForTesting
        internal const val GET_AVAILABLE_DISK_SPACE = "getAvailableDiskSpace"

        @VisibleForTesting
        internal const val UPDATE_SIGNALS = "updateSignals"

        @VisibleForTesting
        internal const val ERROR = "error"

        private const val HEARTBEAT_INTERVAL = 6.0

        @VisibleForTesting
        internal val eventMap = mapOf(
            CHECKPOINT_0 to Sdk.SDKMetric.SDKMetricType.AD_START_EVENT,
            CLICK_URL to Sdk.SDKMetric.SDKMetricType.AD_CLICK_EVENT
        )
    }

    fun setEventListener(listener: AdEventListener?) {
        bus = listener
    }

    internal fun setPresenterDelegate(presenterDelegate: PresenterDelegate?) {
        this.presenterDelegate = presenterDelegate
    }

    internal fun setOpenActivityDelegate(appStoreDelegate: OpenActivityDelegate?) {
        this.appStoreDelegate = appStoreDelegate
    }

    fun onViewConfigurationChanged() {
        vungleWebClient.notifyPropertiesChange(true)
    }

    fun start() {
        Logger.d(TAG, "start()")
        adWidget.resumeWeb()
        setAdVisibility(true)
    }

    fun stop() {
        Logger.d(TAG, "stop()")
        adWidget.pauseWeb()
        setAdVisibility(false)
    }

    fun detach(@MRAIDAdWidget.AdStopReason stopReason: Int) {
        Logger.d(TAG, "detach()")
        val isChangingConfigurations =
            stopReason and MRAIDAdWidget.AdStopReason.IS_CHANGING_CONFIGURATION != 0
        val isFinishing = stopReason and MRAIDAdWidget.AdStopReason.IS_AD_FINISHING != 0

        vungleWebClient.setWebViewObserver(null)
        vungleWebClient.setMraidDelegate(null)

        if (!isChangingConfigurations && isFinishing && !isDestroying.getAndSet(true)) {
            bus?.onNext("end", null, placement.referenceId)
        }

        val trackerDelay = omTracker.stop()
        val isMaliBuggy = platform.isProblematicMaliDevice()
        adWidget.destroyWebView(trackerDelay, isMaliBuggy)
        if (heartbeatEnabled) {
            suspendableTimer.cancel()
        }
    }

    fun setAdVisibility(isViewable: Boolean) {
        vungleWebClient.setAdVisibility(isViewable)
    }

    fun onViewTouched(event: MotionEvent?) {
        event?.let {
            Logger.d(TAG, "user interaction")
            lastUserInteractionTimestamp = System.currentTimeMillis()
            clickCoordinateTracker.trackCoordinate(it)
        }
    }

    private fun sendAdCloseEvent() {
        adStartTime?.also {
            val duration = System.currentTimeMillis() - it
            advertisement.getTpatUrls(
                AD_CLOSE,
                duration.toString(),
                platform.volumeLevel.toString(),
            )?.forEach { url ->
                val request = TpatRequest.Builder(url)
                    .tpatKey(AD_CLOSE)
                    .withLogEntry(logEntry)
                    .build()
                tpatSender.sendTpat(request)
            }
        }
    }

    private fun closeView() {
        if (ThreadUtil.isMainThread) {
            executor.execute { sendAdCloseEvent() }
        } else {
            sendAdCloseEvent()
        }

        ThreadUtil.runOnUiThread {
            adWidget.close()
        }
    }

    fun handleExit() {
        if (backEnabled) {
            // Pass the action through the bridge. The template will then decide what to do.
            adWidget.showWebsite("javascript:window.vungle.mraidBridgeExt.requestMRAIDClose()")
        }
    }

    private fun createDeeplinkCallback(deeplinkUrl: String?): PresenterAdOpenCallback {
        return object : PresenterAdOpenCallback {
            override fun onDeeplinkClick(opened: Boolean) {
                if (!opened) {
                    LinkError(
                        SDKError.Reason.DEEPLINK_OPEN_FAILED,
                        "Fail to open $deeplinkUrl",
                    ).setLogEntry(logEntry).logErrorNoReturnValue()
                }

                val deeplinkClickTpatUrls = advertisement.getTpatUrls(
                    DEEPLINK_CLICK,
                    opened.toString()
                )
                deeplinkClickTpatUrls?.forEach { url ->
                    val request = TpatRequest.Builder(url)
                        .tpatKey(DEEPLINK_CLICK)
                        .withLogEntry(logEntry)
                        .build()
                    tpatSender.sendTpat(request)
                }
            }
        }
    }

    @VisibleForTesting
    internal fun checkInlineInstallIntent(intent: Intent): Pair<Boolean, String?> {
        val componentName = intent.resolveActivity(adWidget.context.packageManager)
        val packageName = componentName?.packageName
        return Pair(packageName == GOOGLE_PLAY_PACKAGE, packageName)
    }

    private fun launchInlineInstall(url: String?): Boolean {
        val intent = ExternalRouter.getIntentFromUrl(adWidget.context, url)
        if (intent == null) {
            logInlineInstallFailure("url: $url, message: intent is null")
            return false
        }

        val (ret, packageName) = checkInlineInstallIntent(intent)
        if (!ret) {
            logInlineInstallFailure("url: $url, message: resolveInfo $packageName")
            return false
        }

        val (opened, error) = appStoreDelegate?.openInlineInstall(intent) ?: Pair(false, null)
        if (!opened) {
            logInlineInstallFailure("url: $url, message: $error")
            return false
        }

        logInlineInstallSuccess("url: $url")
        return true
    }

    internal fun logInlineInstallSuccess(message: String? = null) {
        val metric = SingleValueMetric(Sdk.SDKMetric.SDKMetricType.INLINE_INSTALL_STATUS).also {
            it.value = Constants.INLINE_INSTALL_STATUS_SUCCESS
        }
        AnalyticsClient.logMetric(metric, logEntry, message)
    }

    private fun logInlineInstallFailure(message: String? = null) {
        ThreadUtil.runOnUiThread {
            vungleWebClient.notifyPresentAppStoreFailed()
        }
        val metric = SingleValueMetric(Sdk.SDKMetric.SDKMetricType.INLINE_INSTALL_STATUS).also {
            it.value = Constants.INLINE_INSTALL_STATUS_FAILURE
        }
        AnalyticsClient.logMetric(metric, logEntry, message)
        AnalyticsClient.logError(
            SDKError.Reason.INLINE_INSTALL_ERROR, "$message", logEntry
        )
    }

    private fun triggerEventMetricForTpat(key: String) {
        val event = eventMap[key]
        if (event != null) {
            AnalyticsClient.logMetric(SingleValueMetric(event), logEntry)
        }
    }

    fun getViewStatus(): Long? = when {
        !cp0Fired -> CP_0_NOT_FIRED
        else -> null
    }

    override fun processCommand(command: String, arguments: JsonObject): Boolean {
        when (command) {
            CLOSE -> {
                closeView()
                return true
            }

            CONSENT_ACTION -> {
                val action: String? = JsonUtil.getContentStringValue(arguments, "event")
                val consentStatus =
                    if (action == PrivacyConsent.OPT_OUT.getValue()) PrivacyConsent.OPT_OUT.getValue() else PrivacyConsent.OPT_IN.getValue()
                PrivacyManager.updateGdprConsent(consentStatus, "vungle_modal", null)
                return true
            }

            ACTION_WITH_VALUE -> {
                val action: String? = JsonUtil.getContentStringValue(arguments, "event")
                val value: String? = JsonUtil.getContentStringValue(arguments, "value")

                if (VIDEO_LENGTH.equals(action, true)) {
                    videoLength = runCatching { value?.toLong() }.getOrNull() ?: 0L
                }

                return true
            }

            TPAT -> {
                val event = JsonUtil.getContentStringValue(arguments, "event")

                if (event.isNullOrEmpty()) {
                    TpatError(
                        SDKError.Reason.EMPTY_TPAT_ERROR,
                        "Empty tpat key"
                    ).setLogEntry(logEntry).logErrorNoReturnValue()
                    return true
                }

                triggerEventMetricForTpat(event)

                val tpatUrls: List<String>? = when (event) {
                    CHECKPOINT_0 -> advertisement.getTpatUrls(
                        event,
                        platform.carrierName,
                        platform.volumeLevel.toString()
                    )

                    VIDEO_LENGTH_TPAT -> advertisement.getTpatUrls(
                        event,
                        videoLength.toString()
                    )

                    else -> advertisement.getTpatUrls(event)
                }

                tpatUrls?.forEach { url ->
                    tpatSender.sendTpat(
                        TpatRequest.Builder(url)
                            .tpatKey(event)
                            .withLogEntry(logEntry)
                            .build()
                    )
                }

                if (event == CHECKPOINT_0 && !cp0Fired) {
                    cp0Fired = true
                    bus?.onNext("adViewed", null, placement.referenceId)
                    ThreadUtil.runOnUiThread { adWidget.visibility = View.VISIBLE }
                }

                return true
            }

            PING_URL -> {
                val requestType: String? =
                    JsonUtil.getContentStringValue(arguments, "requestType")?.uppercase()
                if (requestType !in listOf("GET", "POST")) {
                    TpatError(
                        SDKError.Reason.TPAT_ERROR,
                        "Invalid request type: $requestType. Only 'GET' and 'POST' are supported"
                    ).setLogEntry(logEntry).logErrorNoReturnValue()
                    return true
                }

                val url: String? = JsonUtil.getContentStringValue(arguments, "url")
                val requestBody: String? = JsonUtil.getContentStringValue(arguments, "requestData")
                val retry: Boolean = JsonUtil.getContentStringValue(arguments, "retry").toBoolean()

                // Get headers
                val headerString: String? = JsonUtil.getContentStringValue(arguments, "headers")
                val headers: Map<String, String>?
                try {
                    headers = headerString?.let { Json.decodeFromString<Map<String, String>>(it) }
                } catch (_: Exception) {
                    TpatError(
                        SDKError.Reason.TPAT_ERROR,
                        "Failed to decode header: $headerString"
                    ).setLogEntry(logEntry).logErrorNoReturnValue()
                    return true
                }

                if (!Utils.isUrlValid(url)) {
                    TpatError(
                        SDKError.Reason.EMPTY_TPAT_ERROR,
                        "URL is missing in params from a template for generic tpat"
                    ).setLogEntry(logEntry).logErrorNoReturnValue()
                    return true
                }

                url?.let {
                    val request = TpatRequest.Builder(url)
                        .headers(headers)
                        .body(requestBody)
                        .regularRetry(retry)
                        .tpatKey(PING_URL)
                        .withLogEntry(logEntry)
                        .also {
                            if (requestType == "GET") it.get() else it.post()
                        }
                        .build()
                    tpatSender.sendTpat(request)
                }
                return true
            }

            ACTION -> {
                return true
            }

            OPEN, OPEN_NON_MRAID -> {
                val deeplinkUrl: String? = advertisement.adUnit()?.deeplinkUrl
                val url: String? = JsonUtil.getContentStringValue(arguments, "url")

                if (!FileUtility.isValidUrl(url)) {
                    InvalidCTAUrl("Invalid CTA Url ($url)").setLogEntry(logEntry)
                        .logErrorNoReturnValue()
                }
                // Auto redirect block is only valid for mraid:open case.
                if (shouldBlockAutoRedirect()) {
                    lastUserInteractionTimestamp = 0L
                    AnalyticsClient.logMetric(
                        SingleValueMetric(Sdk.SDKMetric.SDKMetricType.BANNER_AUTO_REDIRECT),
                        logEntry
                    )
                    return true
                }
                lastUserInteractionTimestamp = 0L
                val launched = ExternalRouter.launch(deeplinkUrl, url, adWidget.context, logEntry,
                    createDeeplinkCallback(deeplinkUrl)
                )

                bus?.onNext("open", "adClick", placement.referenceId)

                if (launched) {
                    bus?.onNext("open", "adLeftApplication", placement.referenceId)
                }

                return true
            }

            OPEN_APP_STORE -> {
                val deeplinkUrl: String? = advertisement.adUnit()?.deeplinkUrl
                val inlineURL = JsonUtil.getContentStringValue(arguments, "url")

                if (!FileUtility.isValidUrl(inlineURL)) {
                    InvalidCTAUrl("Invalid InlineInstall Url ($inlineURL)").setLogEntry(logEntry)
                        .logErrorNoReturnValue()
                }

                // record last user interaction time for those auto click cases triggered by template
                lastUserInteractionTimestamp = System.currentTimeMillis()

                // launch deeplink only
                var launched = ExternalRouter.launch(deeplinkUrl, null, adWidget.context, logEntry,
                    createDeeplinkCallback(deeplinkUrl)
                )
                if (!launched) {
                    launched = launchInlineInstall(inlineURL)
                }

                if (launched) {
                    bus?.onNext("open", "adClick", placement.referenceId)
                    bus?.onNext("open", "adLeftApplication", placement.referenceId)
                }

                return true
            }

            USE_CUSTOM_CLOSE -> {
                return true
            }

            USE_CUSTOM_PRIVACY -> {
                return true
            }

            OPEN_PRIVACY -> {
                AnalyticsClient.logMetric(
                    SingleValueMetric(Sdk.SDKMetric.SDKMetricType.PRIVACY_URL_OPENED),
                    logEntry
                )
                val url: String? = JsonUtil.getContentStringValue(arguments, "url")
                if (url.isNullOrEmpty() || !FileUtility.isValidUrl(url)) {
                    PrivacyUrlError(url ?: "nonePrivacyUrl")
                        .setLogEntry(logEntry).logErrorNoReturnValue()
                    return true
                }
                val launched = ExternalRouter.launch(null, url, adWidget.context, logEntry)
                if (launched) {
                    bus?.onNext("open", "adLeftApplication", placement.referenceId)
                } else {
                    PrivacyUrlError(url).setLogEntry(logEntry).logErrorNoReturnValue()
                }
                return true
            }

            SUCCESSFUL_VIEW -> {
                bus?.onNext("successfulView", null, placement.referenceId)
                if (placement.isRewardedVideo() && ConfigManager.isReportIncentivizedEnabled()
                    && !sendReportIncentivized.getAndSet(true)
                ) {
                    executor.execute {
                        val requestParam = CommonRequestBody.RequestParam(
                            placementReferenceId = placement.referenceId,
                            advAppId = advertisement.advAppId(),
                            adStartTime = adStartTime,
                            user = userId
                        )
                        val riCall = vungleApiClient.ri(requestParam)
                        if (riCall == null) {
                            Logger.e(TAG, "Invalid ri call.")
                            NetworkUnreachable("Error RI API for placement: ${placement.referenceId}")
                                .setLogEntry(logEntry).logErrorNoReturnValue()
                            return@execute
                        }
                        riCall.enqueue(object : Callback<Void> {
                            override fun onResponse(
                                call: Call<Void>?,
                                response: Response<Void>?
                            ) {
                                Logger.d(TAG, "send RI success")
                            }

                            override fun onFailure(call: Call<Void>?, t: Throwable?) {
                                Logger.d(TAG, "send RI Failure")
                                NetworkUnreachable("Error RI API calls: ${t?.localizedMessage}")
                                    .setLogEntry(logEntry).logErrorNoReturnValue()
                            }
                        })
                    }
                }
                return true
            }

            UPDATE_SIGNALS -> {
                val signals: String? = JsonUtil.getContentStringValue(arguments, "signals")
                if (!signals.isNullOrEmpty()) {
                    signalManager.updateTemplateSignals(signals)
                }
                return true
            }

            ERROR -> {
                /*
                * mraid://error?code=60007&errorMessage=Programmatic%20fullscreen%20failed%20to%20decode&fatal=true
                * */
                val errorCode: String? = JsonUtil.getContentStringValue(arguments, "code")
                val fatal: String? = JsonUtil.getContentStringValue(arguments, "fatal")
                val isFatal = fatal.toBoolean()
                val errorMsg: String? = JsonUtil.getContentStringValue(arguments, "errorMessage")
                val reason =
                    if (isFatal) SDKError.Reason.AD_CLOSED_TEMPLATE_ERROR
                    else SDKError.Reason.MRAID_ERROR
                val message = "$errorCode : $errorMsg"
                val exception = MraidTemplateError(reason, message)

                ThreadUtil.runOnUiThread {
                    handleWebViewException(exception, isFatal, message)
                }
                return true
            }

            SET_ORIENTATION_PROPERTIES -> {
                val forceOrientation: String? =
                    JsonUtil.getContentStringValue(arguments, "forceOrientation")
                if (!forceOrientation.isNullOrEmpty()) {
                    when (forceOrientation.lowercase(Locale.ENGLISH)) {
                        "landscape" -> {
                            adWidget.setOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE)
                        }

                        "portrait" -> {
                            adWidget.setOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT)
                        }
                    }
                }
                return true
            }

            CREATIVE_HEARTBEAT -> {
                if (heartbeatEnabled) {
                    ThreadUtil.runOnUiThread { suspendableTimer.reset() }
                }
                return true
            }

            GET_AVAILABLE_DISK_SPACE -> {
                try {
                    val context = adWidget.context
                    val dir =
                        context.noBackupFilesDir

                    val availableDiskSpace = pathProvider.getAvailableBytes(dir.path)

                    val webviewSize = getWebViewDataSize(context)

                    ThreadUtil.runOnUiThread {
                        vungleWebClient.notifyDiskAvailableSize(availableDiskSpace, webviewSize)
                    }

                } catch (e: Exception) {
                    Logger.e(TAG, "Failed to get available disk space: ${e.message}")
                }

                return true
            }

            else -> {
                /// Unknown command, but not a fatal error
                MraidTemplateError(
                    SDKError.Reason.MRAID_JS_CALL_EMPTY,
                    "Unknown MRAID Command: $command"
                ).setLogEntry(logEntry).logErrorNoReturnValue()
                Logger.w(TAG, "processCommand# Unknown MRAID Command: $command")
                return true
            }
        }
    }

    private fun makeBusError(reason: VungleError) {
        bus?.onError(reason, placement.referenceId)
    }

    private fun reportErrorAndCloseAd(reason: VungleError) {
        // Log error
        reason.setLogEntry(logEntry).logErrorNoReturnValue()

        makeBusError(reason)
        closeView()
    }

    override fun onReceivedError(errorDesc: String, didCrash: Boolean) {
        if (didCrash) {
            reportErrorAndCloseAd(WebViewError(errorDesc))
        }
    }

    override fun onWebRenderingProcessGone(view: WebView?, didCrash: Boolean?): Boolean {
        val crashed = didCrash ?: true
        val error = WebViewRenderingProcessGone("didCrash=$crashed")
        handleWebViewException(error, crashed)
        return true
    }

    override fun onRenderProcessUnresponsive(
        webView: WebView?,
        webViewRenderProcess: WebViewRenderProcess?
    ) {
        val exception = WebViewRenderProcessUnresponsive("fatal=true")
        handleWebViewException(exception, true)
    }

    private fun handleWebViewException(
        reason: VungleError,
        fatal: Boolean,
        errorMessage: String? = null
    ) {
        Logger.e(
            TAG,
            "handleWebViewException: ${reason.localizedMessage}, fatal: $fatal, errorMsg: $errorMessage"
        )
        // Log error
        reason.setLogEntry(logEntry).logErrorNoReturnValue()

        if (fatal) {
            makeBusError(reason)
            closeView()
        }
    }

    private fun loadMraidAd(): VungleError? {
        val indexHtml = advertisement.indexFilePath?.let { File(it) }
        if (indexHtml == null || !indexHtml.exists()) {
            return IndexHtmlError(
                SDKError.Reason.AD_HTML_FAILED_TO_LOAD,
                "Fail to load html ${indexHtml?.path}"
            )
        }
        adWidget.linkWebView(vungleWebClient, advertisement.getWebViewSettings())
        adWidget.showWebsite("file://" + indexHtml.path)
        // Record if this play uses remote url.
        //recordPlayRemoteUrl()
        return null
    }

    fun prepare() {
        isDestroying.set(false)

        // Process the advertisement settings
        advertisement.adConfig?.getSettings()?.let {
            if (it > 0) {
                backEnabled = (it and AdConfig.IMMEDIATE_BACK) == AdConfig.IMMEDIATE_BACK
            }
        }

        heartbeatEnabled = advertisement.heartbeatEnabled()

        // Check if the advertisement has a specified orientation, then lock the window to that
        val requestedOrientation = when (advertisement.adConfig?.adOrientation) {
            AdConfig.PORTRAIT -> {
                ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT
            }

            AdConfig.LANDSCAPE -> {
                ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE
            }

            else -> {
                ActivityInfo.SCREEN_ORIENTATION_SENSOR
            }
        }
        adWidget.setOrientation(requestedOrientation)

        omTracker.start()

        vungleWebClient.setMraidDelegate(this)
        vungleWebClient.setErrorHandler(this)
        vungleWebClient.setAdVisibility(false)

        if (!advertisement.usePreloading()) {
            val loadError = loadMraidAd()
            if (loadError != null) {
                reportErrorAndCloseAd(loadError)
                return
            }

        }

        adStartTime = System.currentTimeMillis()

        userId = presenterDelegate?.getUserId()

        val titleText = presenterDelegate?.getAlertTitleText() ?: ""
        val bodyText = presenterDelegate?.getAlertBodyText() ?: ""
        val continueText = presenterDelegate?.getAlertContinueButtonText() ?: ""
        val closeText = presenterDelegate?.getAlertCloseButtonText() ?: ""
        advertisement.setIncentivizedText(titleText, bodyText, continueText, closeText)

        val collectedConsent =
            ConfigManager.getGDPRIsCountryDataProtected() && "unknown" == PrivacyManager.getConsentStatus()
        vungleWebClient.setConsentStatus(
            collectedConsent,
            ConfigManager.getGDPRConsentTitle(),
            ConfigManager.getGDPRConsentMessage(),
            ConfigManager.getGDPRButtonAccept(),
            ConfigManager.getGDPRButtonDeny()
        )

        if (collectedConsent) {
            PrivacyManager.updateGdprConsent("opted_out_by_timeout", "vungle_modal", "")
        }

        /// Enable the back button once the delay has elapsed. If no delay, enable immediately
        val delay = advertisement.getShowCloseDelay(placement.isRewardedVideo())
        if (delay > 0) {
            /// Use a simple timer to enact the delay mechanism
            scheduler.schedule({ backEnabled = true }, delay.toLong())
        } else {
            backEnabled = true
        }

        bus?.onNext("start", null, placement.referenceId)
        if (heartbeatEnabled) {
            suspendableTimer.start()
        }
    }

    internal fun shouldBlockAutoRedirect(): Boolean {
        // Allow auto-redirects if config permits
        if (ConfigManager.allowAutoRedirects()) return false

        // Block if there was no user interaction
        if (lastUserInteractionTimestamp == 0L) return true

        // Block if too much time has passed since last user interaction
        val millisSinceUserInteraction = System.currentTimeMillis() - lastUserInteractionTimestamp
        return millisSinceUserInteraction > ConfigManager.afterClickDuration()
    }
}
