@file:JvmName("BaseAdLoader")

package com.vungle.ads.internal.load

import android.content.Context
import androidx.annotation.WorkerThread
import com.vungle.ads.AdExpiredError
import com.vungle.ads.AdPayloadError
import com.vungle.ads.AdResponseEmptyError
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.AssetRequestError
import com.vungle.ads.AssetWriteError
import com.vungle.ads.InvalidAssetUrlError
import com.vungle.ads.InvalidEventIdError
import com.vungle.ads.InvalidTemplateURLError
import com.vungle.ads.NativeAssetError
import com.vungle.ads.OmSdkJsError
import com.vungle.ads.PlacementMismatchError
import com.vungle.ads.PrivacyIconFallbackError
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.SingleValueMetric
import com.vungle.ads.TimeIntervalMetric
import com.vungle.ads.VungleError
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.Constants
import com.vungle.ads.internal.NativeAdInternal
import com.vungle.ads.internal.downloader.AssetDownloadListener
import com.vungle.ads.internal.downloader.AssetDownloadListener.DownloadError.Companion.DEFAULT_SERVER_CODE
import com.vungle.ads.internal.downloader.DefaultPrivacyIconInjector.injectPrivacyIcon
import com.vungle.ads.internal.downloader.DownloadRequest
import com.vungle.ads.internal.downloader.Downloader
import com.vungle.ads.internal.executor.Executors
import com.vungle.ads.internal.model.AdAsset
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.network.TpatRequest
import com.vungle.ads.internal.network.TpatSender
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.omsdk.OMInjector
import com.vungle.ads.internal.platform.DeviceCheckUtils
import com.vungle.ads.internal.presenter.PreloadDelegate
import com.vungle.ads.internal.presenter.WebViewManager
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.protos.Sdk.SDKError
import com.vungle.ads.internal.task.JobRunner
import com.vungle.ads.internal.task.ResendTpatJob
import com.vungle.ads.internal.util.FileUtility
import com.vungle.ads.internal.util.LogEntry
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.PathProvider
import com.vungle.ads.internal.util.Utils
import java.io.File
import java.io.IOException
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicLong

abstract class BaseAdLoader(
    val context: Context,
    val vungleApiClient: VungleApiClient,
    val sdkExecutors: Executors,
    private val omInjector: OMInjector,
    private val downloader: Downloader,
    val pathProvider: PathProvider,
    val adRequest: AdRequest
) {

    companion object {
        private const val TAG = "BaseAdLoader"
        private const val DOWNLOADED_FILE_NOT_FOUND = "Downloaded file not found!"
    }

    private val downloadCount: AtomicLong = AtomicLong(0)

    private val downloadRequiredAssets = mutableSetOf<String>()

    private var adLoaderCallback: AdLoaderCallback? = null

    private var notifySuccess = AtomicBoolean(false)
    private var notifyFailed = AtomicBoolean(false)

    private val adAssets: MutableList<AdAsset> = mutableListOf()

    internal var advertisement: AdPayload? = null
    private var fullyDownloaded = AtomicBoolean(true)
    private var requiredAssetDownloaded = AtomicBoolean(true)

    private val requestToResponseMetric: TimeIntervalMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.AD_REQUEST_TO_RESPONSE_DURATION_MS)

    private var mainVideoSizeMetric =
        SingleValueMetric(Sdk.SDKMetric.SDKMetricType.ASSET_FILE_SIZE)

    private var templateHtmlSizeMetric =
        SingleValueMetric(Sdk.SDKMetric.SDKMetricType.TEMPLATE_HTML_SIZE)

    private var assetDownloadDurationMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.ASSET_DOWNLOAD_DURATION_MS)

    private var adRequiredDownloadDurationMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.AD_REQUIRED_DOWNLOAD_DURATION_MS)

    private var adOptionalDownloadDurationMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.AD_OPTIONAL_DOWNLOAD_DURATION_MS)

    private var adPreloadToReadyDurationMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.AD_PRELOAD_TO_READY_DURATION_MS)

    internal var logEntry: LogEntry? = null

    private var loadStart: Long = 0L

    // check if all the assets downloaded successfully.
    private val assetDownloadListener: AssetDownloadListener
        get() {
            return object : AssetDownloadListener {

                private var partialDownloadRecorded = false

                override fun onStart(downloadRequest: DownloadRequest) {
                    Logger.w(TAG, "onStart called: ${downloadRequest.asset.serverPath}")
                    val adAsset = downloadRequest.asset
                    if (adAsset.isRequired && (adAsset.percentage ?: 0) > 0) {
                        downloadRequest.startPartialDownloadRecord()
                    }
                }

                override fun onProgress(
                    progress: AssetDownloadListener.Progress,
                    downloadRequest: DownloadRequest
                ) {
                    val adAsset = downloadRequest.asset
                    Logger.d(TAG, "Download progress: $progress url: ${adAsset.serverPath}")
                    if (adAsset.isRequired && adAsset.percentage != null && adAsset.percentage > 0
                        && progress.progressPercent >= adAsset.percentage
                        && !partialDownloadRecorded
                    ) {
                        partialDownloadRecorded = true
                        downloadRequest.stopPartialDownloadRecord()
                        Logger.w(
                            TAG,
                            "Download progress: hit chunk percentage=${adAsset.percentage} for url: ${adAsset.serverPath}"
                        )
                        sdkExecutors.backgroundExecutor.execute {
                            downloadRequiredAssets.remove(adAsset.serverPath)
                            if (downloadRequiredAssets.isEmpty()) {
                                // onAdLoaded callback will be triggered when isRequired assets are downloaded.
                                if (requiredAssetDownloaded.get()) {
                                    onRequiredDownloadCompleted()
                                } else {
                                    onAdLoadFailed(AssetRequestError("Failed to download required assets."))
                                    cancel()
                                    return@execute
                                }
                            }
                        }
                    }
                }

                override fun onError(
                    error: AssetDownloadListener.DownloadError?,
                    downloadRequest: DownloadRequest
                ) {
                    Logger.e(TAG, "onError called: $error")
                    sdkExecutors.backgroundExecutor.execute {

                        val adAsset = downloadRequest.asset

                        if (adAsset.isPrivacyIcon) {
                            val file = injectPrivacyIcon(pathProvider.getVmDir())
                            if (file != null && file.exists()) {
                                onSuccess(file, downloadRequest)
                                return@execute
                            }

                            PrivacyIconFallbackError(
                                "Failed to inject default privacy icon"
                            ).setLogEntry(logEntry).logErrorNoReturnValue()
                        }

                        fullyDownloaded.set(false)
                        if (adAsset.isRequired) {
                            requiredAssetDownloaded.set(false)
                        }

                        val errorMsg = "Failed to download assets ${adAsset.serverPath}." +
                                " error: $error" +
                                " proxyEnabled=${DeviceCheckUtils.isProxyEnabled(context)}"
                        AssetRequestError(errorMsg).setLogEntry(logEntry).logErrorNoReturnValue()

                        if (adAsset.isRequired) {
                            downloadRequiredAssets.remove(downloadRequest.asset.serverPath)
                            if (downloadRequiredAssets.isEmpty()) {
                                // call failure callback if all required assets are not downloaded.
                                onAdLoadFailed(AssetRequestError("Error: Failed to download required assets."))
                                // cancel the rest of optional download requests
                                cancel()
                                return@execute
                            }
                        }

                        if (downloadCount.decrementAndGet() <= 0) {
                            onAdLoadFailed(AssetRequestError("Error: Failed to download assets."))
                        }
                    }
                }

                override fun onSuccess(file: File, downloadRequest: DownloadRequest) {
                    sdkExecutors.backgroundExecutor.execute {
                        if (!file.exists()) {
                            onError(
                                AssetDownloadListener.DownloadError(
                                    DEFAULT_SERVER_CODE,
                                    IOException(DOWNLOADED_FILE_NOT_FOUND),
                                    AssetDownloadListener.DownloadError.ErrorReason.FILE_NOT_FOUND_ERROR
                                ),
                                downloadRequest
                            )
                            return@execute
                        }

                        val adAsset = downloadRequest.asset
                        adAsset.fileSize = file.length()
                        adAsset.status = AdAsset.Status.DOWNLOAD_SUCCESS

                        if (adAsset.isHtmlTemplate) {
                            downloadRequest.stopTemplateRecord()
                            templateHtmlSizeMetric.value = file.length()
                            AnalyticsClient.logMetric(
                                templateHtmlSizeMetric,
                                logEntry,
                                metaData = adAsset.serverPath
                            )
                        } else if (adAsset.isMainVideo) {
                            mainVideoSizeMetric.value = file.length()
                            AnalyticsClient.logMetric(
                                mainVideoSizeMetric,
                                logEntry,
                                metaData = adAsset.serverPath
                            )
                        }

                        // Update the asset path in the mraid file map
                        advertisement?.updateAdAssetPath(adAsset.adIdentifier, file)

                        if (adAsset.isHtmlTemplate && !processVmTemplate(adAsset, advertisement)) {
                            fullyDownloaded.set(false)
                            if (adAsset.isRequired) {
                                requiredAssetDownloaded.set(false)
                            }
                        }

                        if (adAsset.isRequired) {
                            downloadRequiredAssets.remove(adAsset.serverPath)
                            if (downloadRequiredAssets.isEmpty()) {
                                // onAdLoaded callback will be triggered when isRequired assets are downloaded.
                                if (requiredAssetDownloaded.get()) {
                                    onRequiredDownloadCompleted()
                                } else {
                                    onAdLoadFailed(AssetRequestError("Failed to download required assets."))
                                    cancel()
                                    return@execute
                                }
                            }
                        }

                        // check if all the assets downloaded successfully.
                        if (downloadCount.decrementAndGet() <= 0) {
                            // set advertisement state to READY
                            if (fullyDownloaded.get()) {
                                onDownloadCompleted(adRequest)
                            } else {
                                onAdLoadFailed(AssetRequestError("Failed to download assets."))
                            }
                        }
                    }
                }
            }
        }

    fun loadAd(adLoaderCallback: AdLoaderCallback) {
        this.adLoaderCallback = adLoaderCallback
        loadStart = System.currentTimeMillis()
        sdkExecutors.backgroundExecutor.execute {
            requestToResponseMetric.markStart()
            requestAd()
        }
    }

    protected abstract fun requestAd()

    abstract fun onAdLoadReady()

    fun cancel() {
        downloader.cancelAll()
    }

    private fun downloadAssets() {
        assetDownloadDurationMetric.markStart()
        adRequiredDownloadDurationMetric.markStart()
        adOptionalDownloadDurationMetric.markStart()
        downloadCount.set(adAssets.size.toLong())
        for (asset in adAssets) {
            val downloadRequest = DownloadRequest(getAssetPriority(asset), asset, logEntry)
            if (asset.isHtmlTemplate) {
                downloadRequest.startTemplateRecord()
            }
            if (asset.isRequired) {
                downloadRequiredAssets.add(asset.serverPath)
            }
            downloader.download(downloadRequest, assetDownloadListener)
        }
    }

    fun onAdLoadFailed(error: VungleError) {
        if (!notifySuccess.get() && notifyFailed.compareAndSet(false, true)) {
            adLoaderCallback?.onFailure(error)
        }
    }

    private fun onAdReady() {
        advertisement?.let {
            // onSuccess can only be called once. For ADO case, it will be called right after
            // template is downloaded. After all assets are downloaded, onSuccess will not be called.
            if (!notifyFailed.get() && notifySuccess.compareAndSet(false, true)) {
                if (it.usePreloading()) {

                    adPreloadToReadyDurationMetric.markStart()
                    val templatePath = it.indexFilePath.toString()
                    Logger.w(TAG, "start preloading")

                    val loadDuration = System.currentTimeMillis() - loadStart
                    WebViewManager.preloadWebView(
                        context,
                        it,
                        adRequest.placement,
                        templatePath,
                        it.getWebViewSettings(),
                        object : PreloadDelegate {
                            override fun onAdReadyToPlay() {

                                adPreloadToReadyDurationMetric.markEnd()
                                AnalyticsClient.logMetric(adPreloadToReadyDurationMetric, logEntry)

                                // After real time ad loaded, will send win notifications.
                                // Use this abstract method to notify sub ad loader doing some clean up job.
                                onAdLoadReady()
                                adLoaderCallback?.onSuccess(it)
                            }

                            override fun onAdFailedToPlay() {

                                adPreloadToReadyDurationMetric.markEnd()
                                AnalyticsClient.logMetric(adPreloadToReadyDurationMetric, logEntry)
                                Logger.e(TAG, "fail to load ad")
                                onAdLoadReady()
                                adLoaderCallback?.onSuccess(it)
                            }

                        }, loadDuration
                    )

                } else {
                    onAdLoadReady()
                    adLoaderCallback?.onSuccess(it)
                }

                // Resend failed tpat if any.
                val jobRunner: JobRunner by inject(context)
                jobRunner.execute(ResendTpatJob.makeJobInfo())
            }
        }
    }

    private fun fileIsValid(file: File, adAsset: AdAsset): Boolean {
        return file.exists() && file.length() == adAsset.fileSize
    }

    private fun getDestinationDir(advertisement: AdPayload): File? {
        return pathProvider.getDownloadsDirForAd(advertisement.eventId())
    }

    private fun processVmTemplate(
        asset: AdAsset,
        advertisement: AdPayload?
    ): Boolean {
        if (advertisement == null) {
            return false
        }
        if (asset.status != AdAsset.Status.DOWNLOAD_SUCCESS) {
            return false
        }
        if (asset.localPath.isEmpty()) {
            return false
        }
        val vmTemplate = File(asset.localPath)
        if (!fileIsValid(vmTemplate, asset)) {
            return false
        }

        val destinationDir = getDestinationDir(advertisement)
        if (destinationDir == null || !destinationDir.isDirectory) {
            Logger.e(TAG, "Unable to access Destination Directory")
            return false
        }

        FileUtility.printDirectoryTree(destinationDir)

        return true
    }

    private fun onRequiredDownloadCompleted() {
        adRequiredDownloadDurationMetric.markEnd()
        AnalyticsClient.logMetric(adRequiredDownloadDurationMetric, logEntry)

        onAdReady()
    }

    @WorkerThread
    private fun onDownloadCompleted(request: AdRequest) {
        Logger.d(TAG, "All download completed $request")
        advertisement?.setAssetFullyDownloaded()
        onAdReady()

        assetDownloadDurationMetric.markEnd()
        AnalyticsClient.logMetric(assetDownloadDurationMetric, logEntry)

        adOptionalDownloadDurationMetric.markEnd()
        AnalyticsClient.logMetric(adOptionalDownloadDurationMetric, logEntry)
    }

    internal fun handleAdMetaData(advertisement: AdPayload, metric: SingleValueMetric? = null) {
        this.advertisement = advertisement
        this.advertisement?.recordExpiryWindowStart()
        // remember the log entry instance for the advertisement
        advertisement.logEntry = logEntry
        logEntry?.eventId = advertisement.eventId()
        logEntry?.creativeId = advertisement.getCreativeId()
        logEntry?.adSource = advertisement.getAdSource()
        logEntry?.mediationName = advertisement.getMediationName()
        logEntry?.vmVersion = advertisement.getViewMasterVersion()
        logEntry?.partialDownloadEnabled = advertisement.isPartialDownloadEnabled()
        logEntry?.adoEnabled = advertisement.adLoadOptimizationEnabled()

        requestToResponseMetric.markEnd()
        AnalyticsClient.logMetric(requestToResponseMetric, logEntry)

        // Update config
        advertisement.config()?.let { config ->
            ConfigManager.initWithConfig(context, config, false, metric)
        }

        val error = validateAdMetadata(advertisement)
        if (error != null) {
            onAdLoadFailed(error.setLogEntry(logEntry).logError())
            return
        }

        val destinationDir: File? = getDestinationDir(advertisement)
        if (destinationDir == null || !destinationDir.isDirectory || !destinationDir.exists()) {
            onAdLoadFailed(
                AssetWriteError("Invalid directory. $destinationDir").setLogEntry(logEntry)
                    .logError()
            )
            return
        }

        injectOMSDKIfNeeded()

        val tpatSender: TpatSender by inject(context)
        // URLs for start to load ad notification when load ad after get the adm and
        // before assets start to download.
        advertisement.adUnit()?.loadAdUrls?.forEach { url ->
            val request = TpatRequest.Builder(url)
                .tpatKey(Constants.LOAD_AD)
                .withLogEntry(logEntry)
                .build()
            tpatSender.sendTpat(request)
        }

        if (adAssets.isNotEmpty()) {
            adAssets.clear()
        }
        adAssets.addAll(advertisement.getDownloadableAssets(destinationDir))

        if (adAssets.isEmpty()) {
            onAdReady()
            return
        }

        downloadAssets()
    }

    private fun getAssetPriority(adAsset: AdAsset): DownloadRequest.Priority {
        return if (adAsset.isRequired) {
            DownloadRequest.Priority.CRITICAL
        } else {
            DownloadRequest.Priority.HIGHEST
        }
    }

    private fun injectOMSDKIfNeeded() {
        if (advertisement?.omEnabled() == true) {
            try {
                omInjector.init()
                omInjector.injectJsFiles(pathProvider.getVmDir())
            } catch (e: Exception) {
                Logger.e(TAG, "Failed to inject OMSDK: ${e.message}")
                OmSdkJsError(
                    SDKError.Reason.OMSDK_JS_WRITE_FAILED,
                    "Failed to inject OMSDK: ${e.message}"
                ).setLogEntry(logEntry).logErrorNoReturnValue()
            }
        }
    }

    private fun validateAdMetadata(adPayload: AdPayload): VungleError? {
        adPayload.adUnit()?.sleep?.let {
            return getErrorInfo(adPayload)
        }

        if (adRequest.placement.referenceId != advertisement?.placementId()) {
            val description =
                "Waterfall request and responses placement don't match ${advertisement?.placementId()}."
            return PlacementMismatchError(description)
        }

        val templateSettingsError = getTemplateError(adPayload)
        if (templateSettingsError != null) {
            return templateSettingsError
        }

        if (adPayload.hasExpired()) {
            return AdExpiredError("The ad markup has expired for playback. " +
                    "Ad expiry: ${adPayload.adUnit()?.expiry}, device: ${System.currentTimeMillis()}")
        }

        if (adPayload.eventId().isNullOrEmpty()) {
            return InvalidEventIdError("Event id is invalid.")
        }

        return null
    }

    private fun getTemplateError(adPayload: AdPayload): VungleError? {
        val templateSettings = adPayload.adUnit()?.templateSettings
        if (templateSettings == null) {
            val description = "Missing template settings"
            return AdResponseEmptyError(description)
        }

        val cacheableReplacements = templateSettings.cacheableReplacements
        if (adPayload.isNativeTemplateType()) {
            cacheableReplacements?.let {
                if (it[NativeAdInternal.TOKEN_MAIN_IMAGE]?.url == null) {
                    return NativeAssetError("Unable to load null main image.")
                }
                if (it[NativeAdInternal.TOKEN_VUNGLE_PRIVACY_ICON_URL]?.url == null) {
                    return NativeAssetError("Unable to load null privacy image.")
                }
            }
        } else {
            val vmUrl = adPayload.adUnit()?.vmURL
            if (vmUrl.isNullOrEmpty()) {
                val description = "Failed to prepare null vmURL for downloading."
                return InvalidTemplateURLError(description)
            }

            if (!Utils.isUrlValid(vmUrl)) {
                val description = "Failed to load vm url: $vmUrl"
                return InvalidTemplateURLError(description)
            }
        }

        cacheableReplacements?.forEach {
            val httpUrl = it.value.url
            if (httpUrl.isNullOrEmpty()) {
                return InvalidAssetUrlError("None asset URL for ${it.key}")
            }
            if (!Utils.isUrlValid(httpUrl)) {
                return InvalidAssetUrlError("Invalid asset URL $httpUrl")
            }
        }

        return null
    }

    private fun getErrorInfo(adPayload: AdPayload): VungleError {
        val errorCode: Int? = adPayload.adUnit()?.errorCode
        val sleep = adPayload.adUnit()?.sleep
        val info = adPayload.adUnit()?.info
        val errorMsg = "Response error: $sleep, Request failed with error: $errorCode, $info"
        return when (errorCode) {
            Sdk.SDKError.Reason.AD_NO_FILL_VALUE,
            Sdk.SDKError.Reason.AD_LOAD_TOO_FREQUENTLY_VALUE,
            Sdk.SDKError.Reason.AD_SERVER_ERROR_VALUE,
            Sdk.SDKError.Reason.AD_PUBLISHER_MISMATCH_VALUE,
            Sdk.SDKError.Reason.AD_INTERNAL_INTEGRATION_ERROR_VALUE -> AdPayloadError(
                SDKError.Reason.forNumber(errorCode),
                errorMsg
            )

            else -> AdPayloadError(SDKError.Reason.PLACEMENT_SLEEP, errorMsg)
        }

    }
}
