package com.vungle.ads.internal

import android.content.Context
import com.vungle.ads.AdConfig
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.BaseAd
import com.vungle.ads.VungleAdSize
import com.vungle.ads.VungleError
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.model.Placement
import com.vungle.ads.internal.presenter.AdPlayCallback
import com.vungle.ads.internal.presenter.AdPlayCallbackWrapper
import com.vungle.ads.internal.util.ThreadUtil
import com.vungle.ads.internal.util.ViewUtility
import kotlin.math.min

internal class BannerAdImpl(
    context: Context,
    placementId: String,
    private val adSize: VungleAdSize,
    adConfig: AdConfig
) :
    BaseAd(context, placementId, adConfig) {

    override fun constructAdInternal(context: Context) = BannerAdInternal(context, adSize)

    fun getAdViewSize(): VungleAdSize {
        return (adInternal as BannerAdInternal).updatedAdSize ?: adSize
    }

    internal val adPlayCallback = (adInternal as BannerAdInternal).wrapCallback(
        object : AdPlayCallback {
            override fun onAdStart(id: String?) {
                signalManager.increaseSessionDepthCounter()
                adInternal.validationToPresentMetric.markEnd()
                AnalyticsClient.logMetric(adInternal.validationToPresentMetric, logEntry)

                presentToDisplayMetric.markStart()
                ThreadUtil.runOnUiThread {
                    adListener?.onAdStart(this@BannerAdImpl)
                }
            }

            override fun onAdImpression(id: String?) {
                ThreadUtil.runOnUiThread {
                    adListener?.onAdImpression(this@BannerAdImpl)
                }
                presentToDisplayMetric.markEnd()
                AnalyticsClient.logMetric(presentToDisplayMetric, logEntry)
                displayToClickMetric.markStart()
            }

            override fun onAdEnd(id: String?) {
                ThreadUtil.runOnUiThread {
                    adListener?.onAdEnd(this@BannerAdImpl)
                }
                showToCloseMetric.markEnd()
                AnalyticsClient.logMetric(showToCloseMetric, logEntry)
            }

            override fun onAdClick(id: String?) {
                ThreadUtil.runOnUiThread {
                    adListener?.onAdClicked(this@BannerAdImpl)
                }
                displayToClickMetric.markEnd()
                AnalyticsClient.logMetric(displayToClickMetric, logEntry)
            }

            override fun onAdRewarded(id: String?) {
                // no-op
            }

            override fun onAdLeftApplication(id: String?) {
                ThreadUtil.runOnUiThread {
                    adListener?.onAdLeftApplication(this@BannerAdImpl)
                }
                AnalyticsClient.logMetric(leaveApplicationMetric, logEntry)
            }

            override fun onFailure(error: VungleError) {
                ThreadUtil.runOnUiThread {
                    adListener?.onAdFailedToPlay(this@BannerAdImpl, error)
                }

                showToFailMetric.markEnd()
                AnalyticsClient.logMetric(showToFailMetric, logEntry, "${error.code}-${error.errorMessage}")
            }
        })

}

internal class BannerAdInternal(context: Context, private val adSize: VungleAdSize) :
    AdInternal(context) {

    internal var updatedAdSize: VungleAdSize? = null

    internal fun wrapCallback(adPlayCallback: AdPlayCallback) =
        object : AdPlayCallbackWrapper(adPlayCallback) {
            override fun onAdStart(id: String?) {
                adState = AdState.PLAYING
                super.onAdStart(id)
            }

            override fun onAdEnd(id: String?) {
                adState = AdState.FINISHED
                super.onAdEnd(id)
            }

            override fun onFailure(error: VungleError) {
                adState = AdState.ERROR
                super.onFailure(error)
            }
        }

    override fun isValidAdTypeForPlacement(placement: Placement): Boolean {
        return placement.isBanner() || placement.isMREC() || placement.isInline()
    }

    override fun isValidAdSize(adSize: VungleAdSize?): Boolean {
        val validSize = adSize?.isValidSize() ?: false

        return validSize
    }

    override fun getAdSizeForAdRequest(): VungleAdSize {
        return adSize
    }

    override fun adLoadedAndUpdateConfigure(advertisement: AdPayload) {
        super.adLoadedAndUpdateConfigure(advertisement)

        // Copy adSize to updatedAdSize with ad_size from server side
        if (adSize.isAdaptiveWidth || adSize.isAdaptiveHeight) {
            val (deviceWidth, deviceHeight) = ViewUtility.getDeviceWidthAndHeightWithOrientation(
                context,
                0
            )
            val w = if (adSize.isAdaptiveWidth) advertisement.adWidth() else adSize.width
            val h = if (adSize.isAdaptiveHeight) advertisement.adHeight() else adSize.height
            val adWidth = min(deviceWidth, w)
            var adHeight = min(deviceHeight, h)
            if (adSize.isAdaptiveHeight && adSize.height > 0) {
                // If adaptive height is enabled and height is set, then we need to limit
                // the height no more than the max height
                adHeight = min(adSize.height, adHeight)
            }

            updatedAdSize = VungleAdSize(adWidth, adHeight)
        }
    }

}
