package com.vungle.ads.internal.model

import com.vungle.ads.internal.Constants
import com.vungle.ads.internal.util.Logger
import java.util.concurrent.atomic.AtomicBoolean

/**
 * A Placement is an object that represents a single ad configuration. Publishers use placements to
 * test ads in different locations within their app, or to try different kinds of campaigns for
 * A/B testing. The identifier is generated by the Vungle dashboard, and also contains a collection of
 * assets.
 *
 * adIdentifier: Ad asset TOKEN in cacheable placements or template.
 * serverPath: Server path of asset
 * localPath: Local path of asset
 * fileType: Type of asset
 * isRequired: If true, then this asset is required to load before playing the ad.
 * percentage: Percentage(0~100) of the chunk to download the asset. If hit the size we can trigger load callback.
 *            If null and is required, then the asset will be fully downloaded. Such as template and native ad.
 */
class AdAsset(
    val adIdentifier: String,
    val serverPath: String,
    val localPath: String,
    val isRequired: Boolean,
    val percentage: Int? = null,
) {

    /**
     * Download Status
     */
    enum class Status {
        NEW, //for internal usage in AssetDownloader only
        DOWNLOAD_RUNNING,
        DOWNLOAD_FAILED,
        DOWNLOAD_SUCCESS,
        PROCESSED
    }

    var status: Status = Status.NEW

    /**
     * File size to check the integrity of file
     */
    var fileSize: Long = 0

    /**
     * Content length of the asset
     */
    var contentLength: Long = 0L

    /**
     * Range start for the asset download.
     */
    var rangeStart: Long = 0L

    /**
     * Range end for the asset download.
     */
    var rangeEnd: Long? = null

    private val waitLock = Object()

    /**
     * Atomic boolean to indicate if the download is enough.
     * This is used to notify the waiting thread when the download has reached the required size.
     */
    private val waitingForDownload = AtomicBoolean(false)

    /**
     * Waits for the download to match the [rangeStart]. This method blocks until the download is marked as enough
     * using [notifyDownloadEnough].
     */
    fun waitForDownload() {
        waitingForDownload.set(true)
        synchronized(waitLock) {
            runCatching {
                waitLock.wait()
            }.onFailure { e ->
                Logger.e("AdAsset", "Interrupted while waiting for file download: $this", e)
            }
        }
    }

    /**
     * Notifies that the download is enough, allowing the waiting thread to continue.
     * This method should be called when the download has reached the required size.
     */
    fun notifyDownloadEnough() {
        if (waitingForDownload.getAndSet(false)) {
            synchronized(waitLock) {
                waitLock.notifyAll()
            }
        }
    }

    fun isWaitingForDownload(): Boolean {
        return waitingForDownload.get()
    }

    val isMainVideo: Boolean
        get() = adIdentifier == Constants.KEY_MAIN_VIDEO

    val isHtmlTemplate: Boolean
        get() = adIdentifier == AdPayload.KEY_VM

    val isPrivacyIcon: Boolean
        get() = adIdentifier == Constants.KEY_PRIVACY_ICON_URL

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other == null || javaClass != other.javaClass) return false
        val adAsset = other as AdAsset
        if (status != adAsset.status) return false
        if (fileSize != adAsset.fileSize) return false
        if (isRequired != adAsset.isRequired) return false
        if (adIdentifier != adAsset.adIdentifier) return false
        if (serverPath != adAsset.serverPath) return false
        if (contentLength != adAsset.contentLength) return false
        if (percentage != adAsset.percentage) return false
        return localPath == adAsset.localPath
    }

    override fun hashCode(): Int {
        var result = adIdentifier.hashCode()
        result = 31 * result + serverPath.hashCode()
        result = 31 * result + localPath.hashCode()
        result = 31 * result + status.hashCode()
        result = 31 * result + (fileSize xor (fileSize ushr 32)).toInt()
        result = 31 * result + isRequired.hashCode()
        result = 31 * result + (contentLength xor (contentLength ushr 32)).toInt()
        result = 31 * result + (percentage?.hashCode() ?: 0)
        return result
    }

    override fun toString(): String {
        return  "AdAsset(adIdentifier=$adIdentifier, serverPath=$serverPath, localPath=$localPath, " +
                "status=$status, fileSize=$fileSize, contentLength=$contentLength, isRequired=$isRequired, " +
                "percentage=$percentage)"
    }

}
