package com.vungle.ads.internal.downloader

import com.vungle.ads.AnalyticsClient
import com.vungle.ads.TimeIntervalMetric
import com.vungle.ads.internal.model.AdAsset
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.util.LogEntry
import java.util.concurrent.atomic.AtomicBoolean

class DownloadRequest(val priority: Priority, val asset: AdAsset, internal val logEntry: LogEntry? = null) {

    private val cancelled = AtomicBoolean(false)

    private var downloadTemplateDuration: TimeIntervalMetric? = null

    private var partialDownloadDurationMetric: TimeIntervalMetric? = null

    override fun toString(): String {
        return "DownloadRequest{" +
                ", priority=" + priority +
                ", url='" + asset.serverPath + '\'' +
                ", path='" + asset.localPath + '\'' +
                ", cancelled=" + cancelled +
                ", logEntry=" + logEntry +
                '}'
    }

    enum class Priority(val priority: Int) {
        CRITICAL(-Int.MAX_VALUE), //for internal usage in AssetDownloader only
        HIGHEST(0),
        HIGH(1),
        LOWEST(Int.MAX_VALUE)

    }

    fun cancel() {
        cancelled.set(true)
    }

    fun isCancelled(): Boolean {
        return cancelled.get()
    }

    fun getPriority(): Int {
        return priority.priority
    }

    fun startPartialDownloadRecord() {
        partialDownloadDurationMetric =
            TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.ASSET_PARTIAL_DOWNLOAD_DURATION_MS)
        partialDownloadDurationMetric?.markStart()
    }

    fun stopPartialDownloadRecord() {
        partialDownloadDurationMetric?.let {
            it.markEnd()
            AnalyticsClient.logMetric(
                timeIntervalMetric = it,
                logEntry = logEntry,
                metaData = "percentage=${asset.percentage} url=${asset.serverPath}"
            )
        }
    }

    fun startTemplateRecord() {
        downloadTemplateDuration =
            TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.TEMPLATE_DOWNLOAD_DURATION_MS)
        downloadTemplateDuration?.markStart()
    }

    fun stopTemplateRecord() {
        downloadTemplateDuration?.let {
            it.markEnd()
            AnalyticsClient.logMetric(
                timeIntervalMetric = it,
                logEntry,
                metaData = asset.serverPath
            )
        }
    }

}
