package com.vungle.ads.internal.downloader


import com.vungle.ads.AnalyticsClient
import com.vungle.ads.AssetRequestError
import com.vungle.ads.AssetWriteError
import com.vungle.ads.InvalidAssetUrlError
import com.vungle.ads.NoSpaceError
import com.vungle.ads.OutOfMemory
import com.vungle.ads.SingleValueMetric
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.Constants.TIMEOUT
import com.vungle.ads.internal.downloader.AssetDownloadListener.DownloadError.Companion.DEFAULT_SERVER_CODE
import com.vungle.ads.internal.downloader.AssetDownloadListener.DownloadError.ErrorReason.Companion.DISK_ERROR
import com.vungle.ads.internal.downloader.AssetDownloadListener.DownloadError.ErrorReason.Companion.FILE_NOT_FOUND_ERROR
import com.vungle.ads.internal.downloader.AssetDownloadListener.DownloadError.ErrorReason.Companion.INTERNAL_ERROR
import com.vungle.ads.internal.downloader.AssetDownloadListener.DownloadError.ErrorReason.Companion.REQUEST_ERROR
import com.vungle.ads.internal.executor.VungleThreadPoolExecutor
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.task.PriorityRunnable
import com.vungle.ads.internal.util.FileUtility
import com.vungle.ads.internal.util.FileUtility.isValidUrl
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.PathProvider
import okhttp3.Cache
import okhttp3.Call
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.Response
import okhttp3.ResponseBody
import okhttp3.internal.http.RealResponseBody
import okio.BufferedSink
import okio.BufferedSource
import okio.GzipSource
import okio.buffer
import okio.sink
import java.io.File
import java.net.ProtocolException
import java.util.concurrent.TimeUnit
import kotlin.math.min

class AssetDownloader(
    private val downloadExecutor: VungleThreadPoolExecutor,
    private val pathProvider: PathProvider
) : Downloader {

    companion object {
        private const val TAG = "AssetDownloader"
        private const val MINIMUM_SPACE_REQUIRED_MB = 20 * 1024 * 1024
        private const val DOWNLOAD_CHUNK_SIZE = 2048 //Same as Okio Segment.SIZE
        private const val CONTENT_ENCODING = "Content-Encoding"
        private const val CONTENT_TYPE = "Content-Type"
        private const val GZIP = "gzip"
        private const val PROGRESS_STEP = 1
        private const val MAX_PERCENT = 100

    }

    private val okHttpClient by lazy { OkHttpSingleton.createOkHttpClient(pathProvider) }

    private val transitioning = mutableListOf<DownloadRequest>()

    private object OkHttpSingleton {
        private var client: OkHttpClient? = null
        fun createOkHttpClient(pathProvider: PathProvider): OkHttpClient {
            return client ?: run {
                val builder = OkHttpClient.Builder()
                    .readTimeout(TIMEOUT.toLong(), TimeUnit.SECONDS)
                    .connectTimeout(TIMEOUT.toLong(), TimeUnit.SECONDS)
                    .cache(null)
                    .followRedirects(true)
                    .followSslRedirects(true)
                if (ConfigManager.isCleverCacheEnabled()) {
                    val diskSize = ConfigManager.getCleverCacheDiskSize()
                    val diskPercentage = ConfigManager.getCleverCacheDiskPercentage()
                    val maxDiskCapacity =
                        pathProvider.getAvailableBytes(pathProvider.getCleverCacheDir().absolutePath) * diskPercentage / 100
                    val diskCapacity = min(diskSize, maxDiskCapacity)
                    if (diskCapacity > 0) {
                        builder.cache(Cache(pathProvider.getCleverCacheDir(), diskCapacity))
                    } else {
                        Logger.w(
                            "OkHttpClientWrapper",
                            "cache disk capacity size <=0, no clever cache active."
                        )
                    }
                }
                builder.build()
            }.also { client = it }
        }
    }

    override fun download(
        downloadRequest: DownloadRequest?,
        downloadListener: AssetDownloadListener?
    ) {

        if (downloadRequest == null) {
            return
        }

        transitioning.add(downloadRequest)

        downloadExecutor.execute(object : PriorityRunnable() {
            override fun run() {
                launchRequest(downloadRequest, downloadListener)
            }

            override val priority: Int
                get() = downloadRequest.getPriority()
        }) {
            val errorMsg = "Failed to execute download request: ${downloadRequest.asset.serverPath}"
            deliverError(
                downloadRequest, downloadListener, AssetDownloadListener.DownloadError(
                    // No need to log to server here because AdLoader will log the error
                    DEFAULT_SERVER_CODE, OutOfMemory(errorMsg), INTERNAL_ERROR
                )
            )
        }
    }

    private fun deliverError(
        downloadRequest: DownloadRequest,
        downloadListener: AssetDownloadListener?,
        downloadError: AssetDownloadListener.DownloadError?
    ) {
        downloadListener?.onError(downloadError, downloadRequest)
    }

    override fun cancel(request: DownloadRequest?) {
        if (request == null || request.isCancelled()) return

        request.cancel()
    }

    override fun cancelAll() {
        transitioning.forEach {
            cancel(it)
        }
        transitioning.clear()
    }

    private fun launchRequest(
        downloadRequest: DownloadRequest,
        downloadListener: AssetDownloadListener?
    ) {
        val downloadAsset = downloadRequest.asset
        Logger.d(
            TAG,
            "launch request in thread: ${Thread.currentThread().id} request: ${downloadAsset.serverPath}"
        )

        if (downloadRequest.isCancelled()) {
            Logger.d(
                TAG,
                "Request ${downloadAsset.serverPath} is cancelled before starting"
            )
            val progress = AssetDownloadListener.Progress()
            progress.status = AssetDownloadListener.Progress.ProgressStatus.CANCELLED
            return
        }

        val progress = AssetDownloadListener.Progress()
        progress.timestampDownloadStart = System.currentTimeMillis()
        var downloadError: AssetDownloadListener.DownloadError? = null

        val url: String = downloadAsset.serverPath
        val path: String = downloadAsset.localPath

        if (url.isEmpty() || !isValidUrl(url)) {
            deliverError(
                downloadRequest, downloadListener, AssetDownloadListener.DownloadError(
                    DEFAULT_SERVER_CODE,
                    InvalidAssetUrlError("invalid url: $url").setLogEntry(downloadRequest.logEntry)
                        .logError(),
                    INTERNAL_ERROR
                )
            )
            return
        }
        if (path.isEmpty()) {
            deliverError(
                downloadRequest, downloadListener, AssetDownloadListener.DownloadError(
                    DEFAULT_SERVER_CODE,
                    AssetWriteError("invalid path: $path").setLogEntry(downloadRequest.logEntry)
                        .logError(),
                    FILE_NOT_FOUND_ERROR
                )
            )
            return
        }
        if (!checkSpaceAvailable(downloadRequest)) {
            deliverError(
                downloadRequest, downloadListener, AssetDownloadListener.DownloadError(
                    DEFAULT_SERVER_CODE,
                    NoSpaceError().setLogEntry(downloadRequest.logEntry).logError(),
                    DISK_ERROR
                )
            )
            return
        }
        val file = File(path)

        var sink: BufferedSink? = null
        var source: BufferedSource? = null
        var totalRead: Long = 0
        var call: Call? = null
        var code: Int = DEFAULT_SERVER_CODE
        var response: Response? = null
        downloadListener?.onStart(downloadRequest)
        try {
            val parentFile = file.parentFile
            if (parentFile != null && !parentFile.exists()) {
                parentFile.mkdirs()
            }
            val requestBuilder = Request.Builder().url(url)

            call = okHttpClient.newCall(requestBuilder.build())
            response = call.execute()

            code = response.code
            if (!response.isSuccessful) {
                throw Downloader.RequestException(response.message)
            }

            response.cacheResponse?.let {
                AnalyticsClient.logMetric(
                    SingleValueMetric(Sdk.SDKMetric.SDKMetricType.CACHED_ASSETS_USED),
                    downloadRequest.logEntry,
                    url
                )
            }

            val body = decodeGzipIfNeeded(response)

            source = body?.source()

            Logger.d(TAG, "Start download from url: $url")

            sink = file.sink().buffer()

            var read: Long
            val contentLength = body?.contentLength() ?: 0L
            progress.status = AssetDownloadListener.Progress.ProgressStatus.STARTED
            progress.sizeBytes = contentLength
            progress.readBytes = totalRead
            progress.progressPercent = 0
            downloadAsset.contentLength = contentLength
            onProgressChanged(downloadRequest, progress, downloadListener)

            var current = 0
            while ((source?.read(sink.buffer, DOWNLOAD_CHUNK_SIZE.toLong())
                    ?: -1L).apply { read = this } > 0
            ) {
                if (!file.exists()) {
                    AssetWriteError("Asset save error $url")
                        .setLogEntry(downloadRequest.logEntry).logErrorNoReturnValue()
                    throw Downloader.RequestException("File is not existing")
                }

                if (downloadRequest.isCancelled()) {
                    progress.status = AssetDownloadListener.Progress.ProgressStatus.CANCELLED
                    break
                }

                progress.status = AssetDownloadListener.Progress.ProgressStatus.IN_PROGRESS

                sink.emit()
                sink.flush()

                totalRead += read
                progress.readBytes = totalRead

                val requiredBytes = downloadAsset.rangeEnd ?: downloadAsset.rangeStart
                if (downloadAsset.isWaitingForDownload() && totalRead >= requiredBytes) {
                    Logger.e(TAG, "Downloader totalRead=$totalRead requiredBytes=$requiredBytes")
                    downloadAsset.notifyDownloadEnough()
                }

                if (contentLength > 0) {
                    current = ((totalRead * MAX_PERCENT) / contentLength).toInt()
                }
                while (progress.progressPercent + PROGRESS_STEP <= min(current, MAX_PERCENT)) {
                    progress.status = AssetDownloadListener.Progress.ProgressStatus.IN_PROGRESS
                    progress.progressPercent += PROGRESS_STEP
                    if (progress.progressPercent >= MAX_PERCENT) {
                        progress.status = AssetDownloadListener.Progress.ProgressStatus.DONE
                    }
                    onProgressChanged(downloadRequest, progress, downloadListener)
                }

            }

            sink.flush()

            // GZIP
            if (progress.status == AssetDownloadListener.Progress.ProgressStatus.IN_PROGRESS) {
                progress.status = AssetDownloadListener.Progress.ProgressStatus.DONE
                onProgressChanged(downloadRequest, progress, downloadListener)
            }

        } catch (ex: Exception) {
            Logger.e("AssetDownloader", "$ex")
            // https://vungle.atlassian.net/browse/AND-4521
            // If asset returns response code 100, ProtocolException will happen when calling
            // call.execute(). That's why to add below check.
            if (ex is ProtocolException) {
                AssetRequestError("Failed to load asset: ${downloadAsset.serverPath}")
                    .setLogEntry(downloadRequest.logEntry).logErrorNoReturnValue()
            }

            progress.status = AssetDownloadListener.Progress.ProgressStatus.ERROR
            downloadError =
                AssetDownloadListener.DownloadError(code, ex, REQUEST_ERROR)
        } finally {
            if (response?.body != null) {
                response.body?.close()
            }

            call?.cancel()

            FileUtility.closeQuietly(sink)
            FileUtility.closeQuietly(source)

            Logger.d(TAG, "download status: ${progress.status}")
            when (progress.status) {
                AssetDownloadListener.Progress.ProgressStatus.ERROR,
                AssetDownloadListener.Progress.ProgressStatus.STARTED -> {
                    deliverError(downloadRequest, downloadListener, downloadError)
                }

                AssetDownloadListener.Progress.ProgressStatus.CANCELLED -> {
                    Logger.d(TAG, "On cancel $downloadRequest")
                    onProgressChanged(downloadRequest, progress, downloadListener)
                }

                else -> {
                    deliverSuccess(file, downloadRequest, downloadListener)
                }
            }
        }
    }

    private fun decodeGzipIfNeeded(networkResponse: Response): ResponseBody? {
        val resp = networkResponse.body
        if (GZIP.equals(
                networkResponse.header(CONTENT_ENCODING),
                ignoreCase = true
            ) && resp != null
        ) {
            val responseBody = GzipSource(resp.source())
            val contentType = networkResponse.header(CONTENT_TYPE)
            return RealResponseBody(contentType, -1L, responseBody.buffer())
        }
        return resp
    }

    private fun deliverSuccess(
        file: File,
        downloadRequest: DownloadRequest, listener: AssetDownloadListener?
    ) {
        Logger.d(TAG, "On success $downloadRequest")
        listener?.onSuccess(file, downloadRequest)
    }

    private fun onProgressChanged(
        downloadRequest: DownloadRequest,
        progress: AssetDownloadListener.Progress,
        downloadListener: AssetDownloadListener?
    ) {
        downloadListener?.onProgress(progress, downloadRequest)
    }

    private fun checkSpaceAvailable(downloadRequest: DownloadRequest): Boolean {
        val availableBytes =
            pathProvider.getAvailableBytes(pathProvider.getVungleDir().absolutePath)
        if (availableBytes < MINIMUM_SPACE_REQUIRED_MB) {
            NoSpaceError("Insufficient space $availableBytes")
                .setLogEntry(downloadRequest.logEntry).logErrorNoReturnValue()
            return false
        }
        return true
    }
}
