package com.vungle.ads

import com.vungle.ads.internal.protos.Sdk.SDKError
import com.vungle.ads.internal.util.LogEntry

/** Class that describes errors that can be thrown by Vungle SDK.
 * @param loggableReason [SDKError.Reason] that can/will be used to log this error.
 * @param errorMessage String message that describes in detail what happened
 * */
sealed class VungleError(
    private val loggableReason: SDKError.Reason,
    val errorMessage: String,
) : Exception(errorMessage) {

    val code = loggableReason.number

    private var logEntry: LogEntry? = null

    override fun getLocalizedMessage(): String? {
        return errorMessage
    }

    internal fun logError(): VungleError {
        logErrorNoReturnValue()
        return this
    }

    internal fun logErrorNoReturnValue() {
        AnalyticsClient.logError(loggableReason, errorMessage, logEntry)
    }

    internal fun setLogEntry(logEntry: LogEntry?): VungleError {
        this.logEntry = logEntry
        return this
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (javaClass != other?.javaClass) return false

        other as VungleError

        if (loggableReason != other.loggableReason) return false
        if (errorMessage != other.errorMessage) return false
        if (logEntry != other.logEntry) return false

        return true
    }

    override fun hashCode(): Int {
        var result = loggableReason.hashCode()
        result = 31 * result + errorMessage.hashCode()
        result = 31 * result + (logEntry?.hashCode() ?: 0)
        return result
    }

}

class InvalidAppId(msg: String): VungleError(
    loggableReason = SDKError.Reason.INVALID_APP_ID,
    errorMessage = msg
)

class InvalidCTAUrl(msg: String): VungleError(
    loggableReason = SDKError.Reason.INVALID_CTA_URL,
    errorMessage = msg
)

class SdkVersionTooLow(msg: String) : VungleError(
    loggableReason = SDKError.Reason.API_REQUEST_ERROR,
    errorMessage = msg,
)

class OutOfMemory(errorMessage: String) : VungleError(
    loggableReason = SDKError.Reason.OUT_OF_MEMORY,
    errorMessage = errorMessage
)

class SdkNotInitialized(message: String) : VungleError(
    loggableReason = SDKError.Reason.SDK_NOT_INITIALIZED,
    errorMessage = message,
)

class AdRetryActiveError : VungleError(
    loggableReason = SDKError.Reason.AD_LOAD_FAIL_RETRY_AFTER,
    errorMessage = "Ads: Server Retry Error active",
)

class AdRetryError : VungleError(
    loggableReason = SDKError.Reason.AD_RESPONSE_RETRY_AFTER,
    errorMessage = "Ads retry-after: Server is busy",
)

class AssetFailedToDeleteError(message: String) : VungleError(
    loggableReason = SDKError.Reason.ASSET_FAILED_TO_DELETE,
    errorMessage = message,
)

class NetworkUnreachable(message: String) : VungleError(
    loggableReason = SDKError.Reason.API_REQUEST_ERROR,
    errorMessage = message,
)

class NetworkTimeoutError : VungleError(
    loggableReason = SDKError.Reason.AD_RESPONSE_TIMED_OUT,
    errorMessage = "Ads: Network Timeout",
)

class PlacementNotFoundError(placementId: String) : VungleError(
    loggableReason = SDKError.Reason.INVALID_PLACEMENT_ID,
    errorMessage = "Placement \'$placementId\' is invalid",
)

class AdExpiredError(message: String) : VungleError(
    loggableReason = SDKError.Reason.AD_EXPIRED,
    errorMessage = "Ad expired: $message"
)

class AdExpiredOnPlayError(message: String? = null) : VungleError(
    loggableReason = SDKError.Reason.AD_EXPIRED_ON_PLAY,
    errorMessage = "Ad expired upon playback request: $message",
)

class AdNotLoadedCantPlay(message: String? = null) : VungleError(
    loggableReason = SDKError.Reason.AD_NOT_LOADED,
    errorMessage = "Failed to retrieve the ad object: $message",
)

class AdCantPlayWithoutWebView(message: String? = null) : VungleError(
    loggableReason = SDKError.Reason.WEBVIEW_ERROR,
    errorMessage = "No WebView when playing ads. $message",
)

class WebViewRenderingProcessGone(message: String? = null) : VungleError(
    loggableReason = SDKError.Reason.WEBVIEW_ERROR,
    errorMessage = "WebRenderingProcessGone: $message",
)

class WebViewRenderProcessUnresponsive(message: String? = null) : VungleError(
    loggableReason = SDKError.Reason.WEBVIEW_ERROR,
    errorMessage = "WebRenderProcessUnresponsive: $message",
)

class WebViewError(message: String? = null) : VungleError(
    loggableReason = SDKError.Reason.WEBVIEW_ERROR,
    errorMessage = "WebViewError: $message",
)

class InvalidAdStateError(
    loggableReason: SDKError.Reason,
    errorMessage: String = "Ad state is invalid",
) : VungleError(
    loggableReason = loggableReason,
    errorMessage = errorMessage
)

class APIFailedStatusCodeError(message: String? = null) : VungleError(
    loggableReason = SDKError.Reason.API_FAILED_STATUS_CODE,
    errorMessage = "Failed to get a successful response: $message",
)

class PlacementMismatchError(message: String) : VungleError(
    loggableReason = SDKError.Reason.AD_LOAD_FAIL_PLACEMENT_ID_MISMATCH,
    errorMessage = message,
)

class AdResponseEmptyError(message: String) : VungleError(
    loggableReason = SDKError.Reason.AD_RESPONSE_EMPTY,
    errorMessage = message,
)

class ConcurrentPlaybackUnsupported(message: String) : VungleError(
    loggableReason = SDKError.Reason.CONCURRENT_PLAYBACK_UNSUPPORTED,
    errorMessage = message,
)

class OmSdkJsError(reason: SDKError.Reason, errorMessage: String) : VungleError(
    loggableReason = reason,
    errorMessage = errorMessage
)

class AdMarkupInvalidError(message: String) : VungleError(
    loggableReason = SDKError.Reason.INVALID_ADUNIT_BID_PAYLOAD,
    errorMessage = message
)

class AdMarkupJsonError(message: String?) : VungleError(
    loggableReason = SDKError.Reason.INVALID_JSON_BID_PAYLOAD,
    errorMessage = "Unable to decode payload into BidPayload object. Error: $message"
)

class EmptyBidPayloadError(placementId: String?) : VungleError(
    loggableReason = SDKError.Reason.AD_LOAD_FAIL_EMPTY_BID_PAYLOAD,
    errorMessage = "$placementId header bidding status does not match with loadAd parameters",
)

class PrivacyUrlError(privacyUrl: String) : VungleError(
    loggableReason = SDKError.Reason.PRIVACY_URL_ERROR,
    errorMessage = "Failed to open privacy url: $privacyUrl"
)

class PlacementAdTypeMismatchError(placementId: String?) : VungleError(
    loggableReason = SDKError.Reason.PLACEMENT_AD_TYPE_MISMATCH,
    errorMessage = "$placementId Ad type does not match with placement type",
)

class MraidTemplateError(loggableReason: SDKError.Reason, message: String) : VungleError(
    loggableReason = loggableReason,
    errorMessage = message
)

class HeartbeatMissingError : VungleError(
    loggableReason = SDKError.Reason.AD_CLOSED_MISSING_HEARTBEAT,
    errorMessage = "Ad closed without receiving heartbeat"
)

class InvalidBannerSizeError(msg: String? = null) : VungleError(
    loggableReason = SDKError.Reason.BANNER_VIEW_INVALID_SIZE,
    errorMessage = "Invalidate size for banner ad: $msg"
)

class NoSpaceError(msg: String = "No space left on device") : VungleError(
    loggableReason = SDKError.Reason.ASSET_FAILED_INSUFFICIENT_SPACE,
    errorMessage = msg,
)

class AdPayloadError(reason: SDKError.Reason, msg: String) : VungleError(
    loggableReason = reason,
    errorMessage = msg,
)

class NativeAssetError(msg: String) : VungleError(
    loggableReason = SDKError.Reason.NATIVE_ASSET_ERROR,
    errorMessage = msg,
)

class InvalidTemplateURLError(msg: String) : VungleError(
    loggableReason = SDKError.Reason.INVALID_TEMPLATE_URL,
    errorMessage = msg,
)

class AssetRequestError(msg: String) : VungleError(
    loggableReason = SDKError.Reason.ASSET_REQUEST_ERROR,
    errorMessage = msg,
)

class InvalidEventIdError(msg: String) : VungleError(
    loggableReason = SDKError.Reason.INVALID_EVENT_ID_ERROR,
    errorMessage = msg,
)

class InvalidAssetUrlError(msg: String) : VungleError(
    loggableReason = SDKError.Reason.INVALID_ASSET_URL,
    errorMessage = msg,
)

class TpatError(reason: SDKError.Reason, msg: String) : VungleError(
    loggableReason = reason,
    errorMessage = msg,
)

class JsonEncodeError(msg: String) : VungleError(
    loggableReason = SDKError.Reason.JSON_ENCODE_ERROR,
    errorMessage = msg,
)

class GzipEncodeError(msg: String) : VungleError(
    loggableReason = SDKError.Reason.GZIP_ENCODE_ERROR,
    errorMessage = msg,
)

class AssetWriteError(msg: String) : VungleError(
    loggableReason = SDKError.Reason.ASSET_WRITE_ERROR,
    errorMessage = msg,
)

class IndexHtmlError(reason: SDKError.Reason, msg: String) : VungleError(
    loggableReason = reason,
    errorMessage = msg
)

class InvalidBidPayloadError : VungleError(
    loggableReason = SDKError.Reason.INVALID_BID_PAYLOAD,
    errorMessage = "Unable to create data object from payload string.",
)

class UserAgentError : VungleError(
    loggableReason = SDKError.Reason.USER_AGENT_ERROR,
    errorMessage = "Fail to get user agent.",
)

class EvaluateJsError(msg: String) : VungleError(
    loggableReason = SDKError.Reason.EVALUATE_JAVASCRIPT_FAILED,
    errorMessage = msg
)

class LinkError(reason: SDKError.Reason, msg: String) : VungleError(
    loggableReason = reason,
    errorMessage = msg
)

class InvalidEndpointError(reason: SDKError.Reason, msg: String) : VungleError(
    loggableReason = reason,
    errorMessage = msg
)

class PrivacyIconFallbackError(errorMessage: String) : VungleError(
    loggableReason = SDKError.Reason.PRIVACY_ICON_FALLBACK_ERROR,
    errorMessage = errorMessage
)
