package com.vungle.ads.internal.model

import androidx.annotation.VisibleForTesting
import com.vungle.ads.AdConfig
import com.vungle.ads.TpatError
import com.vungle.ads.internal.Constants
import com.vungle.ads.internal.Constants.CHECKPOINT_0
import com.vungle.ads.internal.Constants.DEEPLINK_CLICK
import com.vungle.ads.internal.Constants.DEEPLINK_SUCCESS_KEY
import com.vungle.ads.internal.Constants.REMOTE_PLAY_KEY
import com.vungle.ads.internal.Constants.AD_CLOSE
import com.vungle.ads.internal.Constants.AD_LOAD_DURATION
import com.vungle.ads.internal.Constants.AD_DURATION_KEY
import com.vungle.ads.internal.Constants.NETWORK_OPERATOR_KEY
import com.vungle.ads.internal.Constants.AD_LOAD_DURATION_KEY
import com.vungle.ads.internal.Constants.DEVICE_VOLUME_KEY
import com.vungle.ads.internal.Constants.VIDEO_LENGTH_TPAT
import com.vungle.ads.internal.Constants.VIDEO_LENGTH_KEY
import com.vungle.ads.internal.protos.Sdk.SDKError
import com.vungle.ads.internal.util.FileUtility
import com.vungle.ads.internal.util.FileUtility.isValidUrl
import com.vungle.ads.internal.util.LogEntry
import kotlinx.serialization.Contextual
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.Transient
import kotlinx.serialization.builtins.ListSerializer
import kotlinx.serialization.builtins.MapSerializer
import kotlinx.serialization.builtins.serializer
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonTransformingSerializer
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.put
import java.io.File
import java.util.concurrent.ConcurrentHashMap
import java.util.regex.Pattern

@Serializable
class AdPayload(
    @SerialName("ads") private val ads: List<PlacementAdUnit>? = null,
    @SerialName("config") private val config: ConfigPayload? = null
) {

    companion object {
        const val FILE_SCHEME = "file://"
        const val KEY_TEMPLATE = "template"
        const val KEY_VM = "vmURL"
        const val INCENTIVIZED_TITLE_TEXT = "INCENTIVIZED_TITLE_TEXT"
        const val INCENTIVIZED_BODY_TEXT = "INCENTIVIZED_BODY_TEXT"
        const val INCENTIVIZED_CLOSE_TEXT = "INCENTIVIZED_CLOSE_TEXT"
        const val INCENTIVIZED_CONTINUE_TEXT = "INCENTIVIZED_CONTINUE_TEXT"
        private const val UNKNOWN = "unknown"

        const val TPAT_CLICK_COORDINATES_URLS = "video.clickCoordinates"
    }

    private val ad: PlacementAdUnit?
        get() = ads?.let {
            if (it.isNotEmpty()) {
                it[0]
            } else {
                null
            }
        }

    private val adMarkup: AdUnit? get() = ad?.adMarkup

    fun placementId() = ad?.placementReferenceId

    fun eventId() = adMarkup?.id

    fun advAppId() = adMarkup?.advAppId

    fun adWidth() = adMarkup?.adSizeInfo?.width ?: 0

    fun adHeight() = adMarkup?.adSizeInfo?.height ?: 0

    fun adUnit() = adMarkup

    fun config() = config

    fun adLoadOptimizationEnabled() = adMarkup?.adLoadOptimizationEnabled ?: true

    fun isCacheableAssetsRequired() = adMarkup?.isCacheableAssetsRequired ?: true

    @Contextual
    private var mraidFiles = ConcurrentHashMap<String, String>()

    @VisibleForTesting
    var incentivizedTextSettings: MutableMap<String, String> = java.util.HashMap()

    var assetsFullyDownloaded: Boolean = false

    @Transient
    var adConfig: AdConfig? = null

    @Transient
    internal var logEntry: LogEntry? = null

    @Transient
    var assetDirectory: File? = null
        private set

    fun omEnabled(): Boolean {
        return adMarkup?.viewAbility?.om?.isEnabled ?: false
    }

    fun isClickCoordinatesTrackingEnabled(): Boolean {
        return adMarkup?.clickCoordinatesEnabled ?: false
    }

    fun isCriticalAsset(failingUrl: String): Boolean {
        if (!isNativeTemplateType() && adMarkup?.templateURL == failingUrl) {
            return true
        }

        return adMarkup?.templateSettings?.cacheableReplacements
            ?.filter { it.value.url == failingUrl }
            ?.isNotEmpty() ?: false
    }

    fun heartbeatEnabled():Boolean {
        return adMarkup?.templateHeartbeatCheck ?: false
    }

    fun getDownloadableAssets(dir: File): List<AdAsset> {
        assetDirectory = dir
        val ret = mutableListOf<AdAsset>()
        if (!isNativeTemplateType()) {
            adMarkup?.vmURL?.let { url ->
                if (isValidUrl(url)) {
                    val filePath = File(dir, Constants.AD_INDEX_FILE_NAME).absolutePath
                    ret.add(AdAsset(KEY_VM, url, filePath, AdAsset.FileType.ASSET, true))
                }
            } ?:
            adMarkup?.templateURL?.let { url ->
                if (isValidUrl(url)) {
                    // template is always required
                    val filePath = File(dir, KEY_TEMPLATE).absolutePath
                    ret.add(AdAsset(KEY_TEMPLATE, url, filePath, AdAsset.FileType.ZIP, true))
                }
            }
        }

        adMarkup?.templateSettings?.cacheableReplacements?.forEach {
            it.value.apply {
                if (url != null && isValidUrl(url)) {
                    var isAssetRequired = required ?: false // false by default
                    if (isNativeTemplateType()) {
                        // Native ad assets are always required
                        isAssetRequired = true
                    } else if (adLoadOptimizationEnabled()) {
                        if (!isCacheableAssetsRequired()) {
                            // if ad load optimization is enabled and cacheable assets are not required,
                            // all assets are not required even if they are marked as required
                            isAssetRequired = false
                        }
                    } else {
                        // if ad load optimization is disabled, all assets are required
                        isAssetRequired = true
                    }
                    val fileName = FileUtility.guessFileName(url, extension)
                    val filePath = File(dir, fileName).absolutePath
                    ret.add(AdAsset(it.key, url, filePath, AdAsset.FileType.ASSET, isAssetRequired))
                }
            }
        }

        ret.sortByDescending { it.isRequired }

        return ret
    }

    fun getTpatUrls(event: String, value: String? = null, secondValue: String? = null): List<String>? {
        if (adMarkup?.tpat?.containsKey(event) == false) {
            TpatError(SDKError.Reason.INVALID_TPAT_KEY,
                "Arbitrary tpat key: $event")
                .setLogEntry(logEntry).logErrorNoReturnValue()
            return null
        }
        val urls = adMarkup?.tpat?.get(event)
        if (urls.isNullOrEmpty()) {
            TpatError(SDKError.Reason.EMPTY_TPAT_ERROR,
                "Empty tpat key: $event")
                .setLogEntry(logEntry).logErrorNoReturnValue()
            return null
        }
        return when (event) {
            CHECKPOINT_0 -> {
                urls.map {
                    it.complexReplace(REMOTE_PLAY_KEY, "${!assetsFullyDownloaded}")
                        .complexReplace(NETWORK_OPERATOR_KEY, value)
                        .complexReplace(DEVICE_VOLUME_KEY, secondValue)
                }
            }
            VIDEO_LENGTH_TPAT -> urls.map {
                it.complexReplace(VIDEO_LENGTH_KEY, value)
            }
            DEEPLINK_CLICK -> {
                urls.map {
                    it.complexReplace(DEEPLINK_SUCCESS_KEY, value)
                }
            }

            AD_CLOSE -> urls.map {
                it.complexReplace(AD_DURATION_KEY, value)
                    .complexReplace(DEVICE_VOLUME_KEY, secondValue)
            }

            AD_LOAD_DURATION -> urls.map {
                it.complexReplace(AD_LOAD_DURATION_KEY, value)
            }

            else -> urls
        }
    }

    fun hasExpired(): Boolean {
        return adMarkup?.expiry?.run {
            this < System.currentTimeMillis() / 1000L
        } == true
    }

    fun getWinNotifications(): List<String>? {
        return adMarkup?.notification
    }

    fun isNativeTemplateType(): Boolean {
        return templateType() == Constants.TEMPLATE_TYPE_NATIVE
    }

    fun templateType(): String? {
        return adMarkup?.templateType
    }

    fun setIncentivizedText(
        title: String,
        body: String,
        keepWatching: String,
        close: String
    ) {
        if (title.isNotEmpty()) {
            incentivizedTextSettings[INCENTIVIZED_TITLE_TEXT] = title
        }
        if (body.isNotEmpty()) {
            incentivizedTextSettings[INCENTIVIZED_BODY_TEXT] = body
        }
        if (keepWatching.isNotEmpty()) {
            incentivizedTextSettings[INCENTIVIZED_CONTINUE_TEXT] = keepWatching
        }
        if (close.isNotEmpty()) {
            incentivizedTextSettings[INCENTIVIZED_CLOSE_TEXT] = close
        }
    }

    fun setAssetFullyDownloaded() {
        assetsFullyDownloaded = true
    }

    fun getMRAIDArgsInMap(): MutableMap<String, String> {
        requireNotNull(adMarkup?.templateSettings) { "Advertisement does not have MRAID Arguments!" }

        val resultMap = mutableMapOf<String, String>()
        adMarkup?.templateSettings?.normalReplacements?.let {
            resultMap.putAll(it)
        }
        adMarkup?.templateSettings?.cacheableReplacements?.forEach {
            it.value.url?.let { url ->
                resultMap[it.key] = url
            }
        }

        if (mraidFiles.isNotEmpty()) {
            resultMap.putAll(mraidFiles)
        }

        if (incentivizedTextSettings.isNotEmpty()) { // replacing incentivized dialog text
            resultMap.putAll(incentivizedTextSettings)
        }

        return resultMap
    }

    fun createMRAIDArgs(): JsonObject {
        val resultMap = getMRAIDArgsInMap()

        val ret = buildJsonObject {
            resultMap.forEach {
                put(it.key, it.value)
            }
        }

        return ret
    }

    fun getShowCloseDelay(incentivized: Boolean?): Int {
        if (incentivized == true) {
            return adMarkup?.showCloseIncentivized?.let {
                it * 1000
            } ?: 0
        }
        return adMarkup?.showClose?.let {
            it * 1000
        } ?: 0
    }

    fun getCreativeId(): String {
        return adMarkup?.creativeId ?: UNKNOWN
    }

    fun getAdSource(): String? {
        return adMarkup?.adSource
    }

    fun getMediationName(): String? {
        return adMarkup?.mediationName
    }

    fun getViewMasterVersion(): String? {
        return adMarkup?.vmVersion
    }

    fun getWebViewSettings(): WebViewSettings? {
        return adMarkup?.webViewSettings
    }

    private fun valueOrEmpty(value: String?) = value ?: ""

    /** there were some issues with simple `replace`, so we use this instead
     https://github.com/Vungle/vng-android-sdk/pull/284/files/2ab18c23056c0b5669bf81273e59ab4555955250#r1494165075 */
    private fun String.complexReplace(oldValue: String, newValue: String?) =
        replace(Pattern.quote(oldValue).toRegex(), valueOrEmpty(newValue))

    @Synchronized
    fun updateAdAssetPath(adAsset: AdAsset?) {
        if (adAsset != null && KEY_TEMPLATE != adAsset.adIdentifier) {
            val file = File(adAsset.localPath)
            if (file.exists()) {
                adAsset.adIdentifier.let { mraidFiles.put(it, FILE_SCHEME + file.path) }
            }
        }
    }

    @Serializable
    data class PlacementAdUnit(
        @SerialName("placement_reference_id") val placementReferenceId: String? = null,
        @SerialName("ad_markup") val adMarkup: AdUnit? = null,
    )

    @Serializable
    data class AdUnit(
        @SerialName("id") val id: String? = null,
        @SerialName("ad_type") val adType: String? = null,
        @SerialName("ad_source") val adSource: String? = null,
        @SerialName("expiry") val expiry: Int? = null,
        @SerialName("deeplink_url") val deeplinkUrl: String? = null,
        @SerialName("click_coordinates_enabled") val clickCoordinatesEnabled: Boolean? = null,
        @SerialName("ad_load_optimization") val adLoadOptimizationEnabled: Boolean? = null,
        @SerialName("template_heartbeat_check") val templateHeartbeatCheck: Boolean? = null,
        @SerialName("mediation_name") val mediationName: String? = null,
        @SerialName("info") val info: String? = null,
        @SerialName("sleep") val sleep: Int? = null,
        @SerialName("error_code") val errorCode: Int? = null,
        @Serializable(with = TpatSerializer::class) val tpat: Map<String, List<String>>? = null,
        @SerialName("vm_url") val vmURL: String? = null,
        @SerialName("vm_version") val vmVersion: String? = null,
        @SerialName("ad_market_id") val adMarketId: String? = null,
        @SerialName("notification") val notification: List<String>? = null,
        @SerialName("load_ad") val loadAdUrls: List<String>? = null,
        @SerialName("viewability") val viewAbility: ViewAbility? = null,
        @SerialName("template_url") val templateURL: String? = null,
        @SerialName("template_type") val templateType: String? = null,
        @SerialName("template_settings") val templateSettings: TemplateSettings? = null,
        @SerialName("creative_id") val creativeId: String? = null,
        @SerialName("app_id") val advAppId: String? = null,
        @SerialName("show_close") val showClose: Int? = 0,
        @SerialName("show_close_incentivized") val showCloseIncentivized: Int? = 0,
        @SerialName("ad_size") val adSizeInfo: AdSizeInfo? = null,
        @SerialName("cacheable_assets_required") val isCacheableAssetsRequired: Boolean? = null,
        @SerialName("webview_settings") val webViewSettings: WebViewSettings? = null,
    )

    object TpatSerializer : JsonTransformingSerializer<Map<String, List<String>>>(
        MapSerializer(
            String.serializer(),
            ListSerializer(String.serializer())
        )
    ) {
        // Filter out top-level key value pair with the key "moat"
        override fun transformDeserialize(element: JsonElement): JsonElement =
            JsonObject(element.jsonObject.filterNot { (k, _) ->
                k == "moat"
            })
    }

    @Serializable
    data class TemplateSettings(
        @SerialName("normal_replacements") val normalReplacements: Map<String, String>? = null,
        @SerialName("cacheable_replacements") val cacheableReplacements: Map<String, CacheableReplacement>? = null
    )

    @Serializable
    data class CacheableReplacement(val url: String? = null, val extension: String? = null, val required: Boolean? = null)

    @Serializable
    data class ViewAbility(val om: ViewAbilityInfo? = null)

    @Serializable
    data class ViewAbilityInfo(
        @SerialName("is_enabled") val isEnabled: Boolean? = null,
        @SerialName("extra_vast") val extraVast: String? = null
    )

    @Serializable
    data class AdSizeInfo(
        @SerialName("w") val width: Int? = 0,
        @SerialName("h") val height: Int? = 0
    )

    @Serializable
    data class WebViewSettings(
        @SerialName("allow_file_access_from_file_urls") val allowFileAccessFromFileUrls: Boolean? = null,
        @SerialName("allow_universal_access_from_file_urls") val allowUniversalAccessFromFileUrls: Boolean? = null
    )
}

