package com.vungle.ads

import android.content.Context
import android.graphics.drawable.BitmapDrawable
import android.text.TextUtils
import android.view.View
import android.widget.FrameLayout
import android.widget.ImageView
import androidx.annotation.IntDef
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.internal.AdInternal
import com.vungle.ads.internal.Constants.AD_VISIBILITY_INVISIBLE
import com.vungle.ads.internal.Constants.AD_VISIBILITY_VISIBLE
import com.vungle.ads.internal.Constants.AD_VISIBILITY_VISIBLE_LATER
import com.vungle.ads.internal.Constants.CHECKPOINT_0
import com.vungle.ads.internal.ImpressionTracker
import com.vungle.ads.internal.NativeAdInternal
import com.vungle.ads.internal.executor.Executors
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.platform.Platform
import com.vungle.ads.internal.presenter.AdEventListener
import com.vungle.ads.internal.presenter.AdPlayCallback
import com.vungle.ads.internal.presenter.NativeAdPresenter
import com.vungle.ads.internal.presenter.NativePresenterDelegate
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.ui.WatermarkView
import com.vungle.ads.internal.ui.view.MediaView
import com.vungle.ads.internal.util.ImageLoader
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.ThreadUtil
import java.util.concurrent.atomic.AtomicBoolean

class NativeAd private constructor(context: Context, placementId: String, adConfig: AdConfig) :
    BaseAd(context, placementId, adConfig) {

    constructor(context: Context, placementId: String) : this(context, placementId, AdConfig())

    companion object {
        private const val TAG = "NativeAd"
        const val TOP_LEFT = 0
        const val TOP_RIGHT = 1
        const val BOTTOM_LEFT = 2
        const val BOTTOM_RIGHT = 3
    }

    override fun constructAdInternal(context: Context) = NativeAdInternal(context)

    private val imageLoader by lazy { ImageLoader.instance.also { it.init(executors.ioExecutor) } }
    private val executors: Executors by inject(context)

    private var nativeAdAssetMap: MutableMap<String, String>? = null
    private var adIconView: ImageView? = null
    private var adContentView: MediaView? = null
    private val impressionTracker by lazy { ImpressionTracker(context) }
    private var aspectRatio: Float = 0.0f

    // Root view for rendering our privacy icon.
    private var adRootView: FrameLayout? = null
    private var clickableViews: Collection<View>? = null
    private var adOptionsView: NativeAdOptionsView? = null
    private var presenter: NativeAdPresenter? = null

    private val isInvisibleLogged = AtomicBoolean(false)

    @IntDef(TOP_LEFT, TOP_RIGHT, BOTTOM_LEFT, BOTTOM_RIGHT)
    annotation class AdOptionsPosition

    @AdOptionsPosition
    var adOptionsPosition = TOP_RIGHT

    init {
        adOptionsView = NativeAdOptionsView(context)
    }

    override fun onAdLoaded(advertisement: AdPayload) {
        super.onAdLoaded(advertisement)
        nativeAdAssetMap = advertisement.getMRAIDArgsInMap()
        // Call createMediaAspectRatio() to set the aspectRatio of the main media.
        createMediaAspectRatio()
    }

    private val adPlayCallback = object : AdPlayCallback {
        override fun onAdStart(id: String?) {
            adInternal.adState = AdInternal.AdState.PLAYING
            signalManager.increaseSessionDepthCounter()
            adInternal.validationToPresentMetric.markEnd()
            AnalyticsClient.logMetric(adInternal.validationToPresentMetric, logEntry)
            presentToDisplayMetric.markStart()
            ThreadUtil.runOnUiThread {
                adListener?.onAdStart(this@NativeAd)
            }
        }

        override fun onAdImpression(id: String?) {
            ThreadUtil.runOnUiThread {
                adListener?.onAdImpression(this@NativeAd)
            }
            presentToDisplayMetric.markEnd()
            AnalyticsClient.logMetric(presentToDisplayMetric, logEntry)
            displayToClickMetric.markStart()
        }

        override fun onAdEnd(id: String?) {
            adInternal.adState = AdInternal.AdState.FINISHED
            ThreadUtil.runOnUiThread {
                adListener?.onAdEnd(this@NativeAd)
            }
            showToCloseMetric.markEnd()
            AnalyticsClient.logMetric(showToCloseMetric, logEntry)
        }

        override fun onAdClick(id: String?) {
            ThreadUtil.runOnUiThread {
                adListener?.onAdClicked(this@NativeAd)
            }
            displayToClickMetric.markEnd()
            AnalyticsClient.logMetric(displayToClickMetric, logEntry)
        }

        override fun onAdRewarded(id: String?) {
            //no-op
        }

        override fun onAdLeftApplication(id: String?) {
            ThreadUtil.runOnUiThread {
                adListener?.onAdLeftApplication(this@NativeAd)
            }
            AnalyticsClient.logMetric(leaveApplicationMetric, logEntry)
        }

        override fun onFailure(error: VungleError) {
            adInternal.adState = AdInternal.AdState.ERROR
            ThreadUtil.runOnUiThread {
                adListener?.onAdFailedToPlay(this@NativeAd, error)
            }
            showToFailMetric.markEnd()
            AnalyticsClient.logMetric(showToFailMetric, logEntry, error.code.toString())
        }
    }

    /**
     * app icon url
     */
    fun getAppIcon() = nativeAdAssetMap?.get(NativeAdInternal.TOKEN_APP_ICON) ?: ""

    /**
     * app name
     */
    fun getAdTitle() = nativeAdAssetMap?.get(NativeAdInternal.TOKEN_APP_NAME) ?: ""

    /**
     * the ad description
     */
    fun getAdBodyText() = nativeAdAssetMap?.get(NativeAdInternal.TOKEN_APP_DESCRIPTION) ?: ""

    /**
     * call to action phrase
     */
    fun getAdCallToActionText() =
        nativeAdAssetMap?.get(NativeAdInternal.TOKEN_CTA_BUTTON_TEXT) ?: ""

    /**
     * app rating
     */
    fun getAdStarRating(): Double? {
        val ratingValue = nativeAdAssetMap?.get(NativeAdInternal.TOKEN_APP_RATING_VALUE) ?: ""
        return if (!TextUtils.isEmpty(ratingValue)) {
            try {
                java.lang.Double.valueOf(ratingValue)
            } catch (e: Exception) {
                null
            }
        } else null
    }

    /**
     * sponsored text
     */
    fun getAdSponsoredText() = nativeAdAssetMap?.get(NativeAdInternal.TOKEN_SPONSORED_BY) ?: ""

    /**
     * Whether call to action url exists.
     */
    fun hasCallToAction() = getCtaUrl().isNotEmpty()

    /**
     * Media's Aspect ratio
     */
    fun getMediaAspectRatio(): Float  = aspectRatio

    /**
     * Vungle privacy icon url
     */
    internal fun getPrivacyIconUrl() =
        nativeAdAssetMap?.get(NativeAdInternal.TOKEN_VUNGLE_PRIVACY_ICON_URL) ?: ""

    /**
     * Vungle privacy target url
     */
    internal fun getPrivacyUrl() =
        nativeAdAssetMap?.get(NativeAdInternal.TOKEN_VUNGLE_PRIVACY_URL) ?: ""

    /**
     * Call to action url
     */
    internal fun getCtaUrl() = nativeAdAssetMap?.get(NativeAdInternal.TOKEN_CTA_BUTTON_URL) ?: ""

    private fun getMainImagePath() = nativeAdAssetMap?.get(NativeAdInternal.TOKEN_MAIN_IMAGE) ?: ""

    private fun createMediaAspectRatio()  {
        imageLoader.getImageSize(getMainImagePath()) { width, height ->
            aspectRatio = width.toFloat() / height.toFloat()
        }
    }

    fun unregisterView() {
        if (adInternal.adState == AdInternal.AdState.FINISHED) {
            return
        }

        clickableViews?.forEach {
            it.setOnClickListener(null)
        }
        clickableViews = null

        nativeAdAssetMap?.clear()
        impressionTracker.destroy()
        adContentView?.destroy()
        adContentView = null
        adOptionsView?.destroy()
        adOptionsView = null

        try {
            val drawable = adIconView?.drawable
            if (drawable is BitmapDrawable) {
                val bitmap = drawable.bitmap
                if (!bitmap.isRecycled) {
                    bitmap.recycle()
                }
            }
        } catch (e: Exception) {
            Logger.w("NativeAd", "error msg: ${e.localizedMessage}")
        }
        adIconView?.setImageDrawable(null)
        adIconView = null

        presenter?.detach()
    }

    /**
     * Register the given view for this NativeAd to handle impressions and clicks.
     *
     * @param rootView       NativeAd root layout, should containing NativeAd for display
     * @param mediaView      NativeAd content view such as Image, Video
     * @param adIconView     App Icon view
     * @param clickableViews a list of all views that should handle taps event
     */
    fun registerViewForInteraction(
        rootView: FrameLayout,
        mediaView: MediaView,
        adIconView: ImageView?,
        clickableViews: Collection<View>?
    ) {
        AnalyticsClient.logMetric(
            SingleValueMetric(Sdk.SDKMetric.SDKMetricType.PLAY_AD_API), logEntry)
        responseToShowMetric.markEnd()
        AnalyticsClient.logMetric(responseToShowMetric, logEntry)
        adInternal.showToValidationMetric.markStart()
        showToFailMetric.markStart()
        showToCloseMetric.markStart()

        val error = adInternal.canPlayAd(true)
        if (error != null) {
            if (adInternal.isErrorTerminal(error.code)) {
                adInternal.adState = AdInternal.AdState.ERROR
                nativeAdAssetMap?.clear()
            }
            adListener?.onAdFailedToPlay(
                this@NativeAd,
                error
            )
            return
        }

        this.adRootView = rootView
        this.adContentView = mediaView
        this.adIconView = adIconView
        this.clickableViews = clickableViews

        val platform : Platform by inject(context)

        presenter =
            NativeAdPresenter(
                context,
                adInternal as NativePresenterDelegate,
                adInternal.advertisement,
                platform,
            )
        val omSdkData = nativeAdAssetMap?.get(NativeAdInternal.TOKEN_OM_SDK_DATA) ?: ""
        presenter?.initOMTracker(omSdkData)
        presenter?.startTracking(rootView)

        presenter?.setEventListener(AdEventListener(adPlayCallback, adInternal.placement))

        adOptionsView?.setOnClickListener {
            presenter?.processCommand(NativeAdPresenter.OPEN_PRIVACY, getPrivacyUrl())
        }
        (clickableViews ?: listOf(mediaView)).forEach {
            it.setOnClickListener {
                presenter?.processCommand(NativeAdPresenter.DOWNLOAD, getCtaUrl())
            }
        }

        //render privacy icon
        adOptionsView?.renderTo(rootView, adOptionsPosition)

        // track impression
        impressionTracker.addView(rootView, object : ImpressionTracker.ImpressionListener {

            override fun onImpression(view: View?) {
                Logger.d(TAG, "ImpressionTracker checked the native ad view become visible.")
                presenter?.processCommand(NativeAdPresenter.VIDEO_VIEWED)
                presenter?.processCommand(NativeAdPresenter.TPAT, CHECKPOINT_0)
                presenter?.onImpression()

                logViewVisibleOnPlay()
            }

            override fun onViewInvisible(view: View?) {
                if (!isInvisibleLogged.getAndSet(true)) {
                    Logger.d(
                        TAG,
                        "ImpressionTracker checked the native ad view invisible on play."
                    )
                    AnalyticsClient.logMetric(
                        SingleValueMetric(Sdk.SDKMetric.SDKMetricType.AD_VISIBILITY),
                        logEntry, AD_VISIBILITY_INVISIBLE
                    )
                }
            }
        })

        displayImage(getMainImagePath(), mediaView.getMainImage())
        displayImage(getAppIcon(), adIconView)
        displayImage(getPrivacyIconUrl(), adOptionsView?.getPrivacyIcon())

        adConfig.getWatermark()?.let {
            val imageView = WatermarkView(rootView.context, it)
            rootView.addView(imageView)
            imageView.bringToFront()
        }
        adInternal.showToValidationMetric.markEnd()
        AnalyticsClient.logMetric(adInternal.showToValidationMetric, logEntry)
        adInternal.validationToPresentMetric.markStart()

        presenter?.prepare()
    }

    private fun logViewVisibleOnPlay() {
        val value =
            if (isInvisibleLogged.get()) AD_VISIBILITY_VISIBLE_LATER else AD_VISIBILITY_VISIBLE
        AnalyticsClient.logMetric(
            SingleValueMetric(Sdk.SDKMetric.SDKMetricType.AD_VISIBILITY), logEntry, value
        )
        Logger.d(TAG, "Log metric AD_VISIBILITY: $value")
    }

    /**
     * Perform the CTA Action on Click event.
     */
    fun performCTA() {
        presenter?.processCommand(NativeAdPresenter.DOWNLOAD, getCtaUrl())
    }

    private fun displayImage(path: String?, imageView: ImageView?) {
        imageLoader.displayImage(path) {
            if (imageView != null) {
                ThreadUtil.runOnUiThread {
                    imageView.setImageBitmap(it)
                }
            }
        }
    }
}
